<?php

namespace App\Services;

use App\Http\Controllers\BroadcastController;
use App\Models\AutomationRule;
use App\Models\Complaint;
use App\Models\Dispatch;
use Illuminate\Support\Facades\Log;

class RulesEngineService
{
    /**
     * Execute rules for a trigger event
     */
    public function executeRules(string $triggerEvent, array $context = []): void
    {
        $rules = AutomationRule::where('trigger_event', $triggerEvent)
            ->where('is_active', true)
            ->orderBy('priority', 'desc')
            ->get();

        foreach ($rules as $rule) {
            if ($rule->checkConditions($context)) {
                $this->executeRule($rule, $context);
            }
        }
    }

    /**
     * Execute a single rule
     */
    private function executeRule(AutomationRule $rule, array $context): void
    {
        try {
            foreach ($rule->actions as $action) {
                $this->executeAction($action, $rule, $context);
            }

            $rule->incrementExecution();
        } catch (\Exception $e) {
            Log::error("Rule execution failed: " . $e->getMessage(), [
                'rule_id' => $rule->id,
                'rule_name' => $rule->name,
                'action' => $rule->actions,
            ]);
        }
    }

    /**
     * Execute a single action
     */
    private function executeAction(string $action, AutomationRule $rule, array $context): void
    {
        $actionParams = $rule->action_params ?? [];

        switch ($action) {
            case 'auto_dispatch':
                if (isset($context['complaint_id'])) {
                    $complaint = Complaint::find($context['complaint_id']);
                    if ($complaint && $complaint->status === 'pending') {
                        $dispatchService = app(DispatchService::class);
                        $dispatchService->autoDispatch($complaint);
                    }
                }
                break;

            case 'notify_supervisor':
                $this->notifySupervisor($context, $actionParams);
                break;

            case 'capture_cctv':
                if (isset($context['complaint_id'])) {
                    $complaint = Complaint::find($context['complaint_id']);
                    if ($complaint && $complaint->location) {
                        $cctvService = app(CctvService::class);
                        $cctvService->autoCaptureForComplaint($complaint);
                    }
                }
                break;

            case 'escalate':
                if (isset($context['dispatch_id'])) {
                    $dispatch = Dispatch::find($context['dispatch_id']);
                    if ($dispatch) {
                        $dispatchService = app(DispatchService::class);
                        $dispatchService->escalate($dispatch, $actionParams['reason'] ?? 'Auto-escalation');
                    }
                }
                break;

            case 'broadcast_alert':
                $this->broadcastAlert($context, $actionParams);
                break;

            case 'notify_all_units':
                $this->notifyAllUnits($context, $actionParams);
                break;

            default:
                Log::warning("Unknown action: {$action}");
        }
    }

    /**
     * Notify supervisor
     */
    private function notifySupervisor(array $context, array $params): void
    {
        $notificationService = app(NotificationService::class);
        $message = $params['message'] ?? 'Supervisor notification';

        $supervisors = \App\Models\User::where('role', 'admin')
            ->orWhere('role', 'supervisor')
            ->get();

        foreach ($supervisors as $supervisor) {
            $notificationService->sendToUser($supervisor, [
                'type' => 'supervisor_notification',
                'title' => $params['title'] ?? 'Supervisor Notification',
                'message' => $message,
            ]);
        }
    }

    /**
     * Broadcast alert
     */
    private function broadcastAlert(array $context, array $params): void
    {
        $broadcastController = app(BroadcastController::class);
        // Implementation depends on BroadcastController structure
        Log::info("Broadcast alert triggered", $context);
    }

    /**
     * Notify all units
     */
    private function notifyAllUnits(array $context, array $params): void
    {
        $notificationService = app(NotificationService::class);
        $officers = \App\Models\Officer::where('is_active', true)->get();

        foreach ($officers as $officer) {
            if ($officer->user) {
                $notificationService->sendToOfficer($officer, null, [
                    'message' => $params['message'] ?? 'Notification to all units',
                ]);
            }
        }
    }
}

