<?php

namespace App\Services;

use App\Models\Complaint;
use App\Models\Dispatch;
use Illuminate\Support\Facades\Log;

class WorkflowEngineService
{
    /**
     * Execute workflow for complaint
     */
    public function executeComplaintWorkflow(Complaint $complaint): void
    {
        $workflow = $this->getWorkflowForComplaint($complaint);

        foreach ($workflow['steps'] as $step) {
            if ($this->shouldExecuteStep($step, $complaint)) {
                $this->executeStep($step, $complaint);
            }
        }
    }

    /**
     * Get workflow configuration for complaint
     */
    private function getWorkflowForComplaint(Complaint $complaint): array
    {
        // Define workflows based on category and priority
        $workflows = [
            'emergency_critical' => [
                'steps' => [
                    ['action' => 'auto_track_location', 'required' => true],
                    ['action' => 'classify_complaint', 'required' => true],
                    ['action' => 'capture_cctv', 'required' => false],
                    ['action' => 'auto_dispatch', 'required' => true],
                    ['action' => 'notify_supervisor', 'required' => true],
                ],
            ],
            'emergency_high' => [
                'steps' => [
                    ['action' => 'auto_track_location', 'required' => true],
                    ['action' => 'classify_complaint', 'required' => true],
                    ['action' => 'capture_cctv', 'required' => false],
                    ['action' => 'auto_dispatch', 'required' => true],
                ],
            ],
            'default' => [
                'steps' => [
                    ['action' => 'auto_track_location', 'required' => false],
                    ['action' => 'classify_complaint', 'required' => true],
                    ['action' => 'queue_for_review', 'required' => true],
                ],
            ],
        ];

        $workflowKey = 'default';
        if ($complaint->priority === 'critical') {
            $workflowKey = 'emergency_critical';
        } elseif ($complaint->priority === 'high') {
            $workflowKey = 'emergency_high';
        }

        return $workflows[$workflowKey] ?? $workflows['default'];
    }

    /**
     * Check if step should be executed
     */
    private function shouldExecuteStep(array $step, Complaint $complaint): bool
    {
        if ($step['required'] ?? false) {
            return true;
        }

        // Check conditions
        if (isset($step['conditions'])) {
            foreach ($step['conditions'] as $key => $value) {
                if ($complaint->$key != $value) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Execute a workflow step
     */
    private function executeStep(array $step, Complaint $complaint): void
    {
        $action = $step['action'];

        try {
            switch ($action) {
                case 'auto_track_location':
                    // Already handled in ComplaintController
                    break;

                case 'classify_complaint':
                    // Already handled in ComplaintController
                    break;

                case 'capture_cctv':
                    if ($complaint->location) {
                        $cctvService = app(CctvService::class);
                        $cctvService->autoCaptureForComplaint($complaint);
                    }
                    break;

                case 'auto_dispatch':
                    $dispatchService = app(DispatchService::class);
                    $dispatchService->autoDispatch($complaint);
                    break;

                case 'notify_supervisor':
                    $notificationService = app(NotificationService::class);
                    $supervisors = \App\Models\User::where('role', 'admin')
                        ->orWhere('role', 'supervisor')
                        ->get();
                    foreach ($supervisors as $supervisor) {
                        $notificationService->sendToUser($supervisor, [
                            'type' => 'critical_complaint',
                            'title' => 'Critical Complaint',
                            'message' => "Critical complaint received: {$complaint->ticket_number}",
                            'complaint_id' => $complaint->id,
                        ]);
                    }
                    break;

                case 'queue_for_review':
                    $complaint->update(['status' => 'pending']);
                    break;

                default:
                    Log::warning("Unknown workflow action: {$action}");
            }
        } catch (\Exception $e) {
            Log::error("Workflow step execution failed: " . $e->getMessage(), [
                'action' => $action,
                'complaint_id' => $complaint->id,
            ]);
        }
    }
}

