# Solusi Optimasi Performa Halaman Wilayah Administratif

## Masalah yang Diidentifikasi
Halaman wilayah administratif menampilkan data kemudian menghilang karena:
1. Query tanpa pagination memuat semua data sekaligus
2. N+1 problem pada relationship loading
3. Tidak ada loading indicator
4. Recursive statistics calculation yang lambat

## Solusi Implementasi

### 1. Optimasi Controller dengan Pagination
```php
// app/Http/Controllers/WilayahAdministratifController.php
public function index(Request $request)
{
    // Add pagination and optimize query
    $query = WilayahAdministratif::with(['user', 'parent'])
        ->select(['id', 'jenis_wilayah', 'kode_wilayah', 'nama_wilayah', 
                 'parent_id', 'kepala_wilayah', 'jumlah_kk', 'jumlah_penduduk', 'user_id']);

    if ($request->has('tipe')) {
        $query->where('jenis_wilayah', $request->tipe);
    }

    if ($request->has('parent')) {
        $query->where('parent_id', $request->parent);
    }

    // Add pagination (15 items per page)
    $wilayah = $query->orderBy('jenis_wilayah')
                    ->orderBy('kode_wilayah')
                    ->paginate(15);

    // Optimize stats query
    $stats = [
        'total_dusun' => WilayahAdministratif::where('jenis_wilayah', 'dusun')->count(),
        'total_rw' => WilayahAdministratif::where('jenis_wilayah', 'rw')->count(),
        'total_rt' => WilayahAdministratif::where('jenis_wilayah', 'rt')->count(),
        'total_kk' => WilayahAdministratif::sum('jumlah_kk'),
    ];

    $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')
                                   ->select(['id', 'nama_wilayah'])
                                   ->get();
    $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')
                                 ->select(['id', 'nama_wilayah', 'parent_id'])
                                 ->with('parent:id,nama_wilayah')
                                 ->get();

    if ($request->ajax() || $request->wantsJson()) {
        return response()->json([
            'success' => true,
            'data' => $wilayah,
            'stats' => $stats
        ]);
    }

    return view('admin.wilayah-administratif.index', compact('wilayah', 'stats', 'dusunList', 'rwList'));
}
```

### 2. Optimasi Model dengan Caching
```php
// app/Models/WilayahAdministratif.php
use Illuminate\Support\Facades\Cache;

public static function getStatistikWilayah()
{
    return Cache::remember('wilayah_statistik', 3600, function () {
        return [
            'total_dusun' => self::dusun()->aktif()->count(),
            'total_rw' => self::rw()->aktif()->count(),
            'total_rt' => self::rt()->aktif()->count(),
            'total_kk' => self::aktif()->sum('jumlah_kk'),
            'total_penduduk' => self::aktif()->sum('jumlah_penduduk'),
        ];
    });
}

public function updateStatistikPenduduk()
{
    // Optimize dengan single query instead of loops
    if ($this->jenis_wilayah === 'dusun') {
        $stats = $this->children()
                     ->selectRaw('SUM(jumlah_kk) as total_kk, SUM(jumlah_penduduk) as total_penduduk')
                     ->first();
    } elseif ($this->jenis_wilayah === 'rw') {
        $stats = $this->children()
                     ->selectRaw('SUM(jumlah_kk) as total_kk, SUM(jumlah_penduduk) as total_penduduk')
                     ->first();
    } else {
        // RT: hitung langsung dari penduduk
        $jumlahKK = Keluarga::whereHas('anggota', function($query) {
            $query->where('rt', $this->nama_wilayah);
        })->count();
        
        $jumlahPenduduk = Penduduk::where('rt', $this->nama_wilayah)->count();
        
        $stats = (object) ['total_kk' => $jumlahKK, 'total_penduduk' => $jumlahPenduduk];
    }

    $this->update([
        'jumlah_kk' => $stats->total_kk ?? 0,
        'jumlah_penduduk' => $stats->total_penduduk ?? 0
    ]);
    
    // Clear cache after update
    Cache::forget('wilayah_statistik');
}
```

### 3. Frontend Loading Indicator
```php
// resources/views/admin/wilayah-administratif/index.blade.php
@push('styles')
<style>
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(255, 255, 255, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
}
.table-container {
    min-height: 400px;
    position: relative;
}
</style>
@endpush

@section('content')
<div class="container-fluid">
    <!-- Statistics Cards (same as before) -->
    
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <!-- Header content -->
                </div>
                <div class="card-body">
                    <div class="table-container">
                        <div id="loadingIndicator" class="text-center py-4" style="display: none;">
                            <div class="spinner-border text-primary" role="status">
                                <span class="visually-hidden">Loading...</span>
                            </div>
                            <p class="mt-2">Memuat data wilayah...</p>
                        </div>
                        
                        <div class="table-responsive" id="tableContainer">
                            <table class="table table-striped" id="wilayahTable">
                                <!-- Table content -->
                            </table>
                        </div>
                        
                        <!-- Pagination -->
                        <div class="d-flex justify-content-center mt-3">
                            {{ $wilayah->links() }}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    // Show loading indicator
    function showLoading() {
        document.getElementById('loadingIndicator').style.display = 'block';
        document.getElementById('tableContainer').style.opacity = '0.5';
    }
    
    // Hide loading indicator
    function hideLoading() {
        document.getElementById('loadingIndicator').style.display = 'none';
        document.getElementById('tableContainer').style.opacity = '1';
    }
    
    // Initialize DataTable with loading states
    $('#wilayahTable').DataTable({
        "paging": false,
        "searching": true,
        "ordering": true,
        "info": false,
        "responsive": true,
        "processing": true,
        "language": {
            "url": "//cdn.datatables.net/plug-ins/1.10.24/i18n/Indonesian.json",
            "processing": "Memuat data..."
        },
        "initComplete": function() {
            hideLoading();
        }
    });
    
    // Enhanced sync function with loading
    window.syncPopulation = function() {
        if (confirm('Apakah Anda yakin ingin menyinkronkan data penduduk dengan wilayah administratif?')) {
            showLoading();
            
            fetch('{{ route("admin.wilayah-administratif.sync") }}', {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Content-Type': 'application/json'
                }
            })
            .then(response => response.json())
            .then(data => {
                hideLoading();
                if (data.success) {
                    Swal.fire('Berhasil!', data.message, 'success').then(() => {
                        location.reload();
                    });
                } else {
                    Swal.fire('Error!', data.message, 'error');
                }
            })
            .catch(error => {
                hideLoading();
                Swal.fire('Error!', 'Terjadi kesalahan saat sinkronisasi', 'error');
            });
        }
    }
});
</script>
@endpush
```

### 4. Database Indexing
```sql
-- Add indexes untuk mempercepat query
ALTER TABLE wilayah_administratif ADD INDEX idx_jenis_wilayah (jenis_wilayah);
ALTER TABLE wilayah_administratif ADD INDEX idx_parent_id (parent_id);
ALTER TABLE wilayah_administratif ADD INDEX idx_status (status);
ALTER TABLE wilayah_administratif ADD INDEX idx_urutan_tampil (urutan_tampil);
```

### 5. Lazy Loading untuk Statistik
```php
// Buat endpoint terpisah untuk statistik
Route::get('/admin/wilayah-administratif/stats', function() {
    $stats = \App\Models\WilayahAdministratif::getStatistikWilayah();
    return response()->json($stats);
});
```

### 6. Batch Operations untuk Update Statistik
```php
// app/Console/Commands/UpdateWilayahStatistik.php
<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\WilayahAdministratif;

class UpdateWilayahStatistik extends Command
{
    protected $signature = 'wilayah:update-statistik';
    protected $description = 'Update statistik semua wilayah administratif';

    public function handle()
    {
        $this->info('Memulai update statistik wilayah...');
        
        // Update RT terlebih dahulu
        $rtList = WilayahAdministratif::where('jenis_wilayah', 'rt')->get();
        foreach ($rtList as $rt) {
            $rt->updateStatistikPenduduk();
        }
        
        // Kemudian RW
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->get();
        foreach ($rwList as $rw) {
            $rw->updateStatistikPenduduk();
        }
        
        // Terakhir Dusun
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->get();
        foreach ($dusunList as $dusun) {
            $dusun->updateStatistikPenduduk();
        }
        
        $this->info('Update statistik wilayah selesai!');
    }
}
```

## Implementasi Bertahap
1. **Langsung**: Tambahkan loading indicator dan pagination
2. **Hari 1**: Optimasi query dengan select specific columns
3. **Hari 2**: Implementasi caching untuk statistik
4. **Hari 3**: Tambahkan database indexes
5. **Hari 4**: Buat command untuk batch update statistik

## Expected Results
- Loading time berkurang dari beberapa detik menjadi < 1 detik
- Tidak ada lagi tampilan yang menghilang
- User experience yang lebih baik dengan loading indicator
- Reduced server load dengan caching dan pagination
