<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Models\Role;

class CreateSuperAdmin extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'make:superadmin {email?} {password?} {name?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create a superadmin user account';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Creating SuperAdmin Account...');
        $this->newLine();

        // Get input parameters or ask for them
        $email = $this->argument('email') ?: $this->ask('Email address', 'superadmin@aplikasidesa.com');
        $name = $this->argument('name') ?: $this->ask('Full name', 'Super Administrator');
        $password = $this->argument('password') ?: $this->secret('Password (leave empty for auto-generate)');

        // Generate password if not provided
        if (empty($password)) {
            $password = 'SuperAdmin' . rand(1000, 9999) . '!';
            $this->warn("🔑 Auto-generated password: $password");
        }

        // Check if user already exists
        $existingUser = User::where('email', $email)->first();
        if ($existingUser) {
            $this->error("❌ User with email '$email' already exists!");
            
            if ($this->confirm('Do you want to update existing user to superadmin?')) {
                $existingUser->update([
                    'role' => 'superadmin',
                    'is_active' => true
                ]);
                $this->info("✅ User '$email' updated to superadmin!");
                return 0;
            }
            
            return 1;
        }

        // Create or get superadmin role
        $superAdminRole = Role::firstOrCreate(
            ['name' => 'superadmin'],
            [
                'display_name' => 'Super Administrator',
                'description' => 'Full access to all system features',
                'permissions' => [
                    'users.*', 'roles.*', 'penduduk.*', 'keluarga.*', 
                    'surat.*', 'pengaduan.*', 'bantuan.*', 'berita.*',
                    'apbdes.*', 'kegiatan.*', 'organisasi.*', 'wilayah.*',
                    'settings.*', 'dashboard.*', 'laporan.*'
                ],
                'is_active' => true
            ]
        );

        // Create the superadmin user
        try {
            $user = User::create([
                'name' => $name,
                'email' => $email,
                'password' => Hash::make($password),
                'role' => 'superadmin',
                'role_id' => $superAdminRole->id,
                'is_active' => true,
                'email_verified_at' => now()
            ]);

            $this->newLine();
            $this->info('🎉 SuperAdmin account created successfully!');
            $this->newLine();
            
            $this->line('<fg=cyan>Account Details:</>');
            $this->line("👤 Name: <fg=green>$name</>");
            $this->line("📧 Email: <fg=green>$email</>");
            $this->line("🔑 Password: <fg=yellow>$password</>");
            $this->line("🏷️  Role: <fg=green>Super Administrator</>");
            $this->line("🚀 Status: <fg=green>Active</>");
            
            $this->newLine();
            $this->line('<fg=cyan>Access URL:</>');
            $this->line("🌐 Admin Panel: <fg=blue>http://127.0.0.1:8001/admin</>");
            
            $this->newLine();
            $this->warn('⚠️  IMPORTANT: Change the password after first login!');
            $this->newLine();

            return 0;

        } catch (\Exception $e) {
            $this->error('❌ Failed to create superadmin: ' . $e->getMessage());
            return 1;
        }
    }
} 