<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Penduduk;

class MapOrangTuaCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'penduduk:map-orangtua 
                           {--dry-run : Preview mapping tanpa menyimpan perubahan}
                           {--force : Jalankan tanpa konfirmasi}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Mapping nama ayah/ibu ke NIK untuk data penduduk';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔄 Memulai proses mapping nama orang tua ke NIK...');
        $this->newLine();

        // Cek data yang perlu dimapping
        $needMapping = Penduduk::where(function($query) {
            $query->where(function($q) {
                $q->whereNotNull('nama_ayah')
                  ->whereNull('ayah_nik');
            })->orWhere(function($q) {
                $q->whereNotNull('nama_ibu')
                  ->whereNull('ibu_nik');
            });
        })->count();

        if ($needMapping === 0) {
            $this->info('✅ Tidak ada data yang perlu dimapping.');
            return 0;
        }

        $this->info("📊 Ditemukan {$needMapping} record yang perlu dimapping.");
        $this->newLine();

        // Preview mode
        if ($this->option('dry-run')) {
            $this->info('🔍 Mode Preview (Dry Run) - Tidak ada perubahan yang disimpan');
            $this->previewMapping();
            return 0;
        }

        // Konfirmasi
        if (!$this->option('force')) {
            if (!$this->confirm('Lanjutkan proses mapping?')) {
                $this->info('❌ Proses dibatalkan.');
                return 0;
            }
        }

        // Jalankan mapping
        $this->info('🚀 Memulai proses mapping...');
        $results = Penduduk::batchMapOrangTua();

        // Tampilkan hasil
        $this->displayResults($results);

        return 0;
    }

    private function previewMapping()
    {
        $pendudukNeedMapping = Penduduk::where(function($query) {
            $query->where(function($q) {
                $q->whereNotNull('nama_ayah')
                  ->whereNull('ayah_nik');
            })->orWhere(function($q) {
                $q->whereNotNull('nama_ibu')
                  ->whereNull('ibu_nik');
            });
        })->take(10)->get();

        $headers = ['NIK', 'Nama', 'Nama Ayah', 'NIK Ayah Found', 'Nama Ibu', 'NIK Ibu Found'];
        $rows = [];

        foreach ($pendudukNeedMapping as $penduduk) {
            $nikAyah = $penduduk->nama_ayah ? Penduduk::findNikByNama($penduduk->nama_ayah, 'L') : null;
            $nikIbu = $penduduk->nama_ibu ? Penduduk::findNikByNama($penduduk->nama_ibu, 'P') : null;

            $rows[] = [
                $penduduk->nik,
                $penduduk->nama,
                $penduduk->nama_ayah ?: '-',
                $nikAyah ? "✅ {$nikAyah}" : '❌ Tidak ditemukan',
                $penduduk->nama_ibu ?: '-',
                $nikIbu ? "✅ {$nikIbu}" : '❌ Tidak ditemukan'
            ];
        }

        $this->table($headers, $rows);
        
        if (count($pendudukNeedMapping) === 10) {
            $this->info('📝 Menampilkan 10 record pertama. Gunakan --force untuk melihat semua.');
        }
    }

    private function displayResults($results)
    {
        $this->newLine();
        $this->info('📈 Hasil Mapping:');
        $this->line("📊 Total diproses: {$results['processed']} record");
        $this->line("👨 Ayah berhasil dimapping: {$results['ayah_mapped']} record");
        $this->line("👩 Ibu berhasil dimapping: {$results['ibu_mapped']} record");

        if (!empty($results['errors'])) {
            $this->newLine();
            $this->error('❌ Error yang terjadi:');
            foreach ($results['errors'] as $error) {
                $this->line("  - {$error['nama']} ({$error['nik']}): {$error['error']}");
            }
        }

        $this->newLine();
        if ($results['ayah_mapped'] > 0 || $results['ibu_mapped'] > 0) {
            $this->info('✅ Proses mapping selesai dengan sukses!');
        } else {
            $this->warn('⚠️ Tidak ada mapping yang berhasil dilakukan.');
        }
    }
}
