<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ObjekWisata;
use App\Models\TiketWisata;
use App\Services\FileUploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class ObjekWisataController extends Controller
{
    public function index(Request $request)
    {
        $query = ObjekWisata::with(['user', 'lastUpdatedBy']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nama_wisata', 'like', "%{$search}%")
                  ->orWhere('deskripsi', 'like', "%{$search}%")
                  ->orWhere('alamat', 'like', "%{$search}%");
            });
        }
        
        // Filter by category
        if ($request->filled('kategori')) {
            $query->where('kategori_wisata', $request->kategori);
        }
        
        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }
        
        $objekWisata = $query->paginate(20)->withQueryString();
        
        // Get filter options
        $kategoriList = $this->getKategoriList();
        
        // Statistics
        $stats = [
            'total' => ObjekWisata::count(),
            'aktif' => ObjekWisata::aktif()->count(),
            'featured' => ObjekWisata::featured()->count(),
            'recommended' => ObjekWisata::recommended()->count(),
        ];
        
        return view('admin.objek-wisata.index', compact('objekWisata', 'kategoriList', 'stats'));
    }
    
    public function create()
    {
        $kategoriList = $this->getKategoriList();
        
        return view('admin.objek-wisata.create', compact('kategoriList'));
    }
    
    public function store(Request $request)
    {
        // Validate form data
        $request->validate([
            'nama_wisata' => 'required|string|max:255',
            'kategori_wisata' => 'required|in:wisata_alam,wisata_budaya,wisata_sejarah,wisata_kuliner,wisata_edukasi,wisata_adventure,wisata_religi,lainnya',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'harga_tiket_dewasa' => 'nullable|numeric|min:0',
            'harga_tiket_anak' => 'nullable|numeric|min:0',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);
        
        // Validate files separately if present
        if ($request->hasFile('foto_utama')) {
            $request->validate([
                'foto_utama' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            ]);
        }
        
        if ($request->hasFile('galeri_foto')) {
            $request->validate([
                'galeri_foto.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            ]);
        }
        
        $data = $request->all();
        $data['user_id'] = Auth::id();
        $data['status'] = 'aktif';
        $data['slug'] = Str::slug($request->nama_wisata);
        
        
        // Handle longitude field - ensure it's not empty
        if (empty($data['longitude'])) {
            $data['longitude'] = 0;
        }
        
        // Handle latitude field - ensure it's not empty  
        if (empty($data['latitude'])) {
            $data['latitude'] = 0;
        }
        
        // Handle harga_tiket_anak field
        if (empty($data['harga_tiket_anak'])) {
            $data['harga_tiket_anak'] = 0;
        }
        
        // Handle foto utama upload
        if ($request->hasFile('foto_utama')) {
            $uploadService = new FileUploadService();
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto_utama'), 'wisata');
            
            if ($uploadResult['success']) {
                $data['foto_utama'] = $uploadResult['filename'];
                Log::info('Tourism main photo uploaded successfully', ['path' => $uploadResult['path']]);
            } else {
                Log::error('Tourism main photo upload failed', ['error' => $uploadResult['error']]);
                return redirect()->back()
                    ->withErrors(['foto_utama' => 'Gagal mengupload foto utama: ' . $uploadResult['error']])
                    ->withInput();
            }
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            $galeriFiles = [];
            $uploadService = new FileUploadService();
            
            foreach ($request->file('galeri_foto') as $index => $file) {
                $uploadResult = $uploadService->uploadPhotoWithResize($file, 'wisata/galeri');
                
                if ($uploadResult['success']) {
                    $galeriFiles[] = $uploadResult['filename'];
                    Log::info("Tourism gallery file {$index} upload successful", ['path' => $uploadResult['path']]);
                } else {
                    Log::error("Tourism gallery file {$index} upload failed", ['error' => $uploadResult['error']]);
                    return redirect()->back()
                        ->withErrors(['galeri_foto' => "Gagal mengupload foto galeri {$index}: " . $uploadResult['error']])
                        ->withInput();
                }
            }
            
            $data['galeri_foto'] = $galeriFiles;
        }
        
        ObjekWisata::create($data);
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil ditambahkan.');
    }
    
    public function show(ObjekWisata $objek_wisata)
    {
        // Load related data if available
        $objek_wisata->load(['user']);
        
        // If this is an AJAX request, return JSON data like penduduk controller
        if (request()->ajax() || request()->wantsJson()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $objek_wisata->id,
                    'nama_wisata' => $objek_wisata->nama_wisata,
                    'kategori_wisata' => $objek_wisata->kategori_wisata,
                    'deskripsi' => $objek_wisata->deskripsi,
                    'alamat' => $objek_wisata->alamat,
                    'desa' => $objek_wisata->desa,
                    'kecamatan' => $objek_wisata->kecamatan,
                    'latitude' => $objek_wisata->latitude,
                    'longitude' => $objek_wisata->longitude,
                    'harga_tiket_dewasa' => $objek_wisata->harga_tiket_dewasa,
                    'harga_tiket_anak' => $objek_wisata->harga_tiket_anak,
                    'jam_operasional' => $objek_wisata->jam_operasional,
                    'nama_pengelola' => $objek_wisata->nama_pengelola,
                    'telepon' => $objek_wisata->telepon,
                    'email' => $objek_wisata->email,
                    'website' => $objek_wisata->website,
                    'status' => $objek_wisata->status,
                    'is_featured' => $objek_wisata->is_featured,
                    'rating_rata_rata' => $objek_wisata->rating_rata_rata,
                    'jumlah_review' => $objek_wisata->jumlah_review,
                    'total_pengunjung' => $objek_wisata->total_pengunjung,
                    'foto' => $objek_wisata->foto_utama_url, // Use accessor for proper URL
                    'galeri_foto' => $objek_wisata->galeri_foto_urls, // Use accessor for proper URLs
                    'created_at' => $objek_wisata->created_at,
                    'updated_at' => $objek_wisata->updated_at
                ]
            ]);
        }
        
        // Ensure all data is accessible
        $objekWisata = $objek_wisata; // For view compatibility
        
        return view('admin.objek-wisata.show', compact('objekWisata'));
    }
    
    public function edit(ObjekWisata $objek_wisata)
    {
        $kategoriList = $this->getKategoriList();
        $objekWisata = $objek_wisata; // For view compatibility
        
        return view('admin.objek-wisata.edit', compact('objekWisata', 'kategoriList'));
    }
    
    public function update(Request $request, ObjekWisata $objek_wisata)
    {
        // Validate form data
        $request->validate([
            'nama_wisata' => 'required|string|max:255',
            'kategori_wisata' => 'required|in:wisata_alam,wisata_budaya,wisata_sejarah,wisata_kuliner,wisata_edukasi,wisata_adventure,wisata_religi,lainnya',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'harga_tiket_dewasa' => 'required|numeric|min:0',
            'harga_tiket_anak' => 'nullable|numeric|min:0',
            'latitude' => 'nullable|numeric',
            'longitude' => 'nullable|numeric',
        ]);
        
        // Validate files separately if present
        if ($request->hasFile('foto_utama')) {
            $request->validate([
                'foto_utama' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            ]);
        }
        
        if ($request->hasFile('galeri_foto')) {
            $request->validate([
                'galeri_foto.*' => 'image|mimes:jpeg,png,jpg,webp|max:2048',
            ]);
        }
        
        $data = $request->all();
        $data['last_updated_by'] = Auth::id();
        
        // Handle longitude field - ensure it's not empty
        if (empty($data['longitude'])) {
            $data['longitude'] = 0;
        }
        
        // Handle latitude field - ensure it's not empty  
        if (empty($data['latitude'])) {
            $data['latitude'] = 0;
        }
        
        // Handle harga_tiket_anak field
        if (empty($data['harga_tiket_anak'])) {
            $data['harga_tiket_anak'] = 0;
        }
        
        // Handle foto utama upload
        if ($request->hasFile('foto_utama')) {
            // Delete old photo if exists
            if ($objek_wisata->foto_utama) {
                Storage::disk('public')->delete('wisata/' . $objek_wisata->foto_utama);
            }
            
            $uploadService = new FileUploadService();
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto_utama'), 'wisata');
            
            if ($uploadResult['success']) {
                $data['foto_utama'] = $uploadResult['filename'];
                Log::info('Tourism main photo updated successfully', ['path' => $uploadResult['path']]);
            } else {
                Log::error('Tourism main photo update failed', ['error' => $uploadResult['error']]);
                return redirect()->back()
                    ->withErrors(['foto_utama' => 'Gagal mengupload foto utama: ' . $uploadResult['error']])
                    ->withInput();
            }
        }
        
        // Handle removed gallery photos
        if ($request->has('removed_gallery_photos') && !empty($request->removed_gallery_photos)) {
            $removedPhotos = is_array($request->removed_gallery_photos) 
                ? $request->removed_gallery_photos 
                : explode(',', $request->removed_gallery_photos);
            
            // Delete files from storage
            foreach ($removedPhotos as $photoName) {
                if ($photoName) {
                    $photoPath = 'wisata/galeri/' . trim($photoName);
                    if (Storage::disk('public')->exists($photoPath)) {
                        Storage::disk('public')->delete($photoPath);
                        Log::info('Gallery photo deleted', ['path' => $photoPath]);
                    }
                }
            }
        }
        
        // Handle existing gallery photos (keep only those not removed)
        $currentGallery = $objek_wisata->galeri_foto ?? [];
        if (!is_array($currentGallery)) {
            $currentGallery = [];
        }
        
        // Remove deleted photos from current gallery
        if ($request->has('removed_gallery_photos') && !empty($request->removed_gallery_photos)) {
            $removedPhotos = is_array($request->removed_gallery_photos) 
                ? $request->removed_gallery_photos 
                : explode(',', $request->removed_gallery_photos);
            
            $currentGallery = array_filter($currentGallery, function($photo) use ($removedPhotos) {
                return !in_array($photo, $removedPhotos);
            });
        }
        
        // Handle new gallery photos
        if ($request->hasFile('galeri_foto')) {
            $uploadService = new FileUploadService();
            
            foreach ($request->file('galeri_foto') as $index => $file) {
                $uploadResult = $uploadService->uploadPhotoWithResize($file, 'wisata/galeri');
                
                if ($uploadResult['success']) {
                    $currentGallery[] = $uploadResult['filename'];
                    Log::info("Tourism gallery file {$index} upload successful", ['path' => $uploadResult['path']]);
                } else {
                    Log::error("Tourism gallery file {$index} upload failed", ['error' => $uploadResult['error']]);
                    return redirect()->back()
                        ->withErrors(['galeri_foto' => "Gagal mengupload foto galeri {$index}: " . $uploadResult['error']])
                        ->withInput();
                }
            }
        }
        
        // Update gallery data
        $data['galeri_foto'] = array_values($currentGallery); // Re-index array
        
        $objek_wisata->update($data);
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil diperbarui.');
    }
    
    public function destroy(ObjekWisata $objek_wisata)
    {
        $objek_wisata->delete();
        
        return redirect()->route('admin.objek-wisata.index')
                        ->with('success', 'Objek wisata berhasil dihapus.');
    }
    
    public function toggleStatus(ObjekWisata $objek_wisata)
    {
        $newStatus = $objek_wisata->status === 'aktif' ? 'nonaktif' : 'aktif';
        $objek_wisata->update(['status' => $newStatus]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status objek wisata berhasil diubah.',
            'status' => $newStatus
        ]);
    }
    
    public function toggleFeatured(ObjekWisata $objek_wisata)
    {
        $objek_wisata->update(['is_featured' => !$objek_wisata->is_featured]);
        
        return response()->json([
            'success' => true,
            'message' => 'Status featured berhasil diubah.',
            'is_featured' => $objek_wisata->is_featured
        ]);
    }
    
    public function statistik(Request $request)
    {
        $stats = [];
        
        // Monthly statistics for current year
        for ($month = 1; $month <= 12; $month++) {
            $monthlyPengunjung = TiketWisata::whereYear('tanggal_kunjungan', now()->year)
                                           ->whereMonth('tanggal_kunjungan', $month)
                                           ->whereNotNull('check_in_at')
                                           ->sum('jumlah_tiket');
            
            $monthlyPendapatan = TiketWisata::whereYear('tanggal_kunjungan', now()->year)
                                           ->whereMonth('tanggal_kunjungan', $month)
                                           ->where('status_tiket', 'paid')
                                           ->sum('total_bayar');
            
            $stats[] = [
                'bulan' => $month,
                'label' => now()->setMonth($month)->format('M'),
                'pengunjung' => $monthlyPengunjung,
                'pendapatan' => $monthlyPendapatan
            ];
        }
        
        // Top tourism objects
        $topObjek = ObjekWisata::orderBy('total_pengunjung_keseluruhan', 'desc')
                              ->limit(10)
                              ->get(['nama_wisata', 'total_pengunjung_keseluruhan', 'pendapatan_keseluruhan']);
        
        return view('admin.objek-wisata.statistik', compact('stats', 'topObjek'));
    }
    
    public function statistikKunjungan(Request $request)
    {
        $year = $request->get('year', now()->year);
        $month = $request->get('month');
        
        // Build query for statistics
        $query = TiketWisata::with('objekWisata');
        
        // Filter by year
        $query->whereYear('tanggal_kunjungan', $year);
        
        // Filter by month if specified
        if ($month) {
            $query->whereMonth('tanggal_kunjungan', $month);
        }
        
        // Monthly visitor statistics
        $monthlyStats = [];
        for ($m = 1; $m <= 12; $m++) {
            $pengunjung = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                    ->whereMonth('tanggal_kunjungan', $m)
                                    ->where('status_tiket', 'paid')
                                    ->sum('jumlah_tiket');
            
            $pendapatan = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                    ->whereMonth('tanggal_kunjungan', $m)
                                    ->where('status_tiket', 'paid')
                                    ->sum('total_bayar');
            
            $monthlyStats[] = [
                'bulan' => $m,
                'nama_bulan' => now()->setMonth($m)->format('F'),
                'singkatan_bulan' => now()->setMonth($m)->format('M'),
                'pengunjung' => $pengunjung,
                'pendapatan' => $pendapatan
            ];
        }
        
        // Tourism object statistics
        $objekWisataStats = ObjekWisata::with(['tiketWisata' => function($query) use ($year, $month) {
                                         $query->whereYear('tanggal_kunjungan', $year)
                                               ->where('status_tiket', 'paid');
                                         if ($month) {
                                             $query->whereMonth('tanggal_kunjungan', $month);
                                         }
                                     }])
                                     ->get()
                                     ->map(function($objek) {
                                         return [
                                             'id' => $objek->id,
                                             'nama' => $objek->nama_wisata,
                                             'kategori' => $objek->kategori_wisata,
                                             'total_pengunjung' => $objek->tiketWisata->sum('jumlah_tiket'),
                                             'total_pendapatan' => $objek->tiketWisata->sum('total_bayar'),
                                             'total_tiket' => $objek->tiketWisata->count()
                                         ];
                                     })
                                     ->sortByDesc('total_pengunjung')
                                     ->values();
        
        // Overall statistics
        $totalPengunjung = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                     ->when($month, function($query) use ($month) {
                                         return $query->whereMonth('tanggal_kunjungan', $month);
                                     })
                                     ->where('status_tiket', 'paid')
                                     ->sum('jumlah_tiket');
        
        $totalPendapatan = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                     ->when($month, function($query) use ($month) {
                                         return $query->whereMonth('tanggal_kunjungan', $month);
                                     })
                                     ->where('status_tiket', 'paid')
                                     ->sum('total_bayar');
        
        $totalTiket = TiketWisata::whereYear('tanggal_kunjungan', $year)
                                ->when($month, function($query) use ($month) {
                                    return $query->whereMonth('tanggal_kunjungan', $month);
                                })
                                ->where('status_tiket', 'paid')
                                ->count();
        
        $objekWisataAktif = ObjekWisata::where('status', 'aktif')->count();
        
        $overallStats = [
            'total_pengunjung' => $totalPengunjung,
            'total_pendapatan' => $totalPendapatan,
            'total_tiket' => $totalTiket,
            'objek_wisata_aktif' => $objekWisataAktif
        ];
        
        return view('admin.objek-wisata.statistik', compact(
            'monthlyStats', 
            'objekWisataStats', 
            'overallStats', 
            'year', 
            'month'
        ));
    }
    
    private function getKategoriList()
    {
        return [
            'wisata_alam' => 'Wisata Alam',
            'wisata_budaya' => 'Wisata Budaya',
            'wisata_sejarah' => 'Wisata Sejarah',
            'wisata_religi' => 'Wisata Religi',
            'wisata_kuliner' => 'Wisata Kuliner',
            'wisata_edukasi' => 'Wisata Edukasi',
            'wisata_adventure' => 'Wisata Adventure',
            'wisata_air' => 'Wisata Air',
            'wisata_pegunungan' => 'Wisata Pegunungan',
            'agrowisata' => 'Agrowisata',
            'ekowisata' => 'Ekowisata',
            'wisata_buatan' => 'Wisata Buatan',
            'lainnya' => 'Lainnya'
        ];
    }
} 
