<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Pbb;
use App\Models\Penduduk;
use App\Models\TagihanPbb;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class PbbController extends Controller
{
    public function index(Request $request)
    {
        $query = Pbb::with(['penduduk', 'tagihanPbb' => function($q) {
            $q->where('tahun_pajak', now()->year)->orderBy('tahun_pajak', 'desc');
        }]);

        if ($request->has('status')) {
            $status = $request->status;
            $query->whereHas('tagihanPbb', function($q) use ($status) {
                $q->where('tahun_pajak', now()->year)
                  ->where('status_bayar', $status);
            });
        }

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nop', 'like', '%' . $search . '%')
                  ->orWhere('alamat_objek', 'like', '%' . $search . '%')
                  ->orWhere('nama_pemilik', 'like', '%' . $search . '%')
                  ->orWhereHas('penduduk', function($p) use ($search) {
                      $p->where('nama', 'like', '%' . $search . '%');
                  });
            });
        }

        $pbbList = $query->latest()->paginate(15);
        
        // Fix statistik to use proper relationships
        $currentYear = now()->year;
        $statistik = [
            'total_pbb' => Pbb::count(),
            'lunas' => TagihanPbb::where('tahun_pajak', $currentYear)->where('status_bayar', 'Lunas')->count(),
            'belum_lunas' => TagihanPbb::where('tahun_pajak', $currentYear)->where('status_bayar', 'Belum Lunas')->count(),
            'total_tagihan' => TagihanPbb::where('tahun_pajak', $currentYear)->sum('total_tagihan')
        ];

        return view('admin.pbb.index', compact('pbbList', 'statistik'));
    }

    public function create()
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.pbb.create', compact('penduduk'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nop' => 'required|string|size:18|unique:pbb,nop',
            'nik' => 'required|exists:penduduk,nik',
            'nama_pemilik' => 'required|string|max:255',
            'alamat_objek' => 'required|string',
            'dusun' => 'nullable|string|max:100',
            'rt' => 'nullable|string|max:5',
            'rw' => 'nullable|string|max:5',
            'luas_tanah' => 'required|numeric|min:0',
            'luas_bangunan' => 'nullable|numeric|min:0',
            'jenis_tanah' => 'required|in:Sawah,Tegalan,Pekarangan,Perkebunan,Hutan,Rawa,Lainnya',
            'jenis_bangunan' => 'required|in:Rumah Tinggal,Ruko,Pabrik,Gudang,Pertanian,Lainnya,Tidak Ada',
            'njop_tanah_per_m2' => 'required|numeric|min:0',
            'njop_bangunan_per_m2' => 'nullable|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $pbb = new Pbb();
            $pbb->nop = $request->nop;
            $pbb->nik = $request->nik;
            $pbb->nama_pemilik = $request->nama_pemilik;
            $pbb->alamat_objek = $request->alamat_objek;
            $pbb->dusun = $request->dusun;
            $pbb->rt = $request->rt;
            $pbb->rw = $request->rw;
            $pbb->luas_tanah = $request->luas_tanah;
            $pbb->luas_bangunan = $request->luas_bangunan ?? 0;
            $pbb->jenis_tanah = $request->jenis_tanah;
            $pbb->jenis_bangunan = $request->jenis_bangunan;
            
            // Calculate NJOP values
            $pbb->njop_tanah = $request->luas_tanah * $request->njop_tanah_per_m2;
            $pbb->njop_bangunan = ($request->luas_bangunan ?? 0) * ($request->njop_bangunan_per_m2 ?? 0);
            $pbb->njop_total = $pbb->njop_tanah + $pbb->njop_bangunan;
            
            // Calculate NJKP (20% of NJOP for residential)
            $pbb->njkp = $pbb->njop_total * 0.20;
            
            // Calculate PBB (0.5% of NJKP)
            $pbb->pbb_terhutang = $pbb->njkp * 0.005;
            
            $pbb->save();

            DB::commit();

            return redirect()->route('admin.pbb.index')
                ->with('success', 'Data PBB berhasil ditambahkan');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Gagal menyimpan data: ' . $e->getMessage()])->withInput();
        }
    }

    public function show(Pbb $pbb)
    {
        $pbb->load('penduduk', 'tagihanPbb');
        return view('admin.pbb.show', compact('pbb'));
    }

    public function edit(Pbb $pbb)
    {
        $penduduk = Penduduk::orderBy('nama')->get();
        return view('admin.pbb.edit', compact('pbb', 'penduduk'));
    }

    public function update(Request $request, Pbb $pbb)
    {
        $validator = Validator::make($request->all(), [
            'nop' => 'required|string|size:18|unique:pbb,nop,' . $pbb->id,
            'nik' => 'required|exists:penduduk,nik',
            'nama_pemilik' => 'required|string|max:255',
            'alamat_objek' => 'required|string',
            'dusun' => 'nullable|string|max:100',
            'rt' => 'nullable|string|max:5',
            'rw' => 'nullable|string|max:5',
            'luas_tanah' => 'required|numeric|min:0',
            'luas_bangunan' => 'nullable|numeric|min:0',
            'jenis_tanah' => 'required|in:Sawah,Tegalan,Pekarangan,Perkebunan,Hutan,Rawa,Lainnya',
            'jenis_bangunan' => 'required|in:Rumah Tinggal,Ruko,Pabrik,Gudang,Pertanian,Lainnya,Tidak Ada',
            'njop_tanah_per_m2' => 'required|numeric|min:0',
            'njop_bangunan_per_m2' => 'nullable|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        DB::beginTransaction();
        try {
            $pbb->nop = $request->nop;
            $pbb->nik = $request->nik;
            $pbb->nama_pemilik = $request->nama_pemilik;
            $pbb->alamat_objek = $request->alamat_objek;
            $pbb->dusun = $request->dusun;
            $pbb->rt = $request->rt;
            $pbb->rw = $request->rw;
            $pbb->luas_tanah = $request->luas_tanah;
            $pbb->luas_bangunan = $request->luas_bangunan ?? 0;
            $pbb->jenis_tanah = $request->jenis_tanah;
            $pbb->jenis_bangunan = $request->jenis_bangunan;
            
            // Recalculate NJOP values
            $pbb->njop_tanah = $request->luas_tanah * $request->njop_tanah_per_m2;
            $pbb->njop_bangunan = ($request->luas_bangunan ?? 0) * ($request->njop_bangunan_per_m2 ?? 0);
            $pbb->njop_total = $pbb->njop_tanah + $pbb->njop_bangunan;
            
            // Calculate NJKP (20% of NJOP for residential)
            $pbb->njkp = $pbb->njop_total * 0.20;
            
            // Calculate PBB (0.5% of NJKP)
            $pbb->pbb_terhutang = $pbb->njkp * 0.005;
            
            $pbb->save();

            DB::commit();

            return redirect()->route('admin.pbb.index')
                ->with('success', 'Data PBB berhasil diperbarui');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Gagal memperbarui data: ' . $e->getMessage()])->withInput();
        }
    }

    public function destroy(Pbb $pbb)
    {
        try {
            $pbb->delete();
            return redirect()->route('admin.pbb.index')
                ->with('success', 'Data PBB berhasil dihapus');
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Gagal menghapus data: ' . $e->getMessage()]);
        }
    }

    public function bayar(Request $request, Pbb $pbb)
    {
        $validator = Validator::make($request->all(), [
            'jumlah_bayar' => 'required|numeric|min:1',
            'tanggal_bayar' => 'required|date',
            'metode_bayar' => 'required|string',
            'tahun_pajak' => 'required|integer'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator);
        }

        DB::beginTransaction();
        try {
            // Get or create tagihan for the specified year
            $tagihan = $pbb->tagihanPbb()->where('tahun_pajak', $request->tahun_pajak)->first();
            
            if (!$tagihan) {
                // Generate tagihan if not exists
                $tagihan = $pbb->generateTagihanTahunan($request->tahun_pajak);
            }

            // Process payment using TagihanPbb method
            $tagihan->prosesPembayaran(
                $request->jumlah_bayar,
                $request->metode_bayar,
                $request->referensi_bayar ?? null
            );

            DB::commit();

            return back()->with('success', 'Pembayaran berhasil dicatat');
        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => 'Gagal mencatat pembayaran: ' . $e->getMessage()]);
        }
    }

    public function cetakSptpd(Pbb $pbb)
    {
        return view('admin.pbb.cetak-sptpd', compact('pbb'));
    }

    // API Methods
    public function apiIndex(Request $request)
    {
        $query = Pbb::with(['penduduk', 'tagihanPbb']);

        if ($request->has('status')) {
            $status = $request->status;
            $query->whereHas('tagihanPbb', function($q) use ($status) {
                $q->where('tahun_pajak', now()->year)
                  ->where('status_bayar', $status);
            });
        }

        if ($request->has('nik')) {
            $query->where('nik', $request->nik);
        }

        $pbbList = $query->get();

        return response()->json([
            'success' => true,
            'data' => $pbbList
        ]);
    }

    public function apiShow(Pbb $pbb)
    {
        $pbb->load('penduduk', 'tagihanPbb');
        
        return response()->json([
            'success' => true,
            'data' => $pbb
        ]);
    }

    public function getTagihan(Pbb $pbb, Request $request)
    {
        $tahun = $request->get('tahun', now()->year);
        $tagihan = $pbb->tagihanPbb()->where('tahun_pajak', $tahun)->first();
        
        if (!$tagihan) {
            return response()->json([
                'success' => false,
                'message' => 'Tagihan untuk tahun ' . $tahun . ' tidak ditemukan'
            ], 404);
        }
        
        // Update denda if needed
        $tagihan->hitungDenda();
        
        return response()->json([
            'success' => true,
            'data' => [
                'nop' => $pbb->nop,
                'tahun_pajak' => $tagihan->tahun_pajak,
                'nilai_pajak' => $tagihan->nilai_pajak,
                'denda' => $tagihan->denda,
                'total_tagihan' => $tagihan->total_tagihan,
                'jumlah_bayar' => $tagihan->jumlah_bayar,
                'sisa_tagihan' => $tagihan->sisa_tagihan,
                'status_bayar' => $tagihan->status_bayar,
                'jatuh_tempo' => $tagihan->jatuh_tempo,
                'is_jatuh_tempo' => $tagihan->isJatuhTempo()
            ]
        ]);
    }
} 