<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\PerformanceOptimizationService;
use App\Services\BugDetectionService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class PerformanceController extends Controller
{
    /**
     * Get system performance metrics API
     */
    public function getMetrics(Request $request)
    {
        try {
            $metrics = [
                'database' => PerformanceOptimizationService::getDatabaseMetrics(),
                'system_health' => BugDetectionService::healthCheck(),
                'cache_stats' => $this->getCacheStatistics(),
                'timestamp' => now()
            ];
            
            return response()->json([
                'success' => true,
                'data' => $metrics
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Run database optimization
     */
    public function optimizeDatabase(Request $request)
    {
        try {
            $results = PerformanceOptimizationService::optimizeDatabase();
            
            if ($results['success']) {
                return response()->json([
                    'success' => true,
                    'message' => 'Optimasi database berhasil',
                    'data' => $results['results']
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Optimasi database gagal'
                ], 500);
            }
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat optimasi'
            ], 500);
        }
    }

    /**
     * Get cache statistics
     */
    private function getCacheStatistics()
    {
        try {
            $cacheDriver = config('cache.default');
            
            $stats = [
                'driver' => $cacheDriver,
                'status' => 'active'
            ];
            
            if ($cacheDriver === 'database') {
                $cacheCount = DB::table('cache')->count();
                $stats['cached_items'] = $cacheCount;
            }
            
            return $stats;
            
        } catch (\Exception $e) {
            return [
                'driver' => config('cache.default'),
                'status' => 'error'
            ];
        }
    }
} 