<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\TiketWisata;
use App\Models\ObjekWisata;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TiketWisataController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = TiketWisata::with(['objekWisata', 'transaksi']);
        
        // Search
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('kode_booking', 'like', "%{$search}%")
                  ->orWhere('nama_pengunjung', 'like', "%{$search}%")
                  ->orWhere('email_pengunjung', 'like', "%{$search}%")
                  ->orWhere('telepon_pengunjung', 'like', "%{$search}%");
            });
        }
        
        // Filter by object wisata
        if ($request->filled('objek_wisata_id')) {
            $query->where('objek_wisata_id', $request->objek_wisata_id);
        }
        
        // Filter by status pembayaran
        if ($request->filled('status_tiket')) {
            $query->where('status_tiket', $request->status_tiket);
        }
        
        // Filter by status tiket
        if ($request->filled('status_tiket')) {
            $query->where('status_tiket', $request->status_tiket);
        }
        
        // Filter by date range
        if ($request->filled('tanggal_dari') && $request->filled('tanggal_sampai')) {
            $query->whereBetween('tanggal_kunjungan', [$request->tanggal_dari, $request->tanggal_sampai]);
        }
        
        $tiketWisata = $query->orderBy('created_at', 'desc')->paginate(20)->withQueryString();
        
        // Get filter options
        $objekWisataList = ObjekWisata::aktif()->pluck('nama_wisata', 'id');
        
        // Statistics
        $stats = [
            'total' => TiketWisata::count(),
            'paid' => TiketWisata::lunas()->count(),
            'pending' => TiketWisata::pending()->count(),
            'used' => TiketWisata::used()->count(),
            'expired' => TiketWisata::expired()->count(),
        ];
        
        return view('admin.tiket-wisata.index', compact('tiketWisata', 'objekWisataList', 'stats'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $objekWisataList = ObjekWisata::aktif()->pluck('nama_wisata', 'id');
        
        return view('admin.tiket-wisata.create', compact('objekWisataList'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'objek_wisata_id' => 'required|exists:objek_wisata,id',
            'nama_pengunjung' => 'required|string|max:255',
            'email_pengunjung' => 'required|email',
            'telepon_pengunjung' => 'required|string|max:20',
            'tanggal_kunjungan' => 'required|date|after_or_equal:today',
            'jumlah_tiket' => 'required|integer|min:1',
            'jenis_tiket' => 'required|in:dewasa,anak',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $objekWisata = ObjekWisata::findOrFail($request->objek_wisata_id);
        
        // Calculate total price
        $hargaPerTiket = $request->jenis_tiket === 'dewasa' ? 
                        $objekWisata->harga_tiket_dewasa : 
                        $objekWisata->harga_tiket_anak;
        
        $totalBayar = $hargaPerTiket * $request->jumlah_tiket;
        
        $data = $request->all();
        $data['harga_per_tiket'] = $hargaPerTiket;
        $data['total_bayar'] = $totalBayar;
        $data['status_tiket'] = 'paid'; // Manual booking is paid
        $data['status_tiket'] = 'aktif';
        $data['kode_booking'] = TiketWisata::generateKodeBooking();
        $data['qr_code'] = TiketWisata::generateQRCode();
        
        TiketWisata::create($data);
        
        return redirect()->route('admin.tiket-wisata.index')
                        ->with('success', 'Tiket wisata berhasil ditambahkan.');
    }

    /**
     * Display the specified resource.
     */
    public function show(TiketWisata $tiketWisata)
    {
        $tiketWisata->load(['objekWisata', 'transaksi']);
        
        return view('admin.tiket-wisata.show', compact('tiketWisata'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(TiketWisata $tiketWisata)
    {
        $objekWisataList = ObjekWisata::aktif()->pluck('nama_wisata', 'id');
        
        return view('admin.tiket-wisata.edit', compact('tiketWisata', 'objekWisataList'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, TiketWisata $tiketWisata)
    {
        $validator = Validator::make($request->all(), [
            'nama_pengunjung' => 'required|string|max:255',
            'email_pengunjung' => 'required|email',
            'telepon_pengunjung' => 'required|string|max:20',
            'tanggal_kunjungan' => 'required|date',
            'jumlah_tiket' => 'required|integer|min:1',
        ]);
        
        if ($validator->fails()) {
            return redirect()->back()
                           ->withErrors($validator)
                           ->withInput();
        }
        
        $tiketWisata->update($request->all());
        
        return redirect()->route('admin.tiket-wisata.index')
                        ->with('success', 'Tiket wisata berhasil diperbarui.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(TiketWisata $tiketWisata)
    {
        // Check if ticket is used
        if ($tiketWisata->status_tiket === 'digunakan') {
            return redirect()->back()
                           ->with('error', 'Tidak dapat menghapus tiket yang sudah digunakan.');
        }
        
        $tiketWisata->delete();
        
        return redirect()->route('admin.tiket-wisata.index')
                        ->with('success', 'Tiket wisata berhasil dihapus.');
    }

    public function checkIn(TiketWisata $tiketWisata)
    {
        if ($tiketWisata->status_tiket !== 'aktif') {
            return response()->json([
                'success' => false,
                'message' => 'Tiket tidak aktif atau sudah digunakan.'
            ], 422);
        }
        
        if ($tiketWisata->status_tiket !== 'paid') {
            return response()->json([
                'success' => false,
                'message' => 'Pembayaran tiket belum lunas.'
            ], 422);
        }
        
        $tiketWisata->checkIn();
        
        return response()->json([
            'success' => true,
            'message' => 'Check-in berhasil.',
            'check_in_at' => $tiketWisata->check_in_at->format('d/m/Y H:i')
        ]);
    }
    
    public function checkOut(TiketWisata $tiketWisata)
    {
        if (!$tiketWisata->check_in_at) {
            return response()->json([
                'success' => false,
                'message' => 'Pengunjung belum check-in.'
            ], 422);
        }
        
        if ($tiketWisata->check_out_at) {
            return response()->json([
                'success' => false,
                'message' => 'Pengunjung sudah check-out.'
            ], 422);
        }
        
        $tiketWisata->checkOut();
        
        return response()->json([
            'success' => true,
            'message' => 'Check-out berhasil.',
            'check_out_at' => $tiketWisata->check_out_at->format('d/m/Y H:i')
        ]);
    }
    
    public function batalkan(TiketWisata $tiketWisata)
    {
        if ($tiketWisata->status_tiket === 'digunakan') {
            return response()->json([
                'success' => false,
                'message' => 'Tidak dapat membatalkan tiket yang sudah digunakan.'
            ], 422);
        }
        
        $tiketWisata->update([
            'status_tiket' => 'dibatalkan',
            'tanggal_pembatalan' => now(),
            'alasan_pembatalan' => request('alasan', 'Dibatalkan oleh admin')
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Tiket berhasil dibatalkan.'
        ]);
    }
    
    public function laporan(Request $request)
    {
        $periode = $request->get('periode', 'bulan');
        $tahun = $request->get('tahun', now()->year);
        $bulan = $request->get('bulan', now()->month);
        
        $query = TiketWisata::with(['objekWisata']);
        
        if ($periode === 'harian') {
            $query->whereYear('tanggal_kunjungan', $tahun)
                  ->whereMonth('tanggal_kunjungan', $bulan);
        } elseif ($periode === 'bulanan') {
            $query->whereYear('tanggal_kunjungan', $tahun);
        } elseif ($periode === 'tahunan') {
            // All data
        }
        
        $tiketData = $query->get();
        
        // Process statistics
        $statistik = [
            'total_tiket' => $tiketData->count(),
            'total_pengunjung' => $tiketData->sum('jumlah_tiket'),
            'total_pendapatan' => $tiketData->where('status_tiket', 'paid')->sum('total_bayar'),
            'tingkat_okupansi' => 0, // Calculate based on capacity
        ];
        
        // Group by object wisata
        $perObjek = $tiketData->groupBy('objek_wisata_id')->map(function($tickets, $objekId) {
            return [
                'nama_wisata' => $tickets->first()->objekWisata->nama_wisata,
                'total_tiket' => $tickets->count(),
                'total_pengunjung' => $tickets->sum('jumlah_tiket'),
                'total_pendapatan' => $tickets->where('status_tiket', 'paid')->sum('total_bayar'),
            ];
        });
        
        return view('admin.tiket-wisata.laporan', compact('statistik', 'perObjek', 'periode', 'tahun', 'bulan'));
    }
    
    public function bulkAction(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:delete,check_in,check_out,cancel',
            'ids' => 'required|array',
            'ids.*' => 'exists:tiket_wisata,id'
        ]);
        
        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid.',
                'errors' => $validator->errors()
            ], 422);
        }
        
        $tiketList = TiketWisata::whereIn('id', $request->ids);
        $count = $tiketList->count();
        
        switch ($request->action) {
            case 'delete':
                $tiketList->whereNotIn('status_tiket', ['digunakan'])->delete();
                $message = "Tiket yang dapat dihapus telah berhasil dihapus.";
                break;
                
            case 'check_in':
                $tiketList->where('status_tiket', 'aktif')
                         ->where('status_tiket', 'paid')
                         ->update([
                             'check_in_at' => now(),
                             'status_tiket' => 'digunakan'
                         ]);
                $message = "Check-in berhasil untuk tiket yang memenuhi syarat.";
                break;
                
            case 'check_out':
                $tiketList->whereNotNull('check_in_at')
                         ->whereNull('check_out_at')
                         ->update(['check_out_at' => now()]);
                $message = "Check-out berhasil untuk tiket yang memenuhi syarat.";
                break;
                
            case 'cancel':
                $tiketList->whereNotIn('status_tiket', ['digunakan'])
                         ->update([
                             'status_tiket' => 'dibatalkan',
                             'tanggal_pembatalan' => now(),
                             'alasan_pembatalan' => 'Dibatalkan oleh admin (bulk action)'
                         ]);
                $message = "Tiket yang dapat dibatalkan telah berhasil dibatalkan.";
                break;
        }
        
        return response()->json([
            'success' => true,
            'message' => $message
        ]);
    }
}
