<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Umkm;
use App\Models\Penduduk;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class UmkmController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $query = Umkm::with(['pemilik', 'user']);
            
            // Filter berdasarkan kategori
            if ($request->filled('kategori')) {
                $query->where('kategori', $request->kategori);
            }
            
            // Filter berdasarkan status
            if ($request->filled('status')) {
                $query->where('status', $request->status);
            }
            
            // Filter berdasarkan verifikasi
            if ($request->filled('verified')) {
                $query->where('is_verified', $request->verified == '1');
            }
            
            // Search
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function($q) use ($search) {
                    $q->where('nama_umkm', 'like', "%{$search}%")
                      ->orWhere('deskripsi', 'like', "%{$search}%")
                      ->orWhereHas('pemilik', function($q) use ($search) {
                          $q->where('nama', 'like', "%{$search}%");
                      });
                });
            }
            
            $umkm = $query->latest()->paginate(10);
        } catch (\Exception $e) {
            Log::error('Error loading UMKM data: ' . $e->getMessage());
            $umkm = collect()->paginate(10);
        }
        
        // Data untuk filter
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        try {
            $penduduk = Penduduk::orderBy('nama')->get();
        } catch (\Exception $e) {
            // Fallback jika database tidak tersedia
            Log::error('Error loading penduduk data: ' . $e->getMessage());
            $penduduk = collect();
        }
        
        return view('admin.umkm.index', compact('umkm', 'kategoris', 'penduduk'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        try {
            $penduduk = Penduduk::orderBy('nama')->get();
        } catch (\Exception $e) {
            // Fallback jika database tidak tersedia
            Log::error('Error loading penduduk data: ' . $e->getMessage());
            $penduduk = collect();
        }
        
        return view('admin.umkm.create', compact('kategoris', 'penduduk'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nama_umkm' => 'required|string|max:255',
            'kategori' => 'required|in:makanan_minuman,kerajinan_tangan,fashion_tekstil,jasa_layanan,pertanian_perkebunan,peternakan_perikanan,teknologi_digital,perdagangan_retail,pariwisata_budaya,lainnya',
            'pemilik_id' => 'required|exists:penduduk,id',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'nomor_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'website' => 'nullable|url',
            'skala_usaha' => 'required|in:mikro,kecil,menengah',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();
        $data['slug'] = Str::slug($request->nama_umkm);
        
        // Handle logo upload
        if ($request->hasFile('logo')) {
            $logo = $request->file('logo');
            $originalName = $logo->getClientOriginalName();
            $extension = $logo->getClientOriginalExtension();
            
            // Clean filename to avoid URL encoding issues
            $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
            $logoName = time() . '_logo_' . $cleanName . '.' . $extension;
            
            $logo->storeAs('public/umkm/logo', $logoName);
            $data['logo'] = $logoName;
        }
        
        // Handle galeri foto upload
        if ($request->hasFile('galeri_foto')) {
            $galeriFiles = [];
            foreach ($request->file('galeri_foto') as $file) {
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                
                // Clean filename to avoid URL encoding issues
                $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
                $fileName = time() . '_galeri_' . $cleanName . '.' . $extension;
                
                $file->storeAs('public/umkm/galeri', $fileName);
                $galeriFiles[] = $fileName;
            }
            $data['galeri_foto'] = $galeriFiles;
        }
        
        // Handle jam operasional
        if ($request->filled('jam_operasional')) {
            $data['jam_operasional'] = $request->jam_operasional;
        }
        
        // Handle hari libur
        if ($request->filled('hari_libur')) {
            $data['hari_libur'] = $request->hari_libur;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = explode(',', $request->tags);
        }

        try {
            Umkm::create($data);
            return redirect()->route('admin.umkm.index')
                ->with('success', 'Data UMKM berhasil ditambahkan');
        } catch (\Exception $e) {
            Log::error('Error creating UMKM: ' . $e->getMessage());
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan saat menyimpan data UMKM'])
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(Umkm $umkm)
    {
        try {
            $umkm->load(['pemilik', 'produk', 'user']);
            $umkm->incrementViewCount();
        } catch (\Exception $e) {
            // Log error but continue
            Log::error('Error loading UMKM data: ' . $e->getMessage());
        }
        
        return view('admin.umkm.show', compact('umkm'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Umkm $umkm)
    {
        $kategoris = [
            'makanan_minuman' => 'Makanan & Minuman',
            'kerajinan_tangan' => 'Kerajinan Tangan',
            'fashion_tekstil' => 'Fashion & Tekstil',
            'jasa_layanan' => 'Jasa & Layanan',
            'pertanian_perkebunan' => 'Pertanian & Perkebunan',
            'peternakan_perikanan' => 'Peternakan & Perikanan',
            'teknologi_digital' => 'Teknologi Digital',
            'perdagangan_retail' => 'Perdagangan & Retail',
            'pariwisata_budaya' => 'Pariwisata & Budaya',
            'lainnya' => 'Lainnya'
        ];
        
        try {
            $penduduk = Penduduk::orderBy('nama')->get();
        } catch (\Exception $e) {
            // Fallback jika database tidak tersedia
            Log::error('Error loading penduduk data: ' . $e->getMessage());
            $penduduk = collect();
        }
        
        return view('admin.umkm.edit', compact('umkm', 'kategoris', 'penduduk'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Umkm $umkm)
    {
        $validator = Validator::make($request->all(), [
            'nama_umkm' => 'required|string|max:255',
            'kategori' => 'required|in:makanan_minuman,kerajinan_tangan,fashion_tekstil,jasa_layanan,pertanian_perkebunan,peternakan_perikanan,teknologi_digital,perdagangan_retail,pariwisata_budaya,lainnya',
            'pemilik_id' => 'required|exists:penduduk,id',
            'deskripsi' => 'required|string',
            'alamat' => 'required|string',
            'nomor_telepon' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'website' => 'nullable|url',
            'skala_usaha' => 'required|in:mikro,kecil,menengah',
            'status' => 'required|in:aktif,nonaktif,pending_verifikasi',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $data = $validator->validated();
        $data['slug'] = Str::slug($request->nama_umkm);
        
        // Handle checkbox fields
        $data['is_verified'] = $request->has('is_verified');
        $data['aktif_marketplace'] = $request->has('aktif_marketplace');
        $data['bisa_cod'] = $request->has('bisa_cod');
        $data['bisa_transfer'] = $request->has('bisa_transfer');
        
        // Handle logo removal
        if ($request->filled('remove_logo') && $request->remove_logo == '1') {
            if ($umkm->logo) {
                Storage::delete('public/umkm/logo/' . $umkm->logo);
            }
            $data['logo'] = null;
        }
        
        // Handle logo upload
        if ($request->hasFile('logo')) {
            // Delete old logo
            if ($umkm->logo) {
                Storage::delete('public/umkm/logo/' . $umkm->logo);
            }
            
            $logo = $request->file('logo');
            $originalName = $logo->getClientOriginalName();
            $extension = $logo->getClientOriginalExtension();
            
            // Clean filename to avoid URL encoding issues
            $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
            $logoName = time() . '_logo_' . $cleanName . '.' . $extension;
            
            $logo->storeAs('public/umkm/logo', $logoName);
            $data['logo'] = $logoName;
        }
        
        // Handle gallery photo removal
        if ($request->filled('remove_gallery')) {
            $removeIndexes = explode(',', $request->remove_gallery);
            $currentGallery = $umkm->galeri_foto ?? [];
            
            foreach ($removeIndexes as $index) {
                if (isset($currentGallery[$index])) {
                    Storage::delete('public/umkm/galeri/' . $currentGallery[$index]);
                    unset($currentGallery[$index]);
                }
            }
            
            $data['galeri_foto'] = array_values($currentGallery);
        }
        
        // Handle new gallery photos
        if ($request->hasFile('galeri_foto')) {
            $currentGallery = $data['galeri_foto'] ?? [];
            
            foreach ($request->file('galeri_foto') as $file) {
                $originalName = $file->getClientOriginalName();
                $extension = $file->getClientOriginalExtension();
                
                // Clean filename to avoid URL encoding issues
                $cleanName = preg_replace('/[^\w\-_\.]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
                $fileName = time() . '_galeri_' . $cleanName . '.' . $extension;
                
                $file->storeAs('public/umkm/galeri', $fileName);
                $currentGallery[] = $fileName;
            }
            
            $data['galeri_foto'] = $currentGallery;
        }
        
        // Handle jam operasional
        if ($request->filled('jam_operasional')) {
            $data['jam_operasional'] = $request->jam_operasional;
        }
        
        // Handle hari libur
        if ($request->filled('hari_libur')) {
            $data['hari_libur'] = $request->hari_libur;
        }
        
        // Handle tags
        if ($request->filled('tags')) {
            $data['tags'] = array_map('trim', explode(',', $request->tags));
        }

        try {
            $umkm->update($data);
            return redirect()->route('admin.umkm.show', $umkm->id)
                ->with('success', 'Data UMKM berhasil diperbarui');
        } catch (\Exception $e) {
            Log::error('Error updating UMKM: ' . $e->getMessage());
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan saat memperbarui data UMKM'])
                ->withInput();
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Umkm $umkm)
    {
        try {
            // Delete files
            if ($umkm->logo) {
                try {
                    Storage::delete('public/umkm/logo/' . $umkm->logo);
                } catch (\Exception $e) {
                    Log::warning('Error deleting logo: ' . $e->getMessage());
                }
            }
            
            if ($umkm->galeri_foto) {
                foreach ($umkm->galeri_foto as $file) {
                    try {
                        Storage::delete('public/umkm/galeri/' . $file);
                    } catch (\Exception $e) {
                        Log::warning('Error deleting gallery photo: ' . $e->getMessage());
                    }
                }
            }
            
            $umkm->delete();
            return redirect()->route('admin.umkm.index')
                ->with('success', 'Data UMKM berhasil dihapus');
        } catch (\Exception $e) {
            Log::error('Error deleting UMKM: ' . $e->getMessage());
            return redirect()->back()
                ->withErrors(['error' => 'Terjadi kesalahan saat menghapus data UMKM']);
        }
    }

    public function verify(Request $request, Umkm $umkm)
    {
        try {
            $request->validate([
                'status' => 'required|in:aktif,nonaktif',
                'catatan_verifikasi' => 'nullable|string'
            ]);

            $umkm->update([
                'status' => $request->status,
                'is_verified' => $request->status === 'aktif',
                'verified_at' => now(),
                'verified_by' => Auth::id(),
                'catatan_verifikasi' => $request->catatan_verifikasi
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Status UMKM berhasil diperbarui'
            ]);
        } catch (\Exception $e) {
            Log::error('Error verifying UMKM: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat memverifikasi UMKM'
            ], 500);
        }
    }

    public function toggleMarketplace(Umkm $umkm)
    {
        try {
            $umkm->update([
                'aktif_marketplace' => !$umkm->aktif_marketplace
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Status marketplace berhasil diperbarui',
                'aktif_marketplace' => $umkm->aktif_marketplace
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling marketplace: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengubah status marketplace'
            ], 500);
        }
    }

    public function statistik()
    {
        try {
            $totalUmkm = Umkm::count();
            $umkmAktif = Umkm::where('status', 'aktif')->count();
            $umkmVerified = Umkm::where('is_verified', true)->count();
            $umkmMarketplace = Umkm::where('aktif_marketplace', true)->count();
            
            $umkmByKategori = Umkm::selectRaw('kategori, COUNT(*) as total')
                ->groupBy('kategori')
                ->get();
                
            $umkmBySkala = Umkm::selectRaw('skala_usaha, COUNT(*) as total')
                ->groupBy('skala_usaha')
                ->get();
        } catch (\Exception $e) {
            Log::error('Error loading UMKM statistics: ' . $e->getMessage());
            $totalUmkm = 0;
            $umkmAktif = 0;
            $umkmVerified = 0;
            $umkmMarketplace = 0;
            $umkmByKategori = collect();
            $umkmBySkala = collect();
        }

        return view('admin.umkm.statistik', compact(
            'totalUmkm', 'umkmAktif', 'umkmVerified', 'umkmMarketplace',
            'umkmByKategori', 'umkmBySkala'
        ));
    }

    public function laporan(Request $request)
    {
        try {
            $query = Umkm::with(['pemilik', 'produk']);
            
            if ($request->filled('start_date')) {
                $query->whereDate('created_at', '>=', $request->start_date);
            }
            
            if ($request->filled('end_date')) {
                $query->whereDate('created_at', '<=', $request->end_date);
            }
            
            if ($request->filled('kategori')) {
                $query->where('kategori', $request->kategori);
            }
            
            $umkm = $query->get();
        } catch (\Exception $e) {
            Log::error('Error loading UMKM laporan: ' . $e->getMessage());
            $umkm = collect();
        }
        
        return view('admin.umkm.laporan', compact('umkm'));
    }
}

