<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class UploadController extends Controller
{
    /**
     * Upload image untuk CKEditor
     */
    public function uploadImage(Request $request)
    {
        try {
            // Validasi file
            $request->validate([
                'upload' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
            ]);

            $file = $request->file('upload');
            
            // Generate nama file unik
            $fileName = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            
            // Path untuk menyimpan
            $path = 'uploads/editor/' . date('Y/m');
            
            // Pastikan direktori ada
            if (!Storage::disk('public')->exists($path)) {
                Storage::disk('public')->makeDirectory($path, 0755, true);
            }
            
            // Simpan file
            $fullPath = $path . '/' . $fileName;
            $stored = Storage::disk('public')->put($fullPath, file_get_contents($file));
            
            if (!$stored) {
                throw new \Exception('Failed to store file');
            }
            
            // URL untuk diakses
            $url = asset('storage/' . $fullPath);
            
            // Log successful upload
            Log::info('Image uploaded successfully', [
                'file_name' => $fileName,
                'path' => $fullPath,
                'url' => $url
            ]);
            
            // Response format untuk CKEditor
            return response()->json([
                'url' => $url,
                'uploaded' => true,
                'fileName' => $fileName
            ]);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::warning('Upload validation failed', ['errors' => $e->errors()]);
            
            return response()->json([
                'uploaded' => false,
                'error' => [
                    'message' => 'File tidak valid. Pastikan file adalah gambar (JPG, PNG, GIF, WebP) dengan ukuran maksimal 2MB.'
                ]
            ], 422);
            
        } catch (\Exception $e) {
            Log::error('Upload image error: ' . $e->getMessage(), [
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return response()->json([
                'uploaded' => false,
                'error' => [
                    'message' => 'Terjadi kesalahan saat upload gambar: ' . $e->getMessage()
                ]
            ], 500);
        }
    }
    
    /**
     * Get uploaded images list
     */
    public function getImages(Request $request)
    {
        try {
            $page = $request->get('page', 1);
            $perPage = $request->get('per_page', 20);
            
            $basePath = 'uploads/editor';
            $files = [];
            
            // Get all files from uploads/editor directory
            if (Storage::disk('public')->exists($basePath)) {
                $allFiles = Storage::disk('public')->allFiles($basePath);
                
                foreach ($allFiles as $file) {
                    $fileInfo = pathinfo($file);
                    if (in_array(strtolower($fileInfo['extension'] ?? ''), ['jpg', 'jpeg', 'png', 'gif', 'webp'])) {
                        $files[] = [
                            'url' => asset('storage/' . $file),
                            'name' => $fileInfo['basename'],
                            'size' => Storage::disk('public')->size($file),
                            'modified' => Storage::disk('public')->lastModified($file)
                        ];
                    }
                }
            }
            
            // Sort by modified date (newest first)
            usort($files, function($a, $b) {
                return $b['modified'] - $a['modified'];
            });
            
            // Paginate
            $total = count($files);
            $offset = ($page - 1) * $perPage;
            $paginatedFiles = array_slice($files, $offset, $perPage);
            
            return response()->json([
                'files' => $paginatedFiles,
                'pagination' => [
                    'current_page' => $page,
                    'per_page' => $perPage,
                    'total' => $total,
                    'total_pages' => ceil($total / $perPage)
                ]
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get images error: ' . $e->getMessage());
            
            return response()->json([
                'error' => 'Terjadi kesalahan saat mengambil daftar gambar.'
            ], 500);
        }
    }
    
    /**
     * Delete uploaded image
     */
    public function deleteImage(Request $request)
    {
        try {
            $request->validate([
                'path' => 'required|string'
            ]);
            
            $path = $request->input('path');
            
            if (Storage::disk('public')->exists($path)) {
                Storage::disk('public')->delete($path);
                
                return response()->json([
                    'success' => true,
                    'message' => 'Gambar berhasil dihapus.'
                ]);
            }
            
            return response()->json([
                'success' => false,
                'message' => 'Gambar tidak ditemukan.'
            ], 404);
            
        } catch (\Exception $e) {
            Log::error('Delete image error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat menghapus gambar.'
            ], 500);
        }
    }
} 