<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PerangkatDesa;
use Illuminate\Http\Request;

class StaffController extends Controller
{
    /**
     * Display a listing of village staff
     */
    public function index(Request $request)
    {
        try {
            $query = PerangkatDesa::with('penduduk')->where('status', 'Aktif');

            // Order by position hierarchy
            $query->orderByRaw("
                CASE jabatan
                    WHEN 'Kepala Desa' THEN 1
                    WHEN 'Sekretaris Desa' THEN 2
                    WHEN 'Kaur Tata Usaha dan Umum' THEN 3
                    WHEN 'Kaur Keuangan' THEN 4
                    WHEN 'Kaur Perencanaan' THEN 5
                    WHEN 'Kasi Pemerintahan' THEN 6
                    WHEN 'Kasi Kesejahteraan' THEN 7
                    WHEN 'Kasi Pelayanan' THEN 8
                    WHEN 'Kepala Dusun' THEN 9
                    ELSE 99
                END, nama
            ");

            $limit = $request->get('limit', 20);
            $staff = $query->paginate($limit);

            // Transform data for API response
            $transformedData = $staff->getCollection()->map(function ($item) {
                return [
                    'id' => $item->id,
                    'nama' => $item->nama,
                    'jabatan' => $item->jabatan,
                    'nip' => $item->nip,
                    'status' => $item->status,
                    'foto' => $item->foto ? url('storage/' . $item->foto) : null,
                    'alamat' => $item->alamat,
                    'telepon' => $item->telepon,
                    'email' => $item->email,
                    'pendidikan_terakhir' => $item->pendidikan_terakhir,
                    'tanggal_mulai_jabatan' => $item->tanggal_mulai_jabatan,
                    'masa_jabatan' => $item->masa_jabatan,
                    'created_at' => $item->created_at,
                    'updated_at' => $item->updated_at,
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedData,
                'meta' => [
                    'current_page' => $staff->currentPage(),
                    'last_page' => $staff->lastPage(),
                    'per_page' => $staff->perPage(),
                    'total' => $staff->total(),
                ],
                'message' => 'Data perangkat desa berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data perangkat desa',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
