<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ObjekWisata;
use App\Models\TiketWisata;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class TourismController extends Controller
{
    /**
     * Display a listing of tourist attractions
     */
    public function index(Request $request)
    {
        try {
            $query = ObjekWisata::where('status', 'aktif');

            // Filter by category
            if ($request->has('kategori') && $request->kategori) {
                $query->where('kategori', $request->kategori);
            }

            // Search functionality
            if ($request->has('search') && $request->search) {
                $query->where(function($q) use ($request) {
                    $q->where('nama_objek', 'like', '%' . $request->search . '%')
                      ->orWhere('deskripsi', 'like', '%' . $request->search . '%')
                      ->orWhere('lokasi', 'like', '%' . $request->search . '%');
                });
            }

            $page = $request->get('page', 1);
            $limit = $request->get('limit', 10);

            $wisata = $query->orderBy('created_at', 'desc')
                           ->paginate($limit, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'data' => $wisata->items(),
                'meta' => [
                    'current_page' => $wisata->currentPage(),
                    'last_page' => $wisata->lastPage(),
                    'per_page' => $wisata->perPage(),
                    'total' => $wisata->total(),
                ],
                'message' => 'Data objek wisata berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data objek wisata',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified tourist attraction
     */
    public function show($id)
    {
        try {
            $wisata = ObjekWisata::where('status', 'aktif')->findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $wisata,
                'message' => 'Detail objek wisata berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Objek wisata tidak ditemukan',
                'error' => $e->getMessage()
            ], 404);
        }
    }

    /**
     * Book a ticket for tourist attraction
     */
    public function bookTicket(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'objek_wisata_id' => 'required|exists:objek_wisata,id',
                'nama_pengunjung' => 'required|string|max:255',
                'nomor_telepon' => 'required|string|max:20',
                'email' => 'required|email|max:255',
                'jumlah_tiket' => 'required|integer|min:1|max:10',
                'tanggal_kunjungan' => 'required|date|after_or_equal:today',
                'catatan' => 'nullable|string|max:500',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Data tidak valid',
                    'errors' => $validator->errors()
                ], 422);
            }

            $objekWisata = ObjekWisata::findOrFail($request->objek_wisata_id);
            $totalHarga = $objekWisata->harga_tiket * $request->jumlah_tiket;
            $kodeBooking = 'TWT-' . strtoupper(Str::random(8));

            $tiket = TiketWisata::create([
                'objek_wisata_id' => $request->objek_wisata_id,
                'nama_pengunjung' => $request->nama_pengunjung,
                'nomor_telepon' => $request->nomor_telepon,
                'email' => $request->email,
                'jumlah_tiket' => $request->jumlah_tiket,
                'tanggal_kunjungan' => $request->tanggal_kunjungan,
                'total_harga' => $totalHarga,
                'status' => 'pending',
                'kode_booking' => $kodeBooking,
                'catatan' => $request->catatan,
                'user_id' => auth()->id(),
            ]);

            $tiket->load('objekWisata');

            return response()->json([
                'success' => true,
                'data' => $tiket,
                'message' => 'Booking tiket berhasil dibuat'
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal membuat booking tiket',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user's tickets
     */
    public function getTickets(Request $request)
    {
        try {
            $query = TiketWisata::with('objekWisata')->where('user_id', auth()->id());

            // Filter by status
            if ($request->has('status') && $request->status) {
                $query->where('status', $request->status);
            }

            $page = $request->get('page', 1);
            $limit = $request->get('limit', 10);

            $tickets = $query->orderBy('created_at', 'desc')
                            ->paginate($limit, ['*'], 'page', $page);

            return response()->json([
                'success' => true,
                'data' => $tickets->items(),
                'meta' => [
                    'current_page' => $tickets->currentPage(),
                    'last_page' => $tickets->lastPage(),
                    'per_page' => $tickets->perPage(),
                    'total' => $tickets->total(),
                ],
                'message' => 'Data tiket berhasil diambil'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengambil data tiket',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
