<?php

namespace App\Http\Controllers;

use App\Models\ArsipSuratKeluar;
use App\Models\AuditLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Auth;

class ArsipSuratKeluarController extends Controller
{
    public function index(Request $request)
    {
        $query = ArsipSuratKeluar::with('creator');

        if ($request->filled('tipe_surat')) {
            $query->where('tipe_surat', $request->tipe_surat);
        }

        if ($request->filled('tahun')) {
            $query->whereYear('tanggal_surat', $request->tahun);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('nomor_surat', 'like', "%{$search}%")
                  ->orWhere('perihal', 'like', "%{$search}%")
                  ->orWhere('tujuan', 'like', "%{$search}%");
            });
        }

        $suratKeluar = $query->latest('tanggal_surat')->paginate(15);
        
        return view('admin.arsip-surat-keluar.index', compact('suratKeluar'));
    }

    public function create()
    {
        $tipeSurat = [
            'Surat Keterangan', 'Surat Pengantar', 'Surat Izin',
            'Surat Undangan', 'Surat Edaran', 'Surat Permohonan',
            'Surat Keputusan', 'Surat Tugas', 'Surat Rekomendasi'
        ];
        
        return view('admin.arsip-surat-keluar.create', compact('tipeSurat'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'tipe_surat' => 'required|string|max:100',
            'perihal' => 'required|string|max:255',
            'tujuan' => 'required|string|max:255',
            'tanggal_surat' => 'required|date',
            'file_surat' => 'required|file|mimes:pdf,doc,docx|max:5120',
            'keterangan' => 'nullable|string|max:500'
        ]);

        $data = $request->except(['file_surat']);
        $data['created_by'] = Auth::id();
        
        // Generate nomor surat
        $data['nomor_surat'] = ArsipSuratKeluar::generateNomorSurat($request->tipe_surat);

        // Handle file upload
        if ($request->hasFile('file_surat')) {
            $file = $request->file('file_surat');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $data['file_surat'] = $file->storeAs('arsip-surat-keluar', $fileName, 'public');
        }

        $suratKeluar = ArsipSuratKeluar::create($data);

        AuditLog::logDataChange('create', $suratKeluar, null, $data);

        return redirect()->route('admin.arsip-surat-keluar.index')
                        ->with('success', 'Arsip surat keluar berhasil ditambahkan');
    }

    public function show(ArsipSuratKeluar $arsipSuratKeluar)
    {
        return view('admin.arsip-surat-keluar.show', compact('arsipSuratKeluar'));
    }

    public function edit(ArsipSuratKeluar $arsipSuratKeluar)
    {
        $tipeSurat = [
            'Surat Keterangan', 'Surat Pengantar', 'Surat Izin',
            'Surat Undangan', 'Surat Edaran', 'Surat Permohonan',
            'Surat Keputusan', 'Surat Tugas', 'Surat Rekomendasi'
        ];
        
        return view('admin.arsip-surat-keluar.edit', compact('arsipSuratKeluar', 'tipeSurat'));
    }

    public function update(Request $request, ArsipSuratKeluar $arsipSuratKeluar)
    {
        $request->validate([
            'tipe_surat' => 'required|string|max:100',
            'perihal' => 'required|string|max:255',
            'tujuan' => 'required|string|max:255',
            'tanggal_surat' => 'required|date',
            'file_surat' => 'nullable|file|mimes:pdf,doc,docx|max:5120',
            'keterangan' => 'nullable|string|max:500'
        ]);

        $oldData = $arsipSuratKeluar->toArray();
        $data = $request->except(['file_surat']);

        // Handle file upload
        if ($request->hasFile('file_surat')) {
            // Delete old file
            if ($arsipSuratKeluar->file_surat) {
                Storage::disk('public')->delete($arsipSuratKeluar->file_surat);
            }
            
            $file = $request->file('file_surat');
            $fileName = time() . '_' . $file->getClientOriginalName();
            $data['file_surat'] = $file->storeAs('arsip-surat-keluar', $fileName, 'public');
        }

        $arsipSuratKeluar->update($data);

        AuditLog::logDataChange('update', $arsipSuratKeluar, $oldData, $data);

        return redirect()->route('admin.arsip-surat-keluar.index')
                        ->with('success', 'Arsip surat keluar berhasil diperbarui');
    }

    public function destroy(ArsipSuratKeluar $arsipSuratKeluar)
    {
        $oldData = $arsipSuratKeluar->toArray();

        // Delete file
        if ($arsipSuratKeluar->file_surat) {
            Storage::disk('public')->delete($arsipSuratKeluar->file_surat);
        }

        $arsipSuratKeluar->delete();

        AuditLog::logDataChange('delete', $arsipSuratKeluar, $oldData, null);

        return redirect()->route('admin.arsip-surat-keluar.index')
                        ->with('success', 'Arsip surat keluar berhasil dihapus');
    }

    public function download(ArsipSuratKeluar $arsipSuratKeluar)
    {
        if (!$arsipSuratKeluar->file_surat || !Storage::disk('public')->exists($arsipSuratKeluar->file_surat)) {
            return abort(404, 'File tidak ditemukan');
        }

        AuditLog::logActivity([
            'action_type' => 'DOWNLOAD',
            'module' => 'ArsipSuratKeluar',
            'table_name' => 'arsip_surat_keluar',
            'record_id' => $arsipSuratKeluar->id,
            'description' => "Download file surat: {$arsipSuratKeluar->nomor_surat}"
        ]);

        return Storage::disk('public')->download($arsipSuratKeluar->file_surat);
    }

    public function statistik()
    {
        $stats = [
            'total_surat' => ArsipSuratKeluar::count(),
            'per_tipe' => ArsipSuratKeluar::selectRaw('tipe_surat, COUNT(*) as total')
                                        ->groupBy('tipe_surat')
                                        ->get(),
            'per_bulan' => ArsipSuratKeluar::selectRaw('YEAR(tanggal_surat) as tahun, MONTH(tanggal_surat) as bulan, COUNT(*) as total')
                                         ->groupBy('tahun', 'bulan')
                                         ->orderBy('tahun', 'desc')
                                         ->orderBy('bulan', 'desc')
                                         ->limit(12)
                                         ->get()
        ];
        
        return view('admin.arsip-surat-keluar.statistik', compact('stats'));
    }

    public function cetakRegister(Request $request)
    {
        $request->validate([
            'tahun' => 'required|integer|min:2020|max:' . (now()->year + 1),
            'bulan' => 'nullable|integer|min:1|max:12'
        ]);

        $query = ArsipSuratKeluar::whereYear('tanggal_surat', $request->tahun);
        
        if ($request->filled('bulan')) {
            $query->whereMonth('tanggal_surat', $request->bulan);
        }

        $suratKeluar = $query->orderBy('tanggal_surat')->get();

        AuditLog::logActivity([
            'action_type' => 'EXPORT',
            'module' => 'ArsipSuratKeluar',
            'description' => "Cetak register surat keluar tahun {$request->tahun}",
            'metadata' => [
                'tahun' => $request->tahun,
                'bulan' => $request->bulan
            ]
        ]);

        return view('admin.arsip-surat-keluar.register', compact('suratKeluar'));
    }
} 