<?php

namespace App\Http\Controllers;

use App\Models\Berita;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use App\Services\WhatsAppService;

class BeritaController extends Controller
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    /**
     * Display a listing of news
     */
    public function index(Request $request)
    {
        $query = Berita::query();

        // Search functionality
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', '%' . $search . '%')
                  ->orWhere('konten', 'like', '%' . $search . '%')
                  ->orWhere('kategori', 'like', '%' . $search . '%');
            });
        }

        // Filter by category
        if ($request->has('kategori') && $request->kategori) {
            $query->where('kategori', $request->kategori);
        }

        // Filter by status
        if ($request->has('status') && $request->status) {
            $query->where('status', $request->status);
        }

        $berita = $query->withCount('comments')->orderBy('created_at', 'desc')->paginate(10);
        $categories = Berita::distinct('kategori')->pluck('kategori');
        
        // Get statistics including comments
        $stats = [
            'total' => Berita::count(),
            'published' => Berita::where('status', 'published')->count(),
            'draft' => Berita::where('status', 'draft')->count(),
            'total_views' => Berita::sum('views'),
            'total_comments' => \App\Models\BeritaComment::count(),
            'pending_comments' => \App\Models\BeritaComment::where('status', 'pending')->count(),
        ];

        if ($request->ajax()) {
            return response()->json([
                'html' => view('admin.berita.table', compact('berita'))->render(),
                'pagination' => $berita->links()->render()
            ]);
        }

        return view('admin.berita.index', compact('berita', 'categories', 'stats'));
    }

    /**
     * Show the form for creating new news
     */
    public function create()
    {
        $categories = ['Pengumuman', 'Pembangunan', 'Kesehatan', 'Sosial', 'Ekonomi', 'Pendidikan'];
        return view('admin.berita.create', compact('categories'));
    }

    /**
     * Store new news
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'judul' => 'required|string|max:255',
                'konten' => 'required|string',
                'ringkasan' => 'required|string|max:500',
                'kategori' => 'required|string|max:100',
                'status' => 'required|in:draft,published,scheduled',
                'tanggal_publish' => 'nullable|date',
                'prioritas' => 'nullable|in:normal,penting,urgent',
                'gambar' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
                'alt_text' => 'nullable|string|max:255',
                'meta_description' => 'nullable|string|max:160',
                'keywords' => 'nullable|string|max:255',
                'tags' => 'nullable|string|max:255',
                'allow_comments' => 'nullable|boolean',
                'is_featured' => 'nullable|boolean',
                'send_wa_broadcast' => 'nullable|boolean'
            ]);

            // Generate slug
            $validated['slug'] = Str::slug($validated['judul']) . '-' . time();
            $validated['user_id'] = auth()->id() ?? 1; // Current user or default admin

            // Handle status and publish date
            if ($validated['status'] === 'published') {
                $validated['tanggal_terbit'] = now();
            } elseif ($validated['status'] === 'scheduled' && !empty($validated['tanggal_publish'])) {
                $validated['tanggal_terbit'] = $validated['tanggal_publish'];
            }

            // Handle priority default
            $validated['prioritas'] = $validated['prioritas'] ?? 'normal';

            // Handle boolean fields
            $validated['allow_comments'] = $request->has('allow_comments') ? 1 : 0;
            $validated['is_featured'] = $request->has('is_featured') ? 1 : 0;

            // Handle image upload
            if ($request->hasFile('gambar')) {
                $image = $request->file('gambar');
                $imageName = time() . '_' . Str::slug($validated['judul']) . '.' . $image->getClientOriginalExtension();
                
                // Store in berita directory
                $path = $image->storeAs('berita', $imageName, 'public');
                $validated['gambar'] = 'berita/' . $imageName;
            }

            // Create berita
            $berita = Berita::create($validated);

            // Send WhatsApp broadcast if requested and published
            if ($request->has('send_wa_broadcast') && $validated['status'] === 'published') {
                try {
                    $this->broadcastBerita($berita);
                } catch (\Exception $e) {
                    Log::error('WhatsApp broadcast failed: ' . $e->getMessage());
                    // Continue execution even if broadcast fails
                }
            }

            return redirect()->route('admin.berita.index')
                            ->with('success', 'Berita berhasil dibuat!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            return redirect()->back()
                            ->withErrors($e->validator)
                            ->withInput()
                            ->with('error', 'Terdapat kesalahan dalam form. Silakan periksa kembali.');
        } catch (\Exception $e) {
            Log::error('Error creating berita: ' . $e->getMessage());
            
            return redirect()->back()
                            ->withInput()
                            ->with('error', 'Terjadi kesalahan saat menyimpan berita. Silakan coba lagi.');
        }
    }

    /**
     * Show specific news
     */
    public function show(Berita $berita)
    {
        // Increment views
        $berita->increment('views');
        
        // Calculate SEO score
        $seo_score = $this->calculateSeoScore($berita);
        
        // Calculate reading time
        $word_count = str_word_count(strip_tags($berita->konten));
        $reading_time = max(1, ceil($word_count / 200)); // 200 words per minute
        
        // Get related articles
        $related_articles = Berita::where('kategori', $berita->kategori)
            ->where('id', '!=', $berita->getKey())
            ->where('status', 'published')
            ->latest()
            ->limit(3)
            ->get();
        
        // Recent activities (mock data for now)
        $recent_activities = [
            [
                'message' => 'Berita dipublikasikan',
                'time' => $berita->created_at->diffForHumans(),
                'icon' => 'upload',
                'color' => 'success'
            ],
            [
                'message' => 'Berita diedit terakhir',
                'time' => $berita->updated_at->diffForHumans(),
                'icon' => 'edit',
                'color' => 'info'
            ]
        ];
        
        return view('admin.berita.show', compact(
            'berita', 
            'seo_score', 
            'reading_time', 
            'word_count', 
            'related_articles', 
            'recent_activities'
        ));
    }
    
    /**
     * Calculate SEO score based on various factors
     */
    private function calculateSeoScore(Berita $berita)
    {
        $score = 0;
        
        // Title length (optimal: 50-60 characters)
        $titleLength = strlen($berita->judul);
        if ($titleLength >= 50 && $titleLength <= 60) {
            $score += 20;
        } elseif ($titleLength >= 40 && $titleLength <= 70) {
            $score += 15;
        } elseif ($titleLength >= 30) {
            $score += 10;
        }
        
        // Meta description
        if (!empty($berita->meta_description)) {
            $metaLength = strlen($berita->meta_description);
            if ($metaLength >= 150 && $metaLength <= 160) {
                $score += 20;
            } elseif ($metaLength >= 120 && $metaLength <= 170) {
                $score += 15;
            } elseif ($metaLength >= 100) {
                $score += 10;
            }
        }
        
        // Content length (optimal: 300+ words)
        $wordCount = str_word_count(strip_tags($berita->konten));
        if ($wordCount >= 500) {
            $score += 20;
        } elseif ($wordCount >= 300) {
            $score += 15;
        } elseif ($wordCount >= 200) {
            $score += 10;
        }
        
        // Image alt text
        if (!empty($berita->alt_text)) {
            $score += 10;
        }
        
        // Keywords/tags
        if (!empty($berita->keywords) || !empty($berita->tags)) {
            $score += 10;
        }
        
        // Image presence
        if (!empty($berita->gambar)) {
            $score += 10;
        }
        
        // Ringkasan presence
        if (!empty($berita->ringkasan)) {
            $score += 10;
        }
        
        return min(100, $score);
    }

    /**
     * Show the form for editing news
     */
    public function edit(Berita $berita)
    {
        $categories = ['Pengumuman', 'Pembangunan', 'Kesehatan', 'Sosial', 'Ekonomi', 'Pendidikan'];
        return view('admin.berita.edit', compact('berita', 'categories'));
    }

    /**
     * Update news
     */
    public function update(Request $request, Berita $berita)
    {
        // Debug: Log incoming request
        Log::info('Berita update request received', [
            'berita_id' => $berita->getKey(),
            'request_data' => $request->all()
        ]);

        $validated = $request->validate([
            'judul' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255',
            'konten' => 'required|string',
            'ringkasan' => 'required|string|max:500',
            'kategori' => 'required|string|max:100',
            'status' => 'required|in:Draft,Terbit,scheduled,archived',
            'prioritas' => 'nullable|string|in:normal,penting,mendesak',
            'tanggal_publish' => 'nullable|date',
            'meta_description' => 'nullable|string|max:160',
            'keywords' => 'nullable|string|max:255',
            'tags' => 'nullable|string|max:255',
            'alt_text' => 'nullable|string|max:255',
            'allow_comments' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',
            'gambar' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'broadcast_wa' => 'nullable|boolean'
        ]);

        // Update slug if title changed or if slug is empty
        if ($validated['judul'] !== $berita->judul || empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['judul']) . '-' . time();
        }

        // Set publish date based on status
        if ($validated['status'] === 'Terbit' && $berita->status !== 'Terbit') {
            $validated['tanggal_terbit'] = now();
        } elseif ($validated['status'] === 'scheduled' && !empty($validated['tanggal_publish'])) {
            $validated['tanggal_terbit'] = $validated['tanggal_publish'];
        }

        // Handle boolean fields
        $validated['allow_comments'] = $request->has('allow_comments') ? 1 : 0;
        $validated['is_featured'] = $request->has('is_featured') ? 1 : 0;

        // Set default prioritas if not provided
        $validated['prioritas'] = $validated['prioritas'] ?? 'normal';

        // Handle image upload
        if ($request->hasFile('gambar')) {
            // Delete old image
            if ($berita->gambar) {
                Storage::delete('public/' . $berita->gambar);
            }
            
            $image = $request->file('gambar');
            $imageName = time() . '_' . Str::slug($validated['judul']) . '.' . $image->getClientOriginalExtension();
            $image->storeAs('berita', $imageName, 'public');
            $validated['gambar'] = 'berita/' . $imageName;
        }

        $berita->update($validated);

        // Debug: Log successful update
        Log::info('Berita updated successfully', [
            'berita_id' => $berita->getKey(),
            'updated_fields' => array_keys($validated)
        ]);

        // Send WhatsApp broadcast if requested and newly published
        if ($request->broadcast_wa && $validated['status'] === 'Terbit') {
            try {
                $this->broadcastBerita($berita);
            } catch (\Exception $e) {
                Log::error('WhatsApp broadcast failed: ' . $e->getMessage());
            }
        }

        return redirect()->route('admin.berita.index')
                        ->with('success', 'Berita berhasil diperbarui!');
    }

    /**
     * Delete news
     */
    public function destroy(Berita $berita)
    {
        // Delete image if exists
        if ($berita->gambar) {
            Storage::delete('public/' . $berita->gambar);
        }

        $berita->delete();

        return redirect()->route('admin.berita.index')
                        ->with('success', 'Berita berhasil dihapus!');
    }

    /**
     * Auto-save draft berita
     */
    public function autoSave(Request $request)
    {
        try {
            $validated = $request->validate([
                'judul' => 'required|string|max:255',
                'konten' => 'nullable|string',
                'ringkasan' => 'nullable|string|max:500',
                'kategori' => 'nullable|string|max:100',
                'action' => 'required|string',
            ]);

            // Only auto-save if we have enough content
            if (strlen($validated['judul']) < 5) {
                return response()->json([
                    'success' => false,
                    'message' => 'Judul terlalu pendek untuk auto-save'
                ]);
            }

            // Generate slug
            $slug = Str::slug($validated['judul']) . '-draft-' . time();

            // Check if there's an existing draft with similar title
            $existingDraft = Berita::where('judul', 'like', '%' . substr($validated['judul'], 0, 50) . '%')
                                  ->where('status', 'draft')
                                  ->where('user_id', auth()->id() ?? 1)
                                  ->first();

            $beritaData = [
                'judul' => $validated['judul'],
                'slug' => $slug,
                'konten' => $validated['konten'] ?? '',
                'ringkasan' => $validated['ringkasan'] ?? '',
                'kategori' => $validated['kategori'] ?? 'pengumuman',
                'status' => 'draft',
                'user_id' => auth()->id() ?? 1,
                'auto_saved_at' => now(),
            ];

            if ($existingDraft) {
                // Update existing draft
                $existingDraft->update($beritaData);
                $berita = $existingDraft;
            } else {
                // Create new draft
                $berita = Berita::create($beritaData);
            }

            return response()->json([
                'success' => true,
                'message' => 'Draft berhasil disimpan otomatis',
                'berita_id' => $berita->getKey(),
                'saved_at' => $berita->auto_saved_at->format('H:i:s')
            ]);

        } catch (\Exception $e) {
            Log::error('Auto-save error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Gagal menyimpan draft: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Publish news
     */
    public function publish(Berita $berita)
    {
        $berita->update([
            'status' => 'Terbit',
            'tanggal_terbit' => now()
        ]);

        return response()->json([
            'message' => 'Berita berhasil dipublikasikan!'
        ]);
    }

    /**
     * Unpublish news
     */
    public function unpublish(Berita $berita)
    {
        $berita->update([
            'status' => 'Draft',
            'tanggal_terbit' => null
        ]);

        return response()->json([
            'message' => 'Berita berhasil di-unpublish!'
        ]);
    }

    /**
     * Broadcast news via WhatsApp
     */
    protected function broadcastBerita(Berita $berita)
    {
        try {
            // Get all residents with WhatsApp numbers
            $penduduk = \App\Models\Penduduk::whereNotNull('no_wa')->get();
            
            $message = "🗞️ *BERITA DESA TERBARU*\n\n";
            $message .= "*{$berita->judul}*\n\n";
            $message .= $berita->ringkasan . "\n\n";
            $message .= "📅 " . $berita->tanggal_terbit->format('d/m/Y H:i') . "\n";
            $message .= "🏷️ Kategori: {$berita->kategori}\n\n";
            $message .= "Baca selengkapnya: " . url('/berita/' . $berita->slug);

            foreach ($penduduk as $warga) {
                $this->whatsappService->sendMessage($warga->no_wa, $message);
            }

            Log::info("Berita broadcast sent for: {$berita->judul}");
        } catch (\Exception $e) {
            Log::error("Failed to broadcast berita: " . $e->getMessage());
        }
    }

    /**
     * Get news statistics
     */
    public function statistics()
    {
        $stats = [
            'total' => Berita::count(),
            'published' => Berita::where('status', 'Terbit')->count(),
            'draft' => Berita::where('status', 'Draft')->count(),
            'total_views' => Berita::sum('views'),
            'categories' => Berita::selectRaw('kategori, COUNT(*) as count')
                                 ->groupBy('kategori')
                                 ->get(),
            'monthly_stats' => Berita::selectRaw('MONTH(created_at) as month, COUNT(*) as count')
                                   ->whereYear('created_at', date('Y'))
                                   ->groupBy('month')
                                   ->get()
        ];

        return response()->json($stats);
    }

    // Public methods for frontend
    
    /**
     * API news listing for AJAX requests
     */
    public function apiIndex(Request $request)
    {
        $query = Berita::whereIn('status', ['published', 'Terbit']);

        // Filter by category
        if ($request->has('filter') && $request->filter !== 'all') {
            $query->where('kategori', $request->filter);
        }

        // Search
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', '%' . $search . '%')
                  ->orWhere('ringkasan', 'like', '%' . $search . '%');
            });
        }

        // Sort
        $sortBy = $request->get('sort', 'terbaru');
        switch($sortBy) {
            case 'terpopuler':
                $query->orderBy('views', 'desc');
                break;
            case 'az':
                $query->orderBy('judul', 'asc');
                break;
            default:
                $query->orderBy('created_at', 'desc');
        }

        $berita = $query->paginate(6);
        
        return response()->json([
            'success' => true,
            'data' => $berita->items(),
            'hasMore' => $berita->hasMorePages(),
            'currentPage' => $berita->currentPage(),
            'totalPages' => $berita->lastPage()
        ]);
    }
    
    /**
     * Public news listing
     */
    public function publicIndex(Request $request)
    {
        $query = Berita::whereIn('status', ['published', 'Terbit']);

        // Filter by category
        if ($request->has('kategori') && $request->kategori) {
            $query->where('kategori', $request->kategori);
        }

        // Search
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('judul', 'like', '%' . $search . '%')
                  ->orWhere('ringkasan', 'like', '%' . $search . '%');
            });
        }

        $beritaList = $query->orderBy('created_at', 'desc')->paginate(12);
        
        // Statistics
        $totalBerita = Berita::whereIn('status', ['published', 'Terbit'])->count();
        $beritaBulanIni = Berita::whereIn('status', ['published', 'Terbit'])
                                ->whereMonth('created_at', now()->month)
                                ->whereYear('created_at', now()->year)
                                ->count();
        $totalViews = Berita::whereIn('status', ['published', 'Terbit'])->sum('views');

        $categories = Berita::whereIn('status', ['published', 'Terbit'])
                           ->distinct('kategori')
                           ->pluck('kategori');

        $featuredNews = Berita::whereIn('status', ['published', 'Terbit'])
                             ->where('is_featured', true)
                             ->orderBy('created_at', 'desc')
                         ->take(3)
                         ->get();

        return view('frontend.berita', compact(
            'beritaList', 
            'categories', 
            'featuredNews',
            'totalBerita',
            'beritaBulanIni', 
            'totalViews'
        ));
    }

    /**
     * Public news detail
     */
    public function publicShow($slug)
    {
        $berita = Berita::where('slug', $slug)
                       ->whereIn('status', ['published', 'Terbit'])
                       ->with(['comments' => function($query) {
                           $query->where('status', 'approved')->orderBy('created_at', 'desc')->limit(6);
                       }])
                       ->firstOrFail();

        // Increment views
        $berita->increment('views');

        // Get related news
        $relatedBerita = Berita::whereIn('status', ['published', 'Terbit'])
                        ->where('kategori', $berita->kategori)
                        ->where('id', '!=', $berita->id)
                        ->orderBy('created_at', 'desc')
                        ->take(4)
                        ->get();

        return view('frontend.detail-berita', compact('berita', 'relatedBerita'));
    }
    
    /**
     * API endpoint to increment view count
     */
    public function incrementView($id)
    {
        try {
            $berita = Berita::findOrFail($id);
            $berita->increment('views');
            
            return response()->json([
                'success' => true,
                'views' => $berita->views
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan'
            ], 404);
        }
    }

    /**
     * API endpoint to toggle like
     */
    public function toggleLike($id)
    {
        try {
            $berita = Berita::findOrFail($id);
            
            // For now, just increment likes count
            // In a real app, you'd track user likes in a separate table
            $berita->incrementLikes();
            
            return response()->json([
                'success' => true,
                'likes_count' => $berita->likes_count
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan'
            ], 404);
        }
    }

    /**
     * API endpoint to increment share count
     */
    public function incrementShare($id)
    {
        try {
            $berita = Berita::findOrFail($id);
            $berita->incrementShares();
            
            return response()->json([
                'success' => true,
                'shares_count' => $berita->shares_count
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan'
            ], 404);
        }
    }

    /**
     * API endpoint to add comment
     */
    public function addComment(Request $request, $id)
    {
        try {
            $berita = Berita::findOrFail($id);
            
            $request->validate([
                'komentar' => 'required|string|max:500',
                'nama' => 'required|string|max:100',
                'phone' => ['required', 'string', 'max:20', 'regex:/^(08|628|\+628)[0-9]{8,11}$/']
            ]);

            $comment = $berita->comments()->create([
                'user_id' => auth()->id(),
                'nama' => $request->nama,
                'phone' => $request->phone,
                'komentar' => $request->komentar,
                'status' => 'approved'
            ]);

            // Update comments count
            $berita->increment('comments_count');
            
            return response()->json([
                'success' => true,
                'comment' => [
                    'id' => $comment->id,
                    'author_name' => $comment->author_name,
                    'komentar' => $comment->komentar,
                    'time_ago' => $comment->time_ago,
                    'likes_count' => $comment->likes_count
                ],
                'comments_count' => $berita->comments_count
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Data tidak valid',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menambahkan komentar'
            ], 500);
        }
    }

    /**
     * API endpoint to get berita images
     */
    public function getImages($id)
    {
        try {
            $berita = Berita::findOrFail($id);
            
            $images = [];
            
            // Add featured image if exists
            if ($berita->gambar) {
                $images[] = [
                    'src' => $berita->gambar, // Just the relative path
                    'alt' => $berita->alt_text ?: $berita->judul,
                    'type' => 'featured'
                ];
            }
            
            // Extract images from content
            if ($berita->konten) {
                try {
                    $dom = new \DOMDocument();
                    libxml_use_internal_errors(true);
                    $dom->loadHTML('<?xml encoding="utf-8" ?>' . $berita->konten);
                    libxml_clear_errors();
                    
                    $imageElements = $dom->getElementsByTagName('img');
                    foreach ($imageElements as $img) {
                        $src = $img->getAttribute('src');
                        if ($src && !empty(trim($src))) {
                            // Clean up the path to be consistent
                            $cleanSrc = $src;
                            
                            if (strpos($src, 'http') === 0) {
                                // Full URL - check if it's our own domain
                                $baseUrl = url('/storage/');
                                if (strpos($src, $baseUrl) === 0) {
                                    // Extract relative path from our own URL
                                    $cleanSrc = str_replace($baseUrl . '/', '', $src);
                                } else {
                                    // External URL - keep as is
                                    $cleanSrc = $src;
                                }
                            } elseif (strpos($src, '/storage/') === 0) {
                                // Remove /storage/ prefix to get relative path
                                $cleanSrc = ltrim($src, '/storage/');
                            } elseif (strpos($src, 'storage/') === 0) {
                                // Already relative path starting with storage/
                                $cleanSrc = $src;
                            }
                            
                            $images[] = [
                                'src' => $cleanSrc,
                                'alt' => $img->getAttribute('alt') ?: 'Content Image',
                                'type' => 'content'
                            ];
                        }
                    }
                } catch (\Exception $e) {
                    // Silently handle DOM parsing errors
                }
            }
            
            return response()->json([
                'success' => true,
                'images' => $images,
                'count' => count($images)
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Berita tidak ditemukan'
            ], 404);
        }
    }

    /**
     * Show comments for a specific news article
     */
    public function showComments(Berita $berita)
    {
        $comments = \App\Models\BeritaComment::where('berita_id', $berita->id)
                          ->orderBy('created_at', 'desc')
                          ->paginate(20);
        
        $stats = [
            'total' => \App\Models\BeritaComment::where('berita_id', $berita->id)->count(),
            'approved' => \App\Models\BeritaComment::where('berita_id', $berita->id)->where('status', 'approved')->count(),
            'pending' => \App\Models\BeritaComment::where('berita_id', $berita->id)->where('status', 'pending')->count(),
            'rejected' => \App\Models\BeritaComment::where('berita_id', $berita->id)->where('status', 'rejected')->count(),
        ];
        
        return view('admin.berita.comments', compact('berita', 'comments', 'stats'));
    }

    /**
     * Approve a comment
     */
    public function approveComment($beritaId, $commentId)
    {
        try {
            $berita = Berita::findOrFail($beritaId);
            $comment = \App\Models\BeritaComment::where('berita_id', $berita->id)->findOrFail($commentId);
            
            $comment->update(['status' => 'approved']);
            $berita->incrementComments();
            
            return response()->json([
                'success' => true,
                'message' => 'Komentar berhasil disetujui'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menyetujui komentar'
            ], 500);
        }
    }

    /**
     * Reject a comment
     */
    public function rejectComment($beritaId, $commentId)
    {
        try {
            $berita = Berita::findOrFail($beritaId);
            $comment = \App\Models\BeritaComment::where('berita_id', $berita->id)->findOrFail($commentId);
            
            $comment->update(['status' => 'rejected']);
            
            return response()->json([
                'success' => true,
                'message' => 'Komentar berhasil ditolak'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menolak komentar'
            ], 500);
        }
    }

    /**
     * Delete a comment
     */
    public function deleteComment($beritaId, $commentId)
    {
        try {
            $berita = Berita::findOrFail($beritaId);
            $comment = \App\Models\BeritaComment::where('berita_id', $berita->id)->findOrFail($commentId);
            
            $comment->delete();
            
            return response()->json([
                'success' => true,
                'message' => 'Komentar berhasil dihapus'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal menghapus komentar'
            ], 500);
        }
    }
} 
