<?php

namespace App\Http\Controllers;

use App\Models\Bug;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class BugController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('admin')->except(['report', 'store']);
    }

    /**
     * Display bug dashboard
     */
    public function index(Request $request)
    {
        $query = Bug::with(['user', 'resolver'])->latest();

        // Apply filters
        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('module')) {
            $query->where('module', $request->module);
        }

        if ($request->filled('search')) {
            $query->where(function($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        $bugs = $query->paginate(20);

        // Get statistics
        $stats = [
            'total' => Bug::count(),
            'open' => Bug::where('status', 'open')->count(),
            'critical' => Bug::where('severity', 'critical')->count(),
            'high' => Bug::where('severity', 'high')->count(),
            'resolved_today' => Bug::where('status', 'resolved')
                                  ->whereDate('resolved_at', today())
                                  ->count()
        ];

        // Get modules with bugs
        $modules = Bug::distinct('module')
                     ->whereNotNull('module')
                     ->pluck('module')
                     ->sort();

        return view('admin.bugs.index', compact('bugs', 'stats', 'modules'));
    }

    /**
     * Show bug details
     */
    public function show(Bug $bug)
    {
        $bug->load(['user', 'resolver']);
        return view('admin.bugs.show', compact('bug'));
    }

    /**
     * Show create bug form
     */
    public function create()
    {
        return view('admin.bugs.create');
    }

    /**
     * Store manually reported bug
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'severity' => 'required|in:low,medium,high,critical',
            'type' => 'required|in:bug,error,warning,info',
            'module' => 'nullable|string|max:100',
            'url' => 'nullable|url',
            'reproduction_steps' => 'nullable|string'
        ]);

        $environmentData = [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'user_agent' => $request->userAgent(),
            'ip_address' => $request->ip(),
            'reported_manually' => true
        ];

        if ($request->filled('reproduction_steps')) {
            $environmentData['reproduction_steps'] = $request->reproduction_steps;
        }

        $bug = Bug::create([
            'title' => $request->title,
            'description' => $request->description,
            'severity' => $request->severity,
            'type' => $request->type,
            'module' => $request->module,
            'url' => $request->url,
            'user_agent' => $request->userAgent(),
            'ip_address' => $request->ip(),
            'user_id' => Auth::id(),
            'environment_data' => $environmentData,
            'last_occurred_at' => now()
        ]);

        // Log the manual bug report
        Log::info('Manual bug report created', [
            'bug_id' => $bug->id,
            'title' => $bug->title,
            'user_id' => Auth::id()
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Bug report berhasil disimpan',
                'bug' => $bug
            ]);
        }

        return redirect()->route('admin.bugs.show', $bug)
                        ->with('success', 'Bug report berhasil dibuat');
    }

    /**
     * Update bug status
     */
    public function update(Request $request, Bug $bug)
    {
        $request->validate([
            'status' => 'required|in:open,in_progress,resolved,closed',
            'resolution_notes' => 'nullable|string'
        ]);

        $oldStatus = $bug->status;

        $updateData = [
            'status' => $request->status
        ];

        if ($request->status === 'resolved' && $oldStatus !== 'resolved') {
            $updateData['resolved_at'] = now();
            $updateData['resolved_by'] = Auth::id();
        }

        if ($request->filled('resolution_notes')) {
            $updateData['resolution_notes'] = $request->resolution_notes;
        }

        $bug->update($updateData);

        // Log status change
        Log::info('Bug status updated', [
            'bug_id' => $bug->id,
            'old_status' => $oldStatus,
            'new_status' => $request->status,
            'updated_by' => Auth::id()
        ]);

        if ($request->expectsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Status bug berhasil diperbarui',
                'bug' => $bug->fresh()
            ]);
        }

        return redirect()->back()
                        ->with('success', 'Status bug berhasil diperbarui');
    }

    /**
     * Delete bug
     */
    public function destroy(Bug $bug)
    {
        $bug->delete();

        Log::info('Bug deleted', [
            'bug_id' => $bug->id,
            'title' => $bug->title,
            'deleted_by' => Auth::id()
        ]);

        return redirect()->route('admin.bugs.index')
                        ->with('success', 'Bug berhasil dihapus');
    }

    /**
     * Get dashboard statistics
     */
    public function dashboard()
    {
        $stats = Bug::getStatistics('week');
        
        // Recent critical bugs
        $criticalBugs = Bug::critical()
                          ->open()
                          ->with('user')
                          ->latest()
                          ->limit(5)
                          ->get();

        // Bugs by day (last 7 days)
        $bugsByDay = [];
        for ($i = 6; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $bugsByDay[$date->format('M j')] = Bug::whereDate('created_at', $date)->count();
        }

        // Top modules with bugs
        $topModules = Bug::selectRaw('module, COUNT(*) as bug_count')
                        ->whereNotNull('module')
                        ->where('created_at', '>=', now()->subMonth())
                        ->groupBy('module')
                        ->orderByDesc('bug_count')
                        ->limit(5)
                        ->get();

        return view('admin.bugs.dashboard', compact(
            'stats',
            'criticalBugs', 
            'bugsByDay',
            'topModules'
        ));
    }

    /**
     * Auto-capture error from system
     */
    public static function captureError(\Throwable $exception, $request = null, $module = null)
    {
        try {
            // Don't capture in testing environment
            if (app()->environment('testing')) {
                return null;
            }

            // Skip certain types of errors in production
            if (app()->environment('production')) {
                $skipErrors = [
                    'Illuminate\Http\Exceptions\ThrottleRequestsException',
                    'Symfony\Component\HttpKernel\Exception\NotFoundHttpException',
                    'Illuminate\Auth\AuthenticationException'
                ];

                if (in_array(get_class($exception), $skipErrors)) {
                    return null;
                }
            }

            // Extract module from stack trace if not provided
            if (!$module && $request) {
                $route = $request->route();
                if ($route && $route->getController()) {
                    $controllerClass = get_class($route->getController());
                    $module = class_basename($controllerClass);
                }
            }

            $bug = Bug::createFromException($exception, $request, $module);

            // Log the auto-captured error
            Log::error('Auto-captured bug', [
                'bug_id' => $bug->id,
                'exception' => get_class($exception),
                'message' => $exception->getMessage(),
                'module' => $module
            ]);

            return $bug;

        } catch (\Exception $e) {
            // Don't let bug capture system cause more bugs
            Log::error('Error in bug capture system', [
                'error' => $e->getMessage(),
                'original_exception' => $exception->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Mark multiple bugs as resolved
     */
    public function bulkResolve(Request $request)
    {
        $request->validate([
            'bug_ids' => 'required|array',
            'bug_ids.*' => 'exists:bugs,id',
            'resolution_notes' => 'nullable|string'
        ]);

        $updated = Bug::whereIn('id', $request->bug_ids)
                     ->where('status', '!=', 'resolved')
                     ->update([
                         'status' => 'resolved',
                         'resolved_at' => now(),
                         'resolved_by' => Auth::id(),
                         'resolution_notes' => $request->resolution_notes
                     ]);

        Log::info('Bulk bug resolution', [
            'count' => $updated,
            'bug_ids' => $request->bug_ids,
            'resolved_by' => Auth::id()
        ]);

        return response()->json([
            'success' => true,
            'message' => "Berhasil menyelesaikan {$updated} bug",
            'updated' => $updated
        ]);
    }

    /**
     * Export bugs to CSV
     */
    public function export(Request $request)
    {
        $query = Bug::with(['user', 'resolver']);

        // Apply same filters as index
        if ($request->filled('severity')) {
            $query->where('severity', $request->severity);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('module')) {
            $query->where('module', $request->module);
        }

        $bugs = $query->get();

        $filename = 'bugs_export_' . now()->format('Y_m_d_His') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($bugs) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'ID', 'Title', 'Description', 'Severity', 'Status', 
                'Type', 'Module', 'URL', 'User', 'Created At', 
                'Resolved At', 'Resolved By', 'Occurrence Count'
            ]);

            foreach ($bugs as $bug) {
                fputcsv($file, [
                    $bug->id,
                    $bug->title,
                    $bug->description,
                    $bug->severity,
                    $bug->status,
                    $bug->type,
                    $bug->module,
                    $bug->url,
                    $bug->user ? $bug->user->name : 'N/A',
                    $bug->created_at->format('Y-m-d H:i:s'),
                    $bug->resolved_at ? $bug->resolved_at->format('Y-m-d H:i:s') : '',
                    $bug->resolver ? $bug->resolver->name : '',
                    $bug->occurrence_count
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Public bug report form (for users)
     */
    public function report()
    {
        return view('bugs.report');
    }

    /**
     * Get bug statistics for API
     */
    public function statistics(Request $request)
    {
        $period = $request->get('period', 'week');
        $stats = Bug::getStatistics($period);

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }
} 