<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\ProdukUmkm;
use App\Services\TripayService;
use App\Services\PaymentMethodSyncService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class CheckoutController extends Controller
{
    protected $tripayService;
    protected $paymentMethodSyncService;

    public function __construct(TripayService $tripayService, PaymentMethodSyncService $paymentMethodSyncService)
    {
        $this->tripayService = $tripayService;
        $this->paymentMethodSyncService = $paymentMethodSyncService;
    }

    /**
     * Show checkout page
     */
    public function index(Request $request)
    {
        // Get cart items from session or request
        $cartItems = $request->session()->get('cart', []);
        
        if (empty($cartItems)) {
            return redirect()->route('produk-umkm')->with('error', 'Keranjang kosong');
        }

        // Calculate totals
        $subtotal = 0;
        $items = [];
        
        foreach ($cartItems as $item) {
            $produk = ProdukUmkm::find($item['produk_id']);
            if ($produk) {
                $itemTotal = $produk->harga_akhir * $item['quantity'];
                $subtotal += $itemTotal;
                
                $items[] = [
                    'produk' => $produk,
                    'quantity' => $item['quantity'],
                    'subtotal' => $itemTotal
                ];
            }
        }

        $shippingCost = 10000; // Fixed shipping cost
        $total = $subtotal + $shippingCost;

        // Get payment methods from sync service
        $paymentMethods = $this->paymentMethodSyncService->getUmkmPaymentMethods();

        return view('frontend.checkout', compact('items', 'subtotal', 'shippingCost', 'total', 'paymentMethods'));
    }

    /**
     * Process checkout
     */
    public function process(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'customer_phone' => 'required|string|max:20',
            'customer_address' => 'required|string|max:500',
            'payment_method' => 'required|string',
            'notes' => 'nullable|string|max:500'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        try {
            DB::beginTransaction();

            // Get cart items
            $cartItems = $request->session()->get('cart', []);
            if (empty($cartItems)) {
                throw new \Exception('Keranjang kosong');
            }

            // Calculate order totals
            $subtotal = 0;
            $orderItems = [];
            
            foreach ($cartItems as $item) {
                $produk = ProdukUmkm::find($item['produk_id']);
                if (!$produk) {
                    throw new \Exception('Produk tidak ditemukan');
                }

                if ($produk->stok < $item['quantity']) {
                    throw new \Exception("Stok {$produk->nama_produk} tidak mencukupi");
                }

                $itemSubtotal = $produk->harga_akhir * $item['quantity'];
                $subtotal += $itemSubtotal;

                $orderItems[] = [
                    'produk_umkm_id' => $produk->id,
                    'product_name' => $produk->nama_produk,
                    'product_price' => $produk->harga_akhir,
                    'quantity' => $item['quantity'],
                    'subtotal' => $itemSubtotal
                ];
            }

            $shippingCost = 10000;
            $totalAmount = $subtotal + $shippingCost;

            // Get payment fee
            $feeAmount = $this->paymentMethodSyncService->calculatePaymentFee($totalAmount, $request->payment_method);
            $finalAmount = $totalAmount + $feeAmount;

            // Create order
            $order = Order::create([
                'order_id' => 'ORD-' . time() . '-' . rand(1000, 9999),
                'customer_name' => $request->customer_name,
                'customer_email' => $request->customer_email,
                'customer_phone' => $request->customer_phone,
                'customer_address' => $request->customer_address,
                'total_amount' => $totalAmount,
                'fee_amount' => $feeAmount,
                'final_amount' => $finalAmount,
                'shipping_cost' => $shippingCost,
                'payment_method' => $request->payment_method,
                'status' => Order::STATUS_PENDING,
                'notes' => $request->notes
            ]);

            // Create order items
            foreach ($orderItems as $item) {
                $item['order_id'] = $order->id;
                OrderItem::create($item);
            }

            // Update product stock
            foreach ($cartItems as $item) {
                $produk = ProdukUmkm::find($item['produk_id']);
                $produk->decrement('stok', $item['quantity']);
            }

            // Create Tripay transaction
            if (!empty($this->tripayService)) {
                $tripayData = [
                    'payment_method' => $request->payment_method,
                    'amount' => $finalAmount,
                    'customer_name' => $request->customer_name,
                    'customer_email' => $request->customer_email,
                    'customer_phone' => $request->customer_phone,
                    'order_items' => array_map(function($item) {
                        return [
                            'name' => $item['product_name'],
                            'price' => $item['product_price'],
                            'quantity' => $item['quantity']
                        ];
                    }, $orderItems)
                ];

                $tripayResult = $this->tripayService->createTransaction($tripayData);
                
                if ($tripayResult) {
                    $order->update([
                        'tripay_reference' => $tripayResult['reference'],
                        'tripay_merchant_ref' => $tripayResult['merchant_ref'],
                        'payment_channel' => $tripayResult['payment_name'],
                        'expired_at' => now()->addSeconds($tripayResult['expired_time'] - time())
                    ]);

                    // Clear cart
                    $request->session()->forget('cart');

                    DB::commit();

                    return redirect()->route('checkout.payment', $order->id);
                } else {
                    throw new \Exception('Gagal membuat transaksi pembayaran');
                }
            } else {
                // Fallback for when Tripay is not configured
                DB::commit();
                
                // Clear cart
                $request->session()->forget('cart');
                
                return redirect()->route('checkout.success')->with('order_id', $order->id);
            }

        } catch (\Exception $e) {
            DB::rollback();
            return back()->withErrors(['error' => $e->getMessage()])->withInput();
        }
    }

    /**
     * Show payment page
     */
    public function payment($orderId)
    {
        $order = Order::with('orderItems.produk')->findOrFail($orderId);
        
        if ($order->isPaid()) {
            return redirect()->route('checkout.success')->with('order_id', $order->id);
        }

        // Get payment instruction from Tripay
        $paymentData = null;
        if ($order->tripay_reference && !empty($this->tripayService)) {
            $paymentResult = $this->tripayService->getTransaction($order->tripay_reference);
            $paymentData = $paymentResult['success'] ? $paymentResult['data'] : null;
        }

        return view('frontend.payment', compact('order', 'paymentData'));
    }

    /**
     * Show success page
     */
    public function success()
    {
        $orderId = session('order_id');
        $order = $orderId ? Order::find($orderId) : null;
        
        return view('frontend.checkout-success', compact('order'));
    }

    /**
     * Add to cart
     */
    public function addToCart(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'produk_id' => 'required|exists:produk_umkm,id',
            'quantity' => 'required|integer|min:1'
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => 'Data tidak valid']);
        }

        $produk = ProdukUmkm::find($request->produk_id);
        
        if (!$produk || $produk->stok < $request->quantity) {
            return response()->json(['success' => false, 'message' => 'Stok tidak mencukupi']);
        }

        // Get current cart
        $cart = $request->session()->get('cart', []);
        
        // Check if product already in cart
        $found = false;
        foreach ($cart as &$item) {
            if ($item['produk_id'] == $request->produk_id) {
                $item['quantity'] += $request->quantity;
                $found = true;
                break;
            }
        }

        // Add new item if not found
        if (!$found) {
            $cart[] = [
                'produk_id' => $request->produk_id,
                'quantity' => $request->quantity
            ];
        }

        $request->session()->put('cart', $cart);

        return response()->json([
            'success' => true, 
            'message' => 'Produk berhasil ditambahkan ke keranjang',
            'cart_count' => count($cart)
        ]);
    }
} 