<?php

namespace App\Http\Controllers;

use App\Models\Penduduk;
use App\Models\Keluarga;
use App\Models\AnggotaKeluarga;
use App\Models\WilayahAdministratif;
use App\Services\FileUploadService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class PendudukController extends Controller
{
    public function index(Request $request)
    {
        try {
            // Simplified query without complex joins
            $query = Penduduk::query();

            // Apply wilayah kerja scope based on user role
            $user = \Illuminate\Support\Facades\Auth::user();
            
            // Debug: log user info
            \Illuminate\Support\Facades\Log::info('User info for penduduk query', [
                'user_id' => $user->id,
                'user_role' => $user->role,
                'user_dusun' => $user->dusun,
                'user_rt' => $user->rt,
                'user_rw' => $user->rw
            ]);
            
            // Apply wilayah kerja scope
            $query->byWilayahKerja($user);

            // Filter berdasarkan pencarian
            if ($request->search) {
                $query->where(function($q) use ($request) {
                    $q->where('nama', 'like', '%' . $request->search . '%')
                      ->orWhere('nik', 'like', '%' . $request->search . '%')
                      ->orWhere('no_kk', 'like', '%' . $request->search . '%');
                });
            }

            // Filter berdasarkan dusun (hanya jika user bisa akses)
            if ($request->dusun && $user->dusun === $request->dusun) {
                $query->where('dusun', $request->dusun);
            }

            // Filter berdasarkan RT/RW (hanya jika user bisa akses)
            if ($request->rt && $user->rt === $request->rt) {
                $query->where('rt', $request->rt);
            }
            if ($request->rw && $user->rw === $request->rw) {
                $query->where('rw', $request->rw);
            }

            // Filter berdasarkan status portal
            if ($request->portal_status) {
                switch ($request->portal_status) {
                    case 'aktif':
                        $query->whereNotNull('password')->where('password_must_change', false);
                        break;
                    case 'harus_ganti':
                        $query->whereNotNull('password')->where('password_must_change', true);
                        break;
                    case 'belum_diatur':
                        $query->whereNull('password');
                        break;
                }
            }

            // Get total count before pagination for debugging
            $totalCount = $query->count();
            \Illuminate\Support\Facades\Log::info('Penduduk query result', [
                'total_count' => $totalCount,
                'user_role' => $user->role
            ]);
            
            // If no data found, try without wilayah scope for debugging
            if ($totalCount === 0) {
                \Illuminate\Support\Facades\Log::warning('No data found with wilayah scope, trying without scope');
                $fallbackQuery = Penduduk::query();
                $fallbackCount = $fallbackQuery->count();
                \Illuminate\Support\Facades\Log::info('Fallback query result', ['total_count' => $fallbackCount]);
                
                // Use fallback query if it has data
                if ($fallbackCount > 0) {
                    $query = $fallbackQuery;
                }
            }
            
            $penduduk = $query->paginate(20);
            
            // Data untuk filter dengan error handling
            try {
                $dusunList = Penduduk::distinct()->whereNotNull('dusun')->pluck('dusun')->filter();
            } catch (\Exception $e) {
                $dusunList = collect();
            }
            
            try {
                $rtList = Penduduk::distinct()->whereNotNull('rt')->pluck('rt')->filter();
            } catch (\Exception $e) {
                $rtList = collect();
            }
            
            try {
                $rwList = Penduduk::distinct()->whereNotNull('rw')->pluck('rw')->filter();
            } catch (\Exception $e) {
                $rwList = collect();
            }

            // Statistik data penduduk dengan error handling
            $stats = [
                'total' => 0,
                'laki_laki' => 0,
                'perempuan' => 0,
                'anak' => 0,
            ];
            
            try {
                $stats['total'] = Penduduk::count();
            } catch (\Exception $e) {
                Log::error('Error counting total penduduk: ' . $e->getMessage());
            }
            
            try {
                $stats['laki_laki'] = Penduduk::where('jenis_kelamin', 'L')->count();
            } catch (\Exception $e) {
                Log::error('Error counting laki-laki: ' . $e->getMessage());
            }
            
            try {
                $stats['perempuan'] = Penduduk::where('jenis_kelamin', 'P')->count();
            } catch (\Exception $e) {
                Log::error('Error counting perempuan: ' . $e->getMessage());
            }
            
            try {
                $stats['anak'] = Penduduk::whereNotNull('tanggal_lahir')
                    ->where('tanggal_lahir', '>', now()->subYears(18))
                    ->count();
            } catch (\Exception $e) {
                Log::error('Error counting anak: ' . $e->getMessage());
            }
            
            // Password statistics
            try {
                $stats['password_aktif'] = Penduduk::whereNotNull('password')
                    ->where('password_must_change', false)
                    ->count();
            } catch (\Exception $e) {
                Log::error('Error counting password aktif: ' . $e->getMessage());
                $stats['password_aktif'] = 0;
            }
            
            try {
                $stats['password_harus_ganti'] = Penduduk::whereNotNull('password')
                    ->where('password_must_change', true)
                    ->count();
            } catch (\Exception $e) {
                Log::error('Error counting password harus ganti: ' . $e->getMessage());
                $stats['password_harus_ganti'] = 0;
            }
            
            try {
                $stats['password_belum_diatur'] = Penduduk::whereNull('password')->count();
            } catch (\Exception $e) {
                Log::error('Error counting password belum diatur: ' . $e->getMessage());
                $stats['password_belum_diatur'] = 0;
            }
            
            try {
                $stats['login_hari_ini'] = Penduduk::whereDate('last_login_at', today())->count();
            } catch (\Exception $e) {
                Log::error('Error counting login hari ini: ' . $e->getMessage());
                $stats['login_hari_ini'] = 0;
            }

            return view('admin.penduduk.index', compact('penduduk', 'dusunList', 'rtList', 'rwList', 'stats'));
        } catch (\Exception $e) {
            // Log the error
            Log::error('Error loading penduduk index: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            // Provide fallback data
            $penduduk = new \Illuminate\Pagination\LengthAwarePaginator([], 0, 20);
            $dusunList = collect();
            $rtList = collect();
            $rwList = collect();
            $stats = [
                'total' => 0,
                'laki_laki' => 0,
                'perempuan' => 0,
                'anak' => 0,
                'password_aktif' => 0,
                'password_harus_ganti' => 0,
                'password_belum_diatur' => 0,
                'login_hari_ini' => 0,
            ];

            return view('admin.penduduk.index', compact('penduduk', 'dusunList', 'rtList', 'rwList', 'stats'))
                ->with('error', 'Terjadi kesalahan saat memuat data penduduk: ' . $e->getMessage());
        }
    }

    public function indexTest(Request $request)
    {
        try {
            $query = Penduduk::with(['keluarga', 'anggotaKeluarga']);

            // Simple pagination for test
            $penduduk = $query->take(10)->get();
            
            // Statistik data penduduk dengan error handling
            $stats = [
                'total' => Penduduk::count() ?? 0,
                'laki_laki' => Penduduk::where('jenis_kelamin', 'L')->count() ?? 0,
                'perempuan' => Penduduk::where('jenis_kelamin', 'P')->count() ?? 0,
                'anak' => Penduduk::whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) < 18')->count() ?? 0,
            ];

            return view('admin.penduduk.index-simple', compact('penduduk', 'stats'));
        } catch (\Exception $e) {
            // Log the error
            Log::error('Error loading penduduk test index: ' . $e->getMessage());
            
            // Provide fallback data
            $penduduk = collect();
            $stats = [
                'total' => 0,
                'laki_laki' => 0,
                'perempuan' => 0,
                'anak' => 0,
            ];

            return view('admin.penduduk.index-simple', compact('penduduk', 'stats'))
                ->with('error', 'Terjadi kesalahan saat memuat data penduduk. Silakan coba lagi.');
        }
    }

    public function create()
    {
        // Get all families for the dropdown with proper error handling
        try {
            $keluarga = \App\Models\Keluarga::with(['kepalaKeluarga' => function($query) {
                            $query->select('id', 'nik', 'nama', 'foto');
                        }])
                        ->select('id', 'no_kk', 'kepala_keluarga_nik', 'alamat', 'dusun', 'rt', 'rw')
                        ->orderBy('no_kk')
                        ->get();
        } catch (\Exception $e) {
            // Fallback to empty collection if error occurs
            Log::error('Error loading keluarga data in create: ' . $e->getMessage());
            $keluarga = collect();
        }

        // Get wilayah administratif data from wilayah_administratif table
        try {
            $dusunList = \App\Models\WilayahAdministratif::where('jenis_wilayah', 'dusun')
                ->where('status', 'aktif')
                ->orderBy('nama_wilayah')
                ->pluck('nama_wilayah');
            
            $rwList = \App\Models\WilayahAdministratif::where('jenis_wilayah', 'rw')
                ->where('status', 'aktif')
                ->orderBy('nama_wilayah')
                ->pluck('nama_wilayah');
            
            $rtList = \App\Models\WilayahAdministratif::where('jenis_wilayah', 'rt')
                ->where('status', 'aktif')
                ->orderBy('nama_wilayah')
                ->pluck('nama_wilayah');
                
            Log::info('Wilayah data loaded successfully', [
                'dusun_count' => $dusunList->count(),
                'rw_count' => $rwList->count(),
                'rt_count' => $rtList->count()
            ]);
        } catch (\Exception $e) {
            Log::error('Error loading wilayah data in create: ' . $e->getMessage());
            // Fallback to penduduk table if wilayah_administratif table doesn't exist
            try {
                $dusunList = \App\Models\Penduduk::distinct()->whereNotNull('dusun')->pluck('dusun')->filter()->sort();
                $rwList = \App\Models\Penduduk::distinct()->whereNotNull('rw')->pluck('rw')->filter()->sort();
                $rtList = \App\Models\Penduduk::distinct()->whereNotNull('rt')->pluck('rt')->filter()->sort();
                Log::info('Fallback to penduduk table successful');
            } catch (\Exception $fallbackError) {
                Log::error('Fallback also failed: ' . $fallbackError->getMessage());
                $dusunList = collect();
                $rwList = collect();
                $rtList = collect();
            }
        }
        
        // Get village settings
        $setting = \App\Models\Setting::getSettings();
                        
        return view('admin.penduduk.create', compact('keluarga', 'dusunList', 'rwList', 'rtList', 'setting'));
    }

    public function store(Request $request)
    {
        // Log request for debugging
        Log::info('Penduduk store request', [
            'user_id' => \Illuminate\Support\Facades\Auth::id(),
            'session_id' => $request->session()->getId(),
            'csrf_token' => $request->input('_token'),
            'has_file' => $request->hasFile('foto')
        ]);
        
        // Validate with Dukcapil if enabled
        if (config('services.dukcapil.api_key')) {
            $dukcapilService = app(\App\Services\DukcapilService::class);
            $nikVerification = $dukcapilService->verifyNik($request->nik);
            
            if (!$nikVerification['success']) {
                return redirect()->back()
                    ->withErrors(['nik' => 'NIK tidak valid atau tidak terdaftar di Dukcapil: ' . $nikVerification['message']])
                    ->withInput();
            }
            
            // Auto-fill data from Dukcapil
            if ($nikVerification['data']) {
                $dukcapilData = $nikVerification['data'];
                $request->merge([
                    'nama' => $dukcapilData['nama'] ?? $request->nama,
                    'tempat_lahir' => $dukcapilData['tempat_lahir'] ?? $request->tempat_lahir,
                    'tanggal_lahir' => $dukcapilData['tanggal_lahir'] ?? $request->tanggal_lahir,
                    'jenis_kelamin' => $dukcapilData['jenis_kelamin'] ?? $request->jenis_kelamin,
                    'agama' => $dukcapilData['agama'] ?? $request->agama,
                ]);
            }
        }

        $validated = $request->validate([
            'nik' => 'required|string|size:16|unique:penduduk',
            'nama' => 'required|string|max:255',
            'no_kk' => 'nullable|string|size:16',
            'keluarga_id' => 'nullable|exists:keluarga,id',
            'is_kepala_keluarga' => 'nullable|boolean',
            'status_keluarga' => 'nullable|string|max:255',
            'hubungan_keluarga' => 'nullable|string|max:255',
            
            // Data orang tua - NIK dan nama
            'ayah_nik' => 'nullable|string|size:16',
            'nama_ayah' => 'nullable|string|max:255',
            'ibu_nik' => 'nullable|string|size:16',
            'nama_ibu' => 'nullable|string|max:255',
            
            'alamat' => 'required|string',
            'dusun' => 'nullable|string',
            'nama_desa' => 'nullable|string',
            'rt' => 'nullable|string|max:5',
            'rw' => 'nullable|string|max:5',
            'tempat_lahir' => 'required|string',
            'tanggal_lahir' => 'required|date',
            
            // Data akta lahir
            'no_akta_lahir' => 'nullable|string|max:50',
            'tanggal_akta_lahir' => 'nullable|date',
            'tempat_akta_lahir' => 'nullable|string|max:255',
            
            'jenis_kelamin' => 'required|in:L,P',
            'status_perkawinan' => 'required|in:Belum Kawin,Kawin,Cerai Hidup,Cerai Mati',
            'agama' => 'required|string',
            'pekerjaan' => 'nullable|string',
            'pendidikan_terakhir' => 'nullable|in:Tidak Sekolah,SD,SMP,SMA,D1,D2,D3,S1,S2,S3',
            'golongan_darah' => 'nullable|in:A,B,AB,O',
            
            'kewarganegaraan' => 'required|in:WNI,WNA',
            'no_wa' => 'nullable|string|max:15',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'status_hidup' => 'nullable|in:Hidup,Meninggal'
        ]);

        // Handle file upload
        if ($request->hasFile('foto')) {
            $uploadService = new FileUploadService();
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'penduduk');
            
            if ($uploadResult['success']) {
                $validated['foto'] = $uploadResult['path'];
            } else {
                return redirect()->back()
                    ->withErrors(['foto' => 'Gagal mengupload foto: ' . $uploadResult['error']])
                    ->withInput();
            }
        }

        // Auto-fill wilayah dari koordinat jika ada
        if ($validated['latitude'] && $validated['longitude'] && !$validated['dusun']) {
            $location = $this->reverseGeocode($validated['latitude'], $validated['longitude']);
            if ($location) {
                $validated['dusun'] = $location['dusun'] ?? null;
                $validated['rt'] = $location['rt'] ?? null;
                $validated['rw'] = $location['rw'] ?? null;
            }
        }

        // Map status_perkawinan to status_kawin for database compatibility
        if (isset($validated['status_perkawinan'])) {
            $validated['status_kawin'] = $validated['status_perkawinan'];
            unset($validated['status_perkawinan']);
        }

        // Map pendidikan_terakhir to pendidikan for database compatibility
        if (isset($validated['pendidikan_terakhir'])) {
            $validated['pendidikan'] = $validated['pendidikan_terakhir'];
        } else {
            $validated['pendidikan'] = 'Tidak Sekolah'; // Default value
        }

        // Auto-fill nama_desa from setting if not provided
        if (empty($validated['nama_desa'])) {
            $setting = \App\Models\Setting::getSettings();
            $validated['nama_desa'] = $setting->nama_desa ?? 'Desa';
        }

        // Validate ayah_nik and ibu_nik exist in database before saving
        if (!empty($validated['ayah_nik'])) {
            $ayahExists = Penduduk::where('nik', $validated['ayah_nik'])->exists();
            if (!$ayahExists) {
                Log::warning('Ayah NIK not found in database (admin)', ['ayah_nik' => $validated['ayah_nik']]);
                $validated['ayah_nik'] = null; // Set to null if not found
            }
        }
        
        if (!empty($validated['ibu_nik'])) {
            $ibuExists = Penduduk::where('nik', $validated['ibu_nik'])->exists();
            if (!$ibuExists) {
                Log::warning('Ibu NIK not found in database (admin)', ['ibu_nik' => $validated['ibu_nik']]);
                $validated['ibu_nik'] = null; // Set to null if not found
            }
        }

        try {
            $penduduk = Penduduk::create($validated);

            // Auto-mapping nama orang tua ke NIK jika belum ada NIK
            if (!empty($validated['nama_ayah']) && empty($validated['ayah_nik'])) {
                $penduduk->mapNamaAyahToNik();
            }
            if (!empty($validated['nama_ibu']) && empty($validated['ibu_nik'])) {
                $penduduk->mapNamaIbuToNik();
            }

            // Buat atau update data keluarga
            $this->handleKeluargaData($penduduk);

            Log::info('Penduduk created successfully', [
                'penduduk_id' => $penduduk->id,
                'nik' => $penduduk->nik,
                'nama' => $penduduk->nama
            ]);

            return redirect()->route('admin.penduduk.index')
                            ->with('success', 'Data penduduk berhasil ditambahkan');
                            
        } catch (\Exception $e) {
            Log::error('Error creating penduduk', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'validated_data' => $validated
            ]);
            
            return redirect()->back()
                            ->withErrors(['error' => 'Terjadi kesalahan saat menyimpan data: ' . $e->getMessage()])
                            ->withInput();
        }
    }

    public function show(Penduduk $penduduk)
    {
        $penduduk->load(['keluarga', 'anggotaKeluarga', 'pelayananSurat', 'pengaduan', 'bantuan']);
        
        // Check if this is an AJAX request for JSON response
        if (request()->wantsJson() || request()->ajax()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $penduduk->id,
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'no_kk' => $penduduk->no_kk,
                    'jenis_kelamin' => $penduduk->jenis_kelamin,
                    'tempat_lahir' => $penduduk->tempat_lahir,
                    'tanggal_lahir' => $penduduk->tanggal_lahir->format('d/m/Y'),
                    'alamat' => $penduduk->alamat,
                    'rt' => $penduduk->rt,
                    'rw' => $penduduk->rw,
                    'dusun' => $penduduk->dusun,
                    'agama' => $penduduk->agama,
                    'status_kawin' => $penduduk->status_kawin,
                    'pekerjaan' => $penduduk->pekerjaan,
                    'pendidikan_terakhir' => $penduduk->pendidikan_terakhir,
                    'golongan_darah' => $penduduk->golongan_darah,
                    'status_hidup' => $penduduk->status_hidup,
                    'kewarganegaraan' => $penduduk->kewarganegaraan,
                    'no_wa' => $penduduk->no_wa,
                    'foto' => \App\Helpers\ImageHelper::getAvatarUrl($penduduk->foto, $penduduk->nama),
                    'keluarga' => $penduduk->keluarga ? [
                        'no_kk' => $penduduk->keluarga->no_kk,
                        'kepala_keluarga' => $penduduk->keluarga->kepala_keluarga_nik
                    ] : null
                ]
            ]);
        }
        
        return view('admin.penduduk.show', compact('penduduk'));
    }

    public function edit(Penduduk $penduduk)
    {
        // Get all families for the dropdown with proper error handling
        $keluarga = \App\Models\Keluarga::with(['kepalaKeluarga' => function($query) {
                        $query->select('id', 'nik', 'nama', 'foto');
                    }])
                    ->orderBy('no_kk')
                    ->get();
        
        // Get village settings
        $setting = \App\Models\Setting::getSettings();
                    
        return view('admin.penduduk.edit', compact('penduduk', 'keluarga', 'setting'));
    }

    public function update(Request $request, Penduduk $penduduk)
    {
        // Debug logging
        Log::info('Penduduk update started', [
            'penduduk_id' => $penduduk->id,
            'has_file' => $request->hasFile('foto'),
            'file_info' => $request->hasFile('foto') ? [
                'name' => $request->file('foto')->getClientOriginalName(),
                'size' => $request->file('foto')->getSize(),
                'mime' => $request->file('foto')->getMimeType()
            ] : null,
            'remove_foto' => $request->has('remove_foto')
        ]);

        $validated = $request->validate([
            'nik' => 'required|string|size:16|unique:penduduk,nik,' . $penduduk->id,
            'nama' => 'required|string|max:255',
            'no_kk' => 'nullable|string|size:16',
            'keluarga_id' => 'nullable|exists:keluarga,id',
            'is_kepala_keluarga' => 'nullable|boolean',
            'status_keluarga' => 'nullable|string|max:255',
            'hubungan_keluarga' => 'nullable|string|max:255',
            
            // Data orang tua - NIK dan nama
            'ayah_nik' => 'nullable|string|size:16',
            'nama_ayah' => 'nullable|string|max:255',
            'ibu_nik' => 'nullable|string|size:16',
            'nama_ibu' => 'nullable|string|max:255',
            
            'alamat' => 'required|string',
            'dusun' => 'nullable|string',
            'nama_desa' => 'nullable|string',
            'rt' => 'nullable|string|max:5',
            'rw' => 'nullable|string|max:5',
            'tempat_lahir' => 'required|string',
            'tanggal_lahir' => 'required|date',
            
            // Data akta lahir
            'no_akta_lahir' => 'nullable|string|max:50',
            'tanggal_akta_lahir' => 'nullable|date',
            'tempat_akta_lahir' => 'nullable|string|max:255',
            
            'jenis_kelamin' => 'required|in:L,P',
            'status_perkawinan' => 'required|in:Belum Kawin,Kawin,Cerai Hidup,Cerai Mati',
            'agama' => 'required|string',
            'pekerjaan' => 'nullable|string',
            'pendidikan_terakhir' => 'nullable|in:Tidak Sekolah,SD,SMP,SMA,D1,D2,D3,S1,S2,S3',
            'golongan_darah' => 'nullable|in:A,B,AB,O',
            
            'kewarganegaraan' => 'required|in:WNI,WNA',
            'no_wa' => 'nullable|string|max:15',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'status_hidup' => 'nullable|in:Hidup,Meninggal'
        ]);

        // Handle foto removal
        if ($request->has('remove_foto') && $request->remove_foto) {
            Log::info('Removing foto for penduduk', ['penduduk_id' => $penduduk->id, 'old_foto' => $penduduk->foto]);
            
            if ($penduduk->foto && Storage::disk('public')->exists($penduduk->foto)) {
                Storage::disk('public')->delete($penduduk->foto);
                Log::info('Old foto deleted successfully');
            }
            $validated['foto'] = null;
        }
        // Handle file upload
        elseif ($request->hasFile('foto')) {
            Log::info('Processing foto upload', [
                'penduduk_id' => $penduduk->id,
                'old_foto' => $penduduk->foto,
                'new_file' => $request->file('foto')->getClientOriginalName()
            ]);
            
            // Delete old photo if exists
            if ($penduduk->foto && Storage::disk('public')->exists($penduduk->foto)) {
                Storage::disk('public')->delete($penduduk->foto);
                Log::info('Old foto deleted before upload');
            }
            
            $uploadService = new FileUploadService();
            $uploadResult = $uploadService->uploadPhotoWithResize($request->file('foto'), 'penduduk');
            
            Log::info('Upload result', $uploadResult);
            
            if ($uploadResult['success']) {
                $validated['foto'] = $uploadResult['path'];
                Log::info('Foto upload successful', ['new_path' => $uploadResult['path']]);
            } else {
                Log::error('Foto upload failed', ['error' => $uploadResult['error']]);
                return redirect()->back()
                    ->withErrors(['foto' => 'Gagal mengupload foto: ' . $uploadResult['error']])
                    ->withInput();
            }
        } else {
            Log::info('No foto changes for penduduk', ['penduduk_id' => $penduduk->id]);
        }

        // Map status_perkawinan to status_kawin for database compatibility
        if (isset($validated['status_perkawinan'])) {
            $validated['status_kawin'] = $validated['status_perkawinan'];
            unset($validated['status_perkawinan']);
        }

        // Map pendidikan_terakhir to pendidikan for database compatibility
        if (isset($validated['pendidikan_terakhir'])) {
            $validated['pendidikan'] = $validated['pendidikan_terakhir'];
        } else {
            $validated['pendidikan'] = 'Tidak Sekolah'; // Default value
        }

        // Auto-fill nama_desa from setting if not provided
        if (empty($validated['nama_desa'])) {
            $setting = \App\Models\Setting::getSettings();
            $validated['nama_desa'] = $setting->nama_desa ?? 'Desa';
        }

        // Validate ayah_nik and ibu_nik exist in database before saving
        if (!empty($validated['ayah_nik'])) {
            $ayahExists = Penduduk::where('nik', $validated['ayah_nik'])->exists();
            if (!$ayahExists) {
                Log::warning('Ayah NIK not found in database (update)', ['ayah_nik' => $validated['ayah_nik']]);
                $validated['ayah_nik'] = null; // Set to null if not found
            }
        }
        
        if (!empty($validated['ibu_nik'])) {
            $ibuExists = Penduduk::where('nik', $validated['ibu_nik'])->exists();
            if (!$ibuExists) {
                Log::warning('Ibu NIK not found in database (update)', ['ibu_nik' => $validated['ibu_nik']]);
                $validated['ibu_nik'] = null; // Set to null if not found
            }
        }

        $penduduk->update($validated);

        // Update data keluarga jika diperlukan
        $this->handleKeluargaData($penduduk);

        return redirect()->route('admin.penduduk.index')
                        ->with('success', 'Data penduduk berhasil diperbarui');
    }

    public function destroy(Penduduk $penduduk)
    {
        try {
            // Start transaction
            DB::beginTransaction();
            
            // Delete related records first to avoid foreign key constraint errors
            
            // 1. Delete anggota keluarga records
            DB::table('anggota_keluarga')->where('nik', $penduduk->nik)->delete();
            
            // 2. Update keluarga records (set kepala_keluarga_nik to null or delete if this is the only member)
            $keluargaRecords = DB::table('keluarga')->where('kepala_keluarga_nik', $penduduk->nik)->get();
            foreach ($keluargaRecords as $keluarga) {
                // Check if there are other members in this family
                $otherMembers = DB::table('anggota_keluarga')->where('no_kk', $keluarga->no_kk)->count();
                if ($otherMembers == 0) {
                    // Delete family if no other members
                    DB::table('keluarga')->where('id', $keluarga->id)->delete();
                } else {
                    // Set kepala_keluarga_nik to null (will need manual reassignment)
                    DB::table('keluarga')->where('id', $keluarga->id)->update(['kepala_keluarga_nik' => null]);
                }
            }
            
            // 3. Delete the penduduk record
            $penduduk->delete();
            
            // Commit transaction
            DB::commit();
            
            // Check if this is an AJAX request for JSON response
            if (request()->wantsJson() || request()->ajax()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Data penduduk berhasil dihapus'
                ]);
            }
            
            return redirect()->route('admin.penduduk.index')
                            ->with('success', 'Data penduduk berhasil dihapus');
        } catch (\Exception $e) {
            // Rollback transaction on error
            DB::rollback();
            
            Log::error('Error deleting penduduk: ' . $e->getMessage(), [
                'penduduk_id' => $penduduk->id,
                'nik' => $penduduk->nik,
                'trace' => $e->getTraceAsString()
            ]);
            
            if (request()->wantsJson() || request()->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Gagal menghapus data penduduk: ' . $e->getMessage()
                ], 500);
            }
            
            return redirect()->route('admin.penduduk.index')
                            ->with('error', 'Gagal menghapus data penduduk: ' . $e->getMessage());
        }
    }

    public function map()
    {
        $penduduk = Penduduk::hasCoordinates()->with('anggotaKeluarga')->get();
        return view('admin.penduduk.map', compact('penduduk'));
    }

    public function publicMap()
    {
        $penduduk = Penduduk::hasCoordinates()->with('anggotaKeluarga')->get();
        return view('public.peta', compact('penduduk'));
    }

    public function mapData(Request $request)
    {
        $query = Penduduk::hasCoordinates();

        if ($request->dusun) {
            $query->byDusun($request->dusun);
        }

        if ($request->rt || $request->rw) {
            $query->byRtRw($request->rt, $request->rw);
        }

        $penduduk = $query->get()->map(function($p) {
            return [
                'id' => $p->id,
                'nama' => $p->nama,
                'alamat' => $p->alamat_lengkap,
                'latitude' => $p->latitude,
                'longitude' => $p->longitude,
                'status_keluarga' => $p->anggotaKeluarga->status_keluarga ?? 'Tidak Diketahui'
            ];
        });

        return response()->json($penduduk);
    }

    public function reverseGeocodeApi(Request $request)
    {
        $lat = $request->lat;
        $lng = $request->lng;
        
        $location = $this->reverseGeocode($lat, $lng);
        
        return response()->json($location);
    }

    private function reverseGeocode($lat, $lng)
    {
        try {
            $response = Http::get("https://nominatim.openstreetmap.org/reverse", [
                'lat' => $lat,
                'lon' => $lng,
                'format' => 'json',
                'addressdetails' => 1
            ]);

            if ($response->successful()) {
                $data = $response->json();
                $address = $data['address'] ?? [];
                
                return [
                    'dusun' => $address['hamlet'] ?? $address['village'] ?? null,
                    'rt' => null, // OSM tidak menyediakan RT/RW, bisa dikustomisasi
                    'rw' => null,
                    'full_address' => $data['display_name'] ?? null
                ];
            }
        } catch (\Exception $e) {
            // Log error jika diperlukan
        }

        return null;
    }

    private function handleKeluargaData($penduduk)
    {
        // Cek apakah keluarga dengan no_kk sudah ada
        $keluarga = Keluarga::where('no_kk', $penduduk->no_kk)->first();
        
        if (!$keluarga) {
            // Buat keluarga baru jika belum ada
            $keluarga = Keluarga::create([
                'no_kk' => $penduduk->no_kk,
                'kepala_keluarga_nik' => $penduduk->nik,
                'alamat' => $penduduk->alamat,
                'dusun' => $penduduk->dusun,
                'rt' => $penduduk->rt,
                'rw' => $penduduk->rw
            ]);
        }

        // Cek apakah penduduk sudah terdaftar sebagai anggota keluarga
        $anggota = AnggotaKeluarga::where('no_kk', $penduduk->no_kk)
                                 ->where('nik', $penduduk->nik)
                                 ->first();

        if (!$anggota) {
            // Tentukan status dalam keluarga
            $existingMembers = AnggotaKeluarga::where('no_kk', $penduduk->no_kk)->count();
            $statusKeluarga = $existingMembers == 0 ? 'Kepala Keluarga' : 'Anggota Keluarga';

            AnggotaKeluarga::create([
                'no_kk' => $penduduk->no_kk,
                'nik' => $penduduk->nik,
                'status_keluarga' => $statusKeluarga
            ]);
        }
    }

    /**
     * Import CSV data
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv|max:2048', // 2MB max
        ]);

        try {
            $file = $request->file('file');
            $data = [];
            $imported = 0;
            $errors = [];
            
            // Read CSV file
            if (($handle = fopen($file->getPathname(), 'r')) !== FALSE) {
                // Auto-detect delimiter (semicolon or comma)
                $firstLine = fgets($handle);
                rewind($handle);
                
                $delimiter = ',';
                if (substr_count($firstLine, ';') > substr_count($firstLine, ',')) {
                    $delimiter = ';';
                }
                
                $header = fgetcsv($handle, 0, $delimiter); // Read header with detected delimiter
                
                // Expected headers - hanya yang wajib (support both 'nama' and 'nama_lengkap', 'alamat' and 'alamat_lengkap')
                $requiredHeaders = ['nik', 'jenis_kelamin', 'tempat_lahir', 'tanggal_lahir'];
                $namaField = in_array('nama_lengkap', $header) ? 'nama_lengkap' : 'nama';
                $alamatField = in_array('alamat_lengkap', $header) ? 'alamat_lengkap' : 'alamat';
                $requiredHeaders[] = $namaField;
                $requiredHeaders[] = $alamatField;
                
                if (!$header) {
                    return redirect()->route('admin.penduduk.index')
                                   ->with('error', 'File CSV tidak memiliki header yang valid');
                }
                
                // Check if required headers exist
                $missingHeaders = array_diff($requiredHeaders, $header);
                if (!empty($missingHeaders)) {
                    return redirect()->route('admin.penduduk.index')
                                   ->with('error', 'Header yang wajib tidak ditemukan: ' . implode(', ', $missingHeaders));
                }
                
                $rowNumber = 1;
                while (($row = fgetcsv($handle, 0, $delimiter)) !== FALSE) {
                    $rowNumber++;
                    
                    try {
                        $data = array_combine($header, $row);
                        
                        // Validate required fields
                        $namaValue = $data[$namaField] ?? '';
                        if (empty($data['nik']) || empty($namaValue)) {
                            $errors[] = "Baris {$rowNumber}: NIK dan Nama Lengkap harus diisi";
                            continue;
                        }
                        
                        // Validate NIK format
                        if (strlen($data['nik']) !== 16 || !is_numeric($data['nik'])) {
                            $errors[] = "Baris {$rowNumber}: NIK harus 16 digit angka";
                            continue;
                        }
                        
                        // Check if NIK already exists
                        if (Penduduk::where('nik', $data['nik'])->exists()) {
                            $errors[] = "Baris {$rowNumber}: NIK {$data['nik']} sudah terdaftar";
                            continue;
                        }
                        
                        // Create penduduk dengan field lengkap sesuai tabel
                        $penduduk = Penduduk::create([
                            'nik' => $data['nik'],
                            'nama' => $namaValue,
                            'no_kk' => $data['no_kk'] ?? $data['nik'],
                            'tempat_lahir' => $data['tempat_lahir'] ?? '',
                            'tanggal_lahir' => $data['tanggal_lahir'] ?? '1990-01-01',
                            'jenis_kelamin' => strtoupper($data['jenis_kelamin']) === 'P' ? 'P' : 'L',
                            'status_keluarga' => $data['shdk'] ?? $data['status_keluarga'] ?? null,
                            'status_kawin' => $data['status_kawin'] ?? 'Belum Kawin',
                            'agama' => $data['agama'] ?? 'Islam',
                            'pendidikan' => $data['pendidikan'] ?? 'Tidak Sekolah',
                            'pendidikan_terakhir' => $data['pendidikan_terakhir'] ?? null,
                            'pekerjaan' => $data['pekerjaan'] ?? 'Belum Bekerja',
                            'nama_ibu' => $data['nama_ibu'] ?? null,
                            'nama_ayah' => $data['nama_ayah'] ?? null,
                            'tanggal_akta_lahir' => $data['tanggal_akta_lahir'] ?? null,
                            'no_akta_lahir' => $data['no_akta_lahir'] ?? null,
                            'alamat' => $data[$alamatField] ?? '',
                            'rt' => $data['rt'] ?? null,
                            'rw' => $data['rw'] ?? null,
                            'desa' => $data['nama_desa'] ?? $data['desa'] ?? null,
                            'dusun' => $data['nama_dusun'] ?? $data['dusun'] ?? null,
                            // Note: kecamatan, kabupaten, provinsi, kode_pos tidak ada di tabel
                            // 'kecamatan' => $data['nama_kecamatan'] ?? null,
                            // 'kabupaten' => $data['nama_kabupaten'] ?? null,
                            // 'provinsi' => $data['nama_provinsi'] ?? null,
                            // 'kode_pos' => $data['kode_pos'] ?? null,
                            'kewarganegaraan' => $data['kewarganegaraan'] ?? 'WNI',
                            'golongan_darah' => $data['golongan_darah'] ?? null,
                            'cacat_fisik' => $data['cacat_fisik'] ?? 'Tidak Ada',
                            'status_hidup' => $data['status_hidup'] ?? 'Hidup',
                            'no_wa' => $data['no_whatsapp'] ?? $data['no_wa'] ?? null,
                            'latitude' => $data['latitude'] ?? null,
                            'longitude' => $data['longitude'] ?? null,
                        ]);
                        
                        $imported++;
                        
                    } catch (\Exception $e) {
                        $errors[] = "Baris {$rowNumber}: " . $e->getMessage();
                    }
                }
                
                fclose($handle);
            }
            
            $message = "Berhasil import {$imported} data penduduk";
            if (count($errors) > 0) {
                $message .= ". " . count($errors) . " data gagal diimport: " . implode(', ', array_slice($errors, 0, 3));
                if (count($errors) > 3) {
                    $message .= " dan " . (count($errors) - 3) . " error lainnya";
                }
            }
            
            if ($imported > 0) {
                return redirect()->route('admin.penduduk.index')->with('success', $message);
            } else {
                return redirect()->route('admin.penduduk.index')->with('error', 'Tidak ada data yang berhasil diimport. ' . implode(', ', $errors));
            }
            
        } catch (\Exception $e) {
            return redirect()->route('admin.penduduk.index')
                           ->with('error', 'Gagal import data: ' . $e->getMessage());
        }
    }

    /**
     * Download Template CSV
     */
    public function downloadTemplate()
    {
        $filename = 'template_penduduk.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() {
            $file = fopen('php://output', 'w');
            
            // CSV header - lengkap sesuai tabel (menggunakan semicolon)
            fputcsv($file, [
                'nik',
                'nama_lengkap',
                'no_kk',
                'tempat_lahir',
                'tanggal_lahir',
                'jenis_kelamin',
                'status_keluarga',
                'status_kawin',
                'agama',
                'pendidikan',
                'pekerjaan',
                'nama_ibu',
                'nama_ayah',
                'tanggal_akta_lahir',
                'no_akta_lahir',
                'alamat_lengkap',
                'rt',
                'rw',
                'nama_desa',
                'nama_dusun',
                'kewarganegaraan',
                'golongan_darah',
                'cacat_fisik',
                'status_hidup',
                'no_whatsapp',
                'latitude',
                'longitude'
            ], ';');
            
            // Sample data lengkap (menggunakan semicolon)
            fputcsv($file, [
                '3201012201990001',
                'John Doe',
                '3201012201990001',
                'Jakarta',
                '1990-01-01',
                'L',
                'Kepala Keluarga',
                'Belum Kawin',
                'Islam',
                'S1',
                'Programmer',
                'Ibu John',
                'Bapak John',
                '1990-01-15',
                'AK-001',
                'Jl. Contoh No. 1',
                '001',
                '001',
                'Desa Contoh',
                'Dusun Contoh',
                'WNI',
                'A',
                'Tidak Ada',
                'Hidup',
                '628123456789',
                '-6.200000',
                '106.816635'
            ], ';');
            
            fputcsv($file, [
                '3201012205920002',
                'Jane Smith',
                '3201012205920002',
                'Bogor',
                '1992-05-15',
                'P',
                'Istri',
                'Kawin',
                'Kristen',
                'S1',
                'Guru',
                'Ibu Jane',
                'Bapak Jane',
                '1992-05-20',
                'AK-002',
                'Jl. Contoh No. 2',
                '002',
                '001',
                'Desa Contoh',
                'Dusun Contoh',
                'WNI',
                'B',
                'Tidak Ada',
                'Hidup',
                '628987654321',
                '-6.201000',
                '106.817635'
            ], ';');
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }
    
    /**
     * Export CSV data
     */
    public function export(Request $request)
    {
        $query = Penduduk::query();
        
        // Apply filters if any
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('nama', 'like', '%' . $request->search . '%')
                  ->orWhere('nik', 'like', '%' . $request->search . '%');
            });
        }
        
        if ($request->dusun) {
            $query->where('dusun', $request->dusun);
        }
        
        $penduduk = $query->get();
        
        $filename = 'data_penduduk_' . date('Y-m-d') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];
        
        $callback = function() use ($penduduk) {
            $file = fopen('php://output', 'w');
            
            // CSV header - lengkap sesuai tabel (menggunakan semicolon)
            fputcsv($file, [
                'nik',
                'nama_lengkap',
                'no_kk',
                'tempat_lahir',
                'tanggal_lahir',
                'jenis_kelamin',
                'status_keluarga',
                'status_kawin',
                'agama',
                'pendidikan',
                'pekerjaan',
                'nama_ibu',
                'nama_ayah',
                'tanggal_akta_lahir',
                'no_akta_lahir',
                'alamat_lengkap',
                'rt',
                'rw',
                'nama_desa',
                'nama_dusun',
                'kewarganegaraan',
                'golongan_darah',
                'cacat_fisik',
                'status_hidup',
                'no_whatsapp',
                'latitude',
                'longitude'
            ], ';');
            
            // Data rows lengkap (menggunakan semicolon)
            foreach ($penduduk as $p) {
                fputcsv($file, [
                    $p->nik,
                    $p->nama,
                    $p->no_kk,
                    $p->tempat_lahir,
                    $p->tanggal_lahir ? $p->tanggal_lahir->format('Y-m-d') : '',
                    $p->jenis_kelamin,
                    $p->status_keluarga ?? '',
                    $p->status_kawin,
                    $p->agama,
                    $p->pendidikan_terakhir ?? '',
                    $p->pekerjaan,
                    $p->nama_ibu,
                    $p->nama_ayah,
                    $p->tanggal_akta_lahir ? $p->tanggal_akta_lahir->format('Y-m-d') : '',
                    $p->no_akta_lahir,
                    $p->alamat,
                                    $p->rt,
                $p->rw,
                $p->nama_desa ?? '',
                $p->dusun ?? '',
                $p->kewarganegaraan,
                    $p->golongan_darah,
                    $p->cacat_fisik ?? 'Tidak Ada',
                    $p->status_hidup,
                    $p->no_wa,
                    $p->latitude ?? '',
                    $p->longitude ?? ''
                ], ';');
            }
            
            fclose($file);
        };
        
        return response()->stream($callback, 200, $headers);
    }

    /**
     * Read Excel file and convert to array
     */
    private function readExcelFile($filePath)
    {
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        
        if ($extension === 'csv') {
            return $this->readCsvFile($filePath);
        }
        
        // Try to read as Excel XML format first
        if ($extension === 'xls' || $extension === 'xlsx') {
            $data = $this->readExcelXmlFile($filePath);
            if (!empty($data)) {
                return $data;
            }
        }
        
        // Fallback to CSV reading
        return $this->readCsvFile($filePath);
    }
    
    /**
     * Read Excel XML file
     */
    private function readExcelXmlFile($filePath)
    {
        $data = [];
        $headers = [];
        
        try {
            $content = file_get_contents($filePath);
            
            // Check if it's XML format
            if (strpos($content, '<?xml') === false) {
                return [];
            }
            
            // Load XML
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($content);
            
            if ($xml === false) {
                return [];
            }
            
            // Register namespaces
            $xml->registerXPathNamespace('ss', 'urn:schemas-microsoft-com:office:spreadsheet');
            
            // Find the first worksheet with data
            $worksheets = $xml->xpath('//ss:Worksheet');
            
            if (empty($worksheets)) {
                return [];
            }
            
            // Get the first worksheet (usually the data sheet)
            $worksheet = $worksheets[0];
            $rows = $worksheet->xpath('.//ss:Row');
            
            if (empty($rows)) {
                return [];
            }
            
            $rowIndex = 0;
            foreach ($rows as $row) {
                $cells = $row->xpath('.//ss:Cell/ss:Data');
                $rowData = [];
                
                foreach ($cells as $cell) {
                    $rowData[] = (string) $cell;
                }
                
                if ($rowIndex === 0) {
                    // First row is headers
                    $headers = array_map('strtolower', $rowData);
                } else if (!empty($rowData)) {
                    // Data rows
                    $normalizedData = [];
                    foreach ($rowData as $index => $value) {
                        $header = $headers[$index] ?? 'col_' . $index;
                        $normalizedData[$this->normalizeHeader($header)] = trim($value);
                    }
                    $data[] = $normalizedData;
                }
                $rowIndex++;
            }
            
        } catch (\Exception $e) {
            Log::error('Error reading Excel XML file: ' . $e->getMessage());
            return [];
        }
        
        return $data;
    }

    /**
     * Read CSV file
     */
    private function readCsvFile($filePath)
    {
        $data = [];
        $headers = [];
        
        if (($handle = fopen($filePath, 'r')) !== FALSE) {
            $rowIndex = 0;
            
            // Try to detect delimiter (tab, semicolon, or comma)
            $firstLine = fgets($handle);
            rewind($handle);
            
            $delimiter = ','; // Default
            
            // Check for tab delimiter first (TSV)
            if (substr_count($firstLine, "\t") > 0) {
                $delimiter = "\t";
            }
            // Then check for semicolon
            elseif (substr_count($firstLine, ';') > substr_count($firstLine, ',')) {
                $delimiter = ';';
            }
            
            while (($row = fgetcsv($handle, 0, $delimiter)) !== FALSE) {
                if ($rowIndex === 0) {
                    // First row is headers
                    $headers = array_map('strtolower', $row);
                } else {
                    // Skip empty rows
                    if (empty(array_filter($row))) {
                        $rowIndex++;
                        continue;
                    }
                    
                    // Data rows
                    $rowData = [];
                    foreach ($row as $index => $value) {
                        $header = $headers[$index] ?? 'col_' . $index;
                        $rowData[$this->normalizeHeader($header)] = trim($value);
                    }
                    $data[] = $rowData;
                }
                $rowIndex++;
            }
            
            fclose($handle);
        }
        
        return $data;
    }

    /**
     * Normalize header names
     */
    private function normalizeHeader($header)
    {
        // Clean header from parentheses and extra info
        $cleanHeader = preg_replace('/\s*\([^)]*\)/', '', $header);
        $cleanHeader = trim(strtolower($cleanHeader));
        
        $mapping = [
            // Basic mappings
            'nik' => 'nik',
            'nama' => 'nama',
            'nama lengkap' => 'nama',
            'nama_lengkap' => 'nama',
            'no kk' => 'no_kk',
            'no_kk' => 'no_kk',
            
            // Birth data
            'tempat lahir' => 'tempat_lahir',
            'tanggal lahir' => 'tanggal_lahir',
            'jenis kelamin' => 'jenis_kelamin',
            
            // Family status
            'shdk' => 'status_keluarga',
            'status_keluarga' => 'status_keluarga',
            'status hubungan dalam keluarga' => 'status_keluarga',
            'status kawin' => 'status_kawin',
            
            // Personal data
            'agama' => 'agama',
            'pendidikan' => 'pendidikan_terakhir',
            'pekerjaan' => 'pekerjaan',
            
            // Parents data
            'nama ibu' => 'nama_ibu',
            'nama ibu kandung' => 'nama_ibu',
            'nama ayah' => 'nama_ayah',
            'nama ayah kandung' => 'nama_ayah',
            
            // Birth certificate
            'tanggal akta lahir' => 'tanggal_akta_lahir',
            'no akta lahir' => 'no_akta_lahir',
            
            // Address data
            'alamat' => 'alamat',
            'alamat lengkap' => 'alamat',
            'alamat_lengkap' => 'alamat',
            'rt' => 'rt',
            'rw' => 'rw',
            'desa' => 'desa',
            'nama desa' => 'desa',
            'nama_desa' => 'desa',
            'dusun' => 'dusun',
            'nama dusun' => 'dusun',
            'nama_dusun' => 'dusun',
            'kecamatan' => 'kecamatan',
            'nama kecamatan' => 'kecamatan',
            'nama_kecamatan' => 'kecamatan',
            'kabupaten' => 'kabupaten',
            'nama kabupaten' => 'kabupaten',
            'nama_kabupaten' => 'kabupaten',
            'provinsi' => 'provinsi',
            'nama provinsi' => 'provinsi',
            'nama_provinsi' => 'provinsi',
            'kode pos' => 'kode_pos',
            'kode_pos' => 'kode_pos',
            
            // Additional data
            'kewarganegaraan' => 'kewarganegaraan',
            'golongan darah' => 'golongan_darah',
            'golongan_darah' => 'golongan_darah',
            'cacat fisik' => 'cacat_fisik',
            'cacat_fisik' => 'cacat_fisik',
            'status hidup' => 'status_hidup',
            'status_hidup' => 'status_hidup',
            
            // Contact and location
            'no whatsapp' => 'no_wa',
            'no_whatsapp' => 'no_wa',
            'latitude' => 'latitude',
            'longitude' => 'longitude'
        ];

        return $mapping[$cleanHeader] ?? strtolower(str_replace([' ', '-'], '_', $cleanHeader));
    }

    /**
     * Parse date string
     */
    private function parseDate($dateString)
    {
        try {
            return \Carbon\Carbon::parse($dateString)->format('Y-m-d');
        } catch (\Exception $e) {
            return '1990-01-01'; // Default date
        }
    }

    /**
     * Parse gender
     */
    private function parseGender($gender)
    {
        $gender = strtolower(trim($gender));
        if (in_array($gender, ['l', 'laki-laki', 'male', '1'])) {
            return 'L';
        }
        return 'P';
    }

    /**
     * Clean phone number
     */
    private function cleanPhoneNumber($phone)
    {
        if (!$phone) return null;
        
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }
        
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        return $phone;
    }

    /**
     * Generate default password for penduduk
     */
    public function generatePassword(Penduduk $penduduk)
    {
        try {
            $password = $penduduk->generateDefaultPassword();
            
            return response()->json([
                'success' => true,
                'message' => 'Password berhasil digenerate',
                'password' => $password
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal generate password: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reset password to default
     */
    public function resetPassword(Penduduk $penduduk)
    {
        try {
            $password = $penduduk->generateDefaultPassword();
            
            return response()->json([
                'success' => true,
                'message' => 'Password berhasil direset',
                'password' => $password
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal reset password: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set custom password
     */
    public function setPassword(Request $request, Penduduk $penduduk)
    {
        $request->validate([
            'password' => 'required|string|min:6',
            'must_change' => 'boolean'
        ]);

        try {
            $penduduk->update([
                'password' => bcrypt($request->password),
                'password_set_at' => now(),
                'password_must_change' => $request->boolean('must_change', false)
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Password berhasil diatur'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal mengatur password: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Force password change on next login
     */
    public function forcePasswordChange(Penduduk $penduduk)
    {
        try {
            $penduduk->update(['password_must_change' => true]);
            
            return response()->json([
                'success' => true,
                'message' => 'Berhasil memaksa ganti password'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Preview bulk password generation
     */
    public function bulkPasswordPreview(Request $request)
    {
        $request->validate([
            'filter_type' => 'required|in:all,dusun,rt_rw',
            'filter_value' => 'nullable|string',
            'max_count' => 'required|integer|min:1|max:500'
        ]);

        try {
            $query = Penduduk::whereNull('password');

            // Apply filters
            if ($request->filter_type === 'dusun' && $request->filter_value) {
                $query->where('dusun', $request->filter_value);
            } elseif ($request->filter_type === 'rt_rw' && $request->filter_value) {
                $rtRw = explode('/', $request->filter_value);
                if (count($rtRw) === 2) {
                    $query->where('rt', $rtRw[0])->where('rw', $rtRw[1]);
                }
            }

            $penduduk = $query->limit($request->max_count)
                             ->select('id', 'nik', 'nama', 'dusun', 'rt', 'rw')
                             ->get();

            return response()->json([
                'success' => true,
                'penduduk' => $penduduk,
                'count' => $penduduk->count()
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal memuat preview: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Generate bulk passwords
     */
    public function bulkPasswordGenerate(Request $request)
    {
        $request->validate([
            'filter_type' => 'required|in:all,dusun,rt_rw',
            'filter_value' => 'nullable|string',
            'max_count' => 'required|integer|min:1|max:500',
            'force_change' => 'boolean'
        ]);

        try {
            $query = Penduduk::whereNull('password');

            // Apply filters
            if ($request->filter_type === 'dusun' && $request->filter_value) {
                $query->where('dusun', $request->filter_value);
            } elseif ($request->filter_type === 'rt_rw' && $request->filter_value) {
                $rtRw = explode('/', $request->filter_value);
                if (count($rtRw) === 2) {
                    $query->where('rt', $rtRw[0])->where('rw', $rtRw[1]);
                }
            }

            $pendudukList = $query->limit($request->max_count)->get();

            $results = [];
            $successCount = 0;

            foreach ($pendudukList as $penduduk) {
                try {
                    $password = $penduduk->generateDefaultPassword();
                    
                    // Set force change if requested
                    if ($request->boolean('force_change', true)) {
                        $penduduk->update(['password_must_change' => true]);
                    }

                    $results[] = [
                        'nik' => $penduduk->nik,
                        'nama' => $penduduk->nama,
                        'password' => $password,
                        'status' => 'Success'
                    ];
                    
                    $successCount++;

                } catch (\Exception $e) {
                    $results[] = [
                        'nik' => $penduduk->nik,
                        'nama' => $penduduk->nama,
                        'password' => null,
                        'status' => 'Error: ' . $e->getMessage()
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'message' => "Berhasil generate {$successCount} password",
                'success_count' => $successCount,
                'total_processed' => count($results),
                'results' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Gagal generate password: ' . $e->getMessage()
            ], 500);
        }
    }
} 
