<?php

namespace App\Http\Controllers;

use App\Models\StrukturOrganisasi;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class StrukturOrganisasiController extends Controller
{
    public function index(Request $request)
    {
        $query = StrukturOrganisasi::with('user');

        if ($request->has('kategori_lembaga')) {
            $query->byKategori($request->kategori_lembaga);
        }

        if ($request->has('status_aktif')) {
            $query->where('status_aktif', $request->status_aktif);
        }

        $struktur = $query->orderByUrutan()->get();

        // If AJAX request or API, return JSON
        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'data' => $struktur->paginate(20)
            ]);
        }

        // Get statistics for the view
        $stats = StrukturOrganisasi::getStatistikOrganisasi();
        
        // Get kategori options
        $kategoriOptions = StrukturOrganisasi::getKategoriLembagaOptions();

        // Return view for web requests
        return view('admin.struktur-organisasi.index', compact('struktur', 'stats', 'kategoriOptions'));
    }

    public function create()
    {
        $kategoriOptions = StrukturOrganisasi::getKategoriLembagaOptions();
        $pendidikanOptions = StrukturOrganisasi::getPendidikanOptions();
        $jenisKelaminOptions = StrukturOrganisasi::getJenisKelaminOptions();
        
        return view('admin.struktur-organisasi.create', compact('kategoriOptions', 'pendidikanOptions', 'jenisKelaminOptions'));
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'kategori_lembaga' => 'required|string',
            'nama_lembaga' => 'required|string|max:255',
            'nama_pengurus' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'nik' => 'nullable|string|size:16',
            'tempat_lahir' => 'nullable|string|max:255',
            'tanggal_lahir' => 'nullable|date',
            'jenis_kelamin' => 'nullable|in:L,P',
            'alamat' => 'nullable|string',
            'nomor_hp' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'pendidikan' => 'nullable|string',
            'pekerjaan' => 'nullable|string|max:255',
            'tanggal_mulai_jabatan' => 'required|date',
            'tanggal_selesai_jabatan' => 'nullable|date|after:tanggal_mulai_jabatan',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'urutan_tampil' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $data = $validator->validated();
        $data['user_id'] = Auth::id();

        // Handle foto upload
        if ($request->hasFile('foto')) {
            $file = $request->file('foto');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('public/organisasi', $filename);
            $data['foto'] = $filename;
        }

        $organisasi = StrukturOrganisasi::create($data);

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => 'Data struktur organisasi berhasil ditambahkan',
                'data' => $organisasi
            ], 201);
        }

        return redirect()->route('admin.struktur-organisasi.index')
                        ->with('success', 'Data struktur organisasi berhasil ditambahkan');
    }

    public function show($id)
    {
        $organisasi = StrukturOrganisasi::with('user')->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $organisasi
        ]);
    }

    public function update(Request $request, $id)
    {
        $organisasi = StrukturOrganisasi::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'kategori_lembaga' => 'required|string',
            'nama_lembaga' => 'required|string|max:255',
            'nama_pengurus' => 'required|string|max:255',
            'jabatan' => 'required|string|max:255',
            'nik' => 'nullable|string|size:16',
            'tempat_lahir' => 'nullable|string|max:255',
            'tanggal_lahir' => 'nullable|date',
            'jenis_kelamin' => 'nullable|in:L,P',
            'alamat' => 'nullable|string',
            'nomor_hp' => 'nullable|string|max:15',
            'email' => 'nullable|email',
            'pendidikan' => 'nullable|string',
            'pekerjaan' => 'nullable|string|max:255',
            'tanggal_mulai_jabatan' => 'required|date',
            'tanggal_selesai_jabatan' => 'nullable|date|after:tanggal_mulai_jabatan',
            'status_aktif' => 'required|in:aktif,tidak_aktif',
            'foto' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            'urutan_tampil' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $validator->validated();

        // Handle foto upload
        if ($request->hasFile('foto')) {
            // Delete old foto
            if ($organisasi->foto && Storage::exists('public/organisasi/' . $organisasi->foto)) {
                Storage::delete('public/organisasi/' . $organisasi->foto);
            }

            $file = $request->file('foto');
            $filename = time() . '_' . $file->getClientOriginalName();
            $path = $file->storeAs('public/organisasi', $filename);
            $data['foto'] = $filename;
        }

        $organisasi->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Data struktur organisasi berhasil diperbarui',
            'data' => $organisasi
        ]);
    }

    public function destroy($id)
    {
        $organisasi = StrukturOrganisasi::findOrFail($id);

        // Delete foto if exists
        if ($organisasi->foto && Storage::exists('public/organisasi/' . $organisasi->foto)) {
            Storage::delete('public/organisasi/' . $organisasi->foto);
        }

        $organisasi->delete();

        return response()->json([
            'success' => true,
            'message' => 'Data struktur organisasi berhasil dihapus'
        ]);
    }

    public function byKategori($kategori)
    {
        $organisasi = StrukturOrganisasi::byKategori($kategori)
            ->aktif()
            ->orderByUrutan()
            ->get();

        return response()->json([
            'success' => true,
            'data' => $organisasi
        ]);
    }

    public function perangkatDesa()
    {
        $perangkat = StrukturOrganisasi::getPerangkatDesaByJabatan();

        return response()->json([
            'success' => true,
            'data' => $perangkat
        ]);
    }

    public function statistik()
    {
        $stats = StrukturOrganisasi::getStatistikOrganisasi();

        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    public function nonaktifkan($id)
    {
        $organisasi = StrukturOrganisasi::findOrFail($id);
        $organisasi->nonaktifkan();

        return response()->json([
            'success' => true,
            'message' => 'Pengurus berhasil dinonaktifkan'
        ]);
    }

    public function aktifkan($id)
    {
        $organisasi = StrukturOrganisasi::findOrFail($id);
        $organisasi->aktifkan();

        return response()->json([
            'success' => true,
            'message' => 'Pengurus berhasil diaktifkan'
        ]);
    }
} 