<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Order;
use App\Services\TripayService;
use Illuminate\Support\Facades\Log;

class TripayController extends Controller
{
    protected $tripayService;

    public function __construct(TripayService $tripayService)
    {
        $this->tripayService = $tripayService;
    }

    /**
     * Handle Tripay callback
     */
    public function callback(Request $request)
    {
        try {
            // Get callback signature
            $callbackSignature = $request->server('HTTP_X_CALLBACK_SIGNATURE');
            $json = $request->getContent();
            $data = json_decode($json, true);

            // Validate signature
            if (!$this->tripayService->validateCallbackSignature($callbackSignature, $json)) {
                Log::error('Invalid Tripay callback signature');
                return response()->json(['success' => false], 400);
            }

            // Find order by reference
            $order = Order::where('tripay_reference', $data['reference'])->first();
            
            if (!$order) {
                Log::error('Order not found for reference: ' . $data['reference']);
                return response()->json(['success' => false], 404);
            }

            // Update order status based on payment status
            switch ($data['status']) {
                case 'PAID':
                    $order->update([
                        'status' => Order::STATUS_PAID,
                        'paid_at' => now()
                    ]);
                    
                    // Additional logic after successful payment
                    $this->handleSuccessfulPayment($order);
                    break;

                case 'EXPIRED':
                    $order->update([
                        'status' => Order::STATUS_EXPIRED
                    ]);
                    
                    // Restore product stock
                    $this->restoreProductStock($order);
                    break;

                case 'FAILED':
                    $order->update([
                        'status' => Order::STATUS_FAILED
                    ]);
                    
                    // Restore product stock
                    $this->restoreProductStock($order);
                    break;

                case 'REFUND':
                    $order->update([
                        'status' => Order::STATUS_CANCELLED
                    ]);
                    break;

                default:
                    Log::info('Unhandled Tripay status: ' . $data['status']);
                    break;
            }

            Log::info('Tripay callback processed successfully', [
                'reference' => $data['reference'],
                'status' => $data['status'],
                'order_id' => $order->order_id
            ]);

            return response()->json(['success' => true]);

        } catch (\Exception $e) {
            Log::error('Tripay callback error: ' . $e->getMessage(), [
                'request_data' => $request->all()
            ]);
            
            return response()->json(['success' => false], 500);
        }
    }

    /**
     * Handle successful payment logic
     */
    private function handleSuccessfulPayment(Order $order)
    {
        try {
            // Change order status to processing
            $order->update(['status' => Order::STATUS_PROCESSING]);

            // Here you can add additional logic:
            // - Send notification to customer
            // - Send notification to UMKM
            // - Update sales statistics
            // - etc.

            Log::info('Successful payment handled for order: ' . $order->order_id);

        } catch (\Exception $e) {
            Log::error('Error handling successful payment: ' . $e->getMessage());
        }
    }

    /**
     * Restore product stock for failed/expired orders
     */
    private function restoreProductStock(Order $order)
    {
        try {
            foreach ($order->orderItems as $item) {
                if ($item->produk) {
                    $item->produk->increment('stok', $item->quantity);
                }
            }

            Log::info('Product stock restored for order: ' . $order->order_id);

        } catch (\Exception $e) {
            Log::error('Error restoring product stock: ' . $e->getMessage());
        }
    }
} 