<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class SecurityHeaders
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);
        
        // Only apply security headers to HTML responses
        if ($response->headers->get('Content-Type') && 
            strpos($response->headers->get('Content-Type'), 'text/html') !== false) {
            
            // Prevent MIME type sniffing
            $response->headers->set('X-Content-Type-Options', 'nosniff');
            
            // Prevent clickjacking
            $response->headers->set('X-Frame-Options', 'SAMEORIGIN');
            
            // Enable XSS protection
            $response->headers->set('X-XSS-Protection', '1; mode=block');
            
            // Referrer policy
            $response->headers->set('Referrer-Policy', 'strict-origin-when-cross-origin');
            
            // Content Security Policy (TailwindCSS & Cloudflare compatible) - Allow geolocation
            $csp = "default-src 'self'; " .
                   "script-src 'self' 'unsafe-inline' 'unsafe-eval' https://cdn.tailwindcss.com https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://static.cloudflareinsights.com https://unpkg.com; " .
                   "style-src 'self' 'unsafe-inline' https://cdn.tailwindcss.com https://cdn.jsdelivr.net https://cdnjs.cloudflare.com https://fonts.googleapis.com; " .
                   "font-src 'self' https://fonts.gstatic.com https://cdnjs.cloudflare.com; " .
                   "img-src 'self' data: https: blob:; " .
                   "connect-src 'self' https://cloudflareinsights.com; " .
                   "frame-src 'self' https://www.youtube.com https://youtube.com; " .
                   "object-src 'none'; " .
                   "base-uri 'self'; " .
                   "form-action 'self'; " .
                   "geolocation 'self';";
            
            $response->headers->set('Content-Security-Policy', $csp);
            
            // Additional security headers - Allow geolocation for registration form
            $response->headers->set('Permissions-Policy', 'microphone=(), camera=()');
            
            // HTTPS enforcement (only if HTTPS is detected)
            if ($request->isSecure()) {
                $response->headers->set('Strict-Transport-Security', 'max-age=31536000; includeSubDomains; preload');
            }
        }
        
        return $response;
    }
}
