<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AgendaDesa extends Model
{
    use HasFactory;

    protected $table = 'agenda_desa';

    protected $fillable = [
        'judul',
        'deskripsi',
        'jenis_kegiatan',
        'tanggal_mulai',
        'tanggal_selesai',
        'tempat',
        'alamat_lengkap',
        'latitude',
        'longitude',
        'penyelenggara',
        'penanggung_jawab',
        'kontak_person',
        'target_peserta',
        'filter_peserta',
        'kapasitas',
        'perlu_daftar',
        'status',
        'tampil_public',
        'banner',
        'kirim_reminder',
        'waktu_reminder',
        'hasil_kegiatan',
        'dokumentasi',
        'jumlah_peserta',
        'created_by'
    ];

    protected $casts = [
        'tanggal_mulai' => 'datetime',
        'tanggal_selesai' => 'datetime',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'filter_peserta' => 'array',
        'waktu_reminder' => 'array',
        'perlu_daftar' => 'boolean',
        'tampil_public' => 'boolean',
        'kirim_reminder' => 'boolean'
    ];

    // Relationships
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scope methods
    public function scopePublik($query)
    {
        return $query->where('tampil_public', true);
    }

    public function scopeTerjadwal($query)
    {
        return $query->where('status', 'Terjadwal');
    }

    public function scopeSedangBerlangsung($query)
    {
        return $query->where('status', 'Berlangsung')
                    ->orWhere(function($q) {
                        $q->where('status', 'Terjadwal')
                          ->where('tanggal_mulai', '<=', now())
                          ->where('tanggal_selesai', '>=', now());
                    });
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_kegiatan', $jenis);
    }

    public function scopeBulanIni($query)
    {
        return $query->whereMonth('tanggal_mulai', now()->month)
                    ->whereYear('tanggal_mulai', now()->year);
    }

    public function scopeMingguIni($query)
    {
        return $query->whereBetween('tanggal_mulai', [
            now()->startOfWeek(),
            now()->endOfWeek()
        ]);
    }

    // Method untuk cek apakah sedang berlangsung
    public function isSedangBerlangsung()
    {
        return now()->isBetween($this->tanggal_mulai, $this->tanggal_selesai) &&
               in_array($this->status, ['Terjadwal', 'Berlangsung']);
    }

    // Method untuk cek apakah sudah selesai
    public function isSudahSelesai()
    {
        return now()->isAfter($this->tanggal_selesai) || $this->status === 'Selesai';
    }

    // Method untuk cek apakah akan dimulai
    public function isAkanDimulai()
    {
        return now()->isBefore($this->tanggal_mulai) && $this->status === 'Terjadwal';
    }

    // Method untuk auto update status
    public function autoUpdateStatus()
    {
        if ($this->status !== 'Terjadwal') {
            return;
        }

        if (now()->isBefore($this->tanggal_mulai)) {
            // Belum dimulai
            return;
        } elseif (now()->isBetween($this->tanggal_mulai, $this->tanggal_selesai)) {
            // Sedang berlangsung
            $this->update(['status' => 'Berlangsung']);
        } elseif (now()->isAfter($this->tanggal_selesai)) {
            // Sudah selesai
            $this->update(['status' => 'Selesai']);
        }
    }

    // Method untuk get durasi kegiatan
    public function getDurasi()
    {
        return $this->tanggal_selesai->diff($this->tanggal_mulai);
    }

    // Method untuk get peserta yang memenuhi filter
    public function getTargetPeserta()
    {
        if ($this->target_peserta === 'Semua Warga') {
            return Penduduk::hidup()->get();
        }

        $query = Penduduk::hidup();

        if (!$this->filter_peserta) {
            return collect();
        }

        switch ($this->target_peserta) {
            case 'RT Tertentu':
                return $query->whereIn('rt', $this->filter_peserta)->get();
            
            case 'RW Tertentu':
                return $query->whereIn('rw', $this->filter_peserta)->get();
            
            case 'Dusun Tertentu':
                return $query->whereIn('dusun', $this->filter_peserta)->get();
            
            case 'Kelompok Usia':
                $minUmur = $this->filter_peserta['min_umur'] ?? 0;
                $maxUmur = $this->filter_peserta['max_umur'] ?? 100;
                return $query->whereRaw('TIMESTAMPDIFF(YEAR, tanggal_lahir, CURDATE()) BETWEEN ? AND ?', 
                                      [$minUmur, $maxUmur])->get();
            
            default:
                return collect();
        }
    }

    // Method untuk kirim reminder
    public function shouldSendReminder()
    {
        if (!$this->kirim_reminder || !$this->waktu_reminder || $this->status !== 'Terjadwal') {
            return false;
        }

        foreach ($this->waktu_reminder as $waktu) {
            $reminderTime = $this->tanggal_mulai->sub($waktu);
            if (now()->format('Y-m-d H:i') === $reminderTime->format('Y-m-d H:i')) {
                return true;
            }
        }

        return false;
    }

    // Attribute untuk status badge
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'Draft' => 'secondary',
            'Terjadwal' => 'primary',
            'Berlangsung' => 'warning',
            'Selesai' => 'success',
            'Dibatalkan' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk get agenda bulan ini
    public static function getAgendaBulanIni()
    {
        return self::publik()
                  ->bulanIni()
                  ->whereIn('status', ['Terjadwal', 'Berlangsung'])
                  ->orderBy('tanggal_mulai')
                  ->get();
    }

    // Method untuk get agenda minggu ini
    public static function getAgendaMingguIni()
    {
        return self::publik()
                  ->mingguIni()
                  ->whereIn('status', ['Terjadwal', 'Berlangsung'])
                  ->orderBy('tanggal_mulai')
                  ->get();
    }
} 