<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Auth;

class ApbdesRealisasi extends Model
{
    use HasFactory;

    protected $table = 'apbdes_realisasi';

    protected $fillable = [
        'apbdes_rencana_id',
        'tanggal_transaksi',
        'nomor_bukti',
        'uraian_transaksi',
        'jumlah',
        'jenis',
        'sumber_dana',
        'penerima_pembayaran',
        'keterangan',
        'bukti_transaksi',
        'status_verifikasi',
        'user_id',
        'verified_at',
        'verified_by'
    ];

    protected $casts = [
        'jumlah' => 'decimal:2',
        'tanggal_transaksi' => 'date',
        'verified_at' => 'datetime'
    ];

    // Relationships
    public function rencana(): BelongsTo
    {
        return $this->belongsTo(ApbdesRencana::class, 'apbdes_rencana_id');
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function verifier(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    // Scopes
    public function scopeVerified($query)
    {
        return $query->where('status_verifikasi', 'verified');
    }

    public function scopePending($query)
    {
        return $query->where('status_verifikasi', 'pending');
    }

    public function scopeRejected($query)
    {
        return $query->where('status_verifikasi', 'rejected');
    }

    public function scopePemasukan($query)
    {
        return $query->where('jenis', 'pemasukan');
    }

    public function scopePengeluaran($query)
    {
        return $query->where('jenis', 'pengeluaran');
    }

    public function scopeByTanggal($query, $start, $end = null)
    {
        if ($end) {
            return $query->whereBetween('tanggal_transaksi', [$start, $end]);
        }
        return $query->whereDate('tanggal_transaksi', $start);
    }

    // Accessors
    public function getFormattedJumlahAttribute()
    {
        return 'Rp ' . number_format($this->jumlah, 0, ',', '.');
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'pending' => '<span class="badge bg-warning">Pending</span>',
            'verified' => '<span class="badge bg-success">Verified</span>',
            'rejected' => '<span class="badge bg-danger">Rejected</span>'
        ];

        return $badges[$this->status_verifikasi] ?? '';
    }

    public function getJenisBadgeAttribute()
    {
        $badges = [
            'pemasukan' => '<span class="badge bg-success">Pemasukan</span>',
            'pengeluaran' => '<span class="badge bg-danger">Pengeluaran</span>'
        ];

        return $badges[$this->jenis] ?? '';
    }

    // Methods
    public function verify($userId = null)
    {
        $this->update([
            'status_verifikasi' => 'verified',
            'verified_at' => now(),
            'verified_by' => $userId ?? (Auth::check() ? Auth::user()->id : null)
        ]);

        // Update realisasi di rencana
        $this->rencana->updateRealisasi();
    }

    public function reject($userId = null)
    {
        $this->update([
            'status_verifikasi' => 'rejected',
            'verified_at' => now(),
            'verified_by' => $userId ?? (Auth::check() ? Auth::user()->id : null)
        ]);
    }

    public static function getJenisOptions()
    {
        return [
            'pemasukan' => 'Pemasukan',
            'pengeluaran' => 'Pengeluaran'
        ];
    }

    public static function getStatusOptions()
    {
        return [
            'pending' => 'Pending',
            'verified' => 'Verified',
            'rejected' => 'Rejected'
        ];
    }

    public static function getTotalPemasukan($rencanaId = null, $startDate = null, $endDate = null)
    {
        $query = self::verified()->pemasukan();
        
        if ($rencanaId) {
            $query->where('apbdes_rencana_id', $rencanaId);
        }
        
        if ($startDate && $endDate) {
            $query->byTanggal($startDate, $endDate);
        }
        
        return $query->sum('jumlah');
    }

    public static function getTotalPengeluaran($rencanaId = null, $startDate = null, $endDate = null)
    {
        $query = self::verified()->pengeluaran();
        
        if ($rencanaId) {
            $query->where('apbdes_rencana_id', $rencanaId);
        }
        
        if ($startDate && $endDate) {
            $query->byTanggal($startDate, $endDate);
        }
        
        return $query->sum('jumlah');
    }
} 