<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class ApbdesRencana extends Model
{
    use HasFactory;

    protected $table = 'apbdes_rencana';

    protected $fillable = [
        'tahun_anggaran',
        'kategori',
        'bidang',
        'kode_rekening',
        'uraian',
        'keterangan',
        'anggaran',
        'realisasi',
        'sisa_anggaran',
        'status',
        'user_id'
    ];

    protected $casts = [
        'anggaran' => 'decimal:2',
        'realisasi' => 'decimal:2',
        'sisa_anggaran' => 'decimal:2',
        'tahun_anggaran' => 'integer'
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function realisasi(): HasMany
    {
        return $this->hasMany(ApbdesRealisasi::class);
    }

    // Scopes
    public function scopeByTahun($query, $tahun)
    {
        return $query->where('tahun_anggaran', $tahun);
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori', $kategori);
    }

    public function scopeByBidang($query, $bidang)
    {
        return $query->where('bidang', $bidang);
    }

    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    // Accessors
    public function getPersentaseRealisasiAttribute()
    {
        if ($this->anggaran > 0) {
            return round(($this->realisasi / $this->anggaran) * 100, 2);
        }
        return 0;
    }

    public function getFormattedAnggaranAttribute()
    {
        return 'Rp ' . number_format($this->anggaran, 0, ',', '.');
    }

    public function getFormattedRealisasiAttribute()
    {
        return 'Rp ' . number_format($this->realisasi, 0, ',', '.');
    }

    public function getFormattedSisaAnggaranAttribute()
    {
        return 'Rp ' . number_format($this->sisa_anggaran, 0, ',', '.');
    }

    // Static methods
    public static function getTotalByKategori($tahun, $kategori)
    {
        return self::byTahun($tahun)
            ->byKategori($kategori)
            ->sum('anggaran');
    }

    // Methods
    public function updateRealisasi()
    {
        $totalRealisasi = $this->realisasi()
            ->where('status_verifikasi', 'verified')
            ->sum('jumlah');
        
        $this->update([
            'realisasi' => $totalRealisasi,
            'sisa_anggaran' => $this->anggaran - $totalRealisasi
        ]);
    }


    public static function getRealisasiByKategori($tahun, $kategori)
    {
        return self::byTahun($tahun)
            ->byKategori($kategori)
            ->aktif()
            ->sum('realisasi');
    }

    public static function getBidangOptions()
    {
        return [
            'pemerintahan' => 'Bidang Pemerintahan',
            'pembangunan' => 'Bidang Pembangunan',
            'pembinaan' => 'Bidang Pembinaan',
            'pemberdayaan' => 'Bidang Pemberdayaan',
            'dana_desa' => 'Dana Desa',
            'add' => 'Alokasi Dana Desa (ADD)',
            'pad' => 'Pendapatan Asli Desa (PAD)',
            'surplus_defisit' => 'Surplus/Defisit'
        ];
    }

    public static function getKategoriOptions()
    {
        return [
            'pendapatan' => 'Pendapatan',
            'belanja' => 'Belanja',
            'pembiayaan' => 'Pembiayaan'
        ];
    }
} 