<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class ArsipSuratKeluar extends Model
{
    use HasFactory;

    protected $table = 'arsip_surat_keluar';

    protected $fillable = [
        'nomor_surat',
        'jenis_surat',
        'perihal',
        'tujuan',
        'tanggal_surat',
        'isi_surat',
        'penandatangan_nama',
        'penandatangan_jabatan',
        'penandatangan_nip',
        'file_surat',
        'lampiran',
        'klasifikasi',
        'status',
        'tanggal_kirim',
        'keterangan',
        'created_by'
    ];

    protected $casts = [
        'tanggal_surat' => 'date',
        'tanggal_kirim' => 'date',
        'tujuan' => 'array'
    ];

    // Relationships
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scope methods
    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_surat', $jenis);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    public function scopeByKlasifikasi($query, $klasifikasi)
    {
        return $query->where('klasifikasi', $klasifikasi);
    }

    public function scopeTahunIni($query)
    {
        return $query->whereYear('tanggal_surat', now()->year);
    }

    // Method untuk generate nomor surat otomatis
    public function generateNomorSurat()
    {
        $tahun = now()->year;
        $counter = self::whereYear('tanggal_surat', $tahun)->count() + 1;
        
        // Format: 001/SK-DESA/I/2025
        $bulan = now()->format('m');
        $bulanRomawi = $this->getRomawi($bulan);
        
        $format = str_pad($counter, 3, '0', STR_PAD_LEFT) . '/' . 
                  $this->getKodeSurat() . '/' . 
                  $bulanRomawi . '/' . 
                  $tahun;
        
        $this->nomor_surat = $format;
        $this->save();
        
        return $format;
    }

    // Helper method untuk kode surat
    private function getKodeSurat()
    {
        return match($this->jenis_surat) {
            'SK', 'Surat Keputusan' => 'SK-DESA',
            'Undangan', 'Surat Undangan' => 'UND-DESA',
            'Edaran', 'Surat Edaran' => 'SE-DESA',
            'Pemberitahuan', 'Surat Pemberitahuan' => 'PENG-DESA',
            'Tugas', 'Surat Tugas' => 'ST-DESA',
            'Rekomendasi', 'Surat Rekomendasi' => 'REK-DESA',
            default => 'DESA'
        };
    }

    // Helper method untuk konversi bulan ke romawi
    private function getRomawi($bulan)
    {
        $romawi = [
            1 => 'I', 2 => 'II', 3 => 'III', 4 => 'IV',
            5 => 'V', 6 => 'VI', 7 => 'VII', 8 => 'VIII',
            9 => 'IX', 10 => 'X', 11 => 'XI', 12 => 'XII'
        ];
        
        return $romawi[(int)$bulan] ?? 'I';
    }

    // Attribute untuk badge color
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status) {
            'Draft' => 'secondary',
            'Dikirim' => 'success',
            'Arsip' => 'info',
            default => 'secondary'
        };
    }

    public function getKlasifikasiBadgeColorAttribute()
    {
        return match($this->klasifikasi) {
            'Biasa' => 'primary',
            'Penting' => 'warning',
            'Rahasia' => 'danger',
            default => 'primary'
        };
    }
} 