<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Berita extends Model
{
    use HasFactory;

    protected $table = 'berita';

    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName()
    {
        return 'id';
    }

    protected $fillable = [
        'judul',
        'slug',
        'konten',
        'ringkasan',
        'gambar',
        'alt_text',
        'kategori',
        'status',
        'prioritas',
        'user_id',
        'tanggal_terbit',
        'tanggal_publish',
        'meta_description',
        'keywords',
        'tags',
        'allow_comments',
        'is_featured',
        'views',
        'likes_count',
        'comments_count',
        'shares_count'
    ];

    protected $casts = [
        'tanggal_terbit' => 'datetime',
        'tanggal_publish' => 'datetime',
        'allow_comments' => 'boolean',
        'is_featured' => 'boolean'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function comments()
    {
        return $this->hasMany(BeritaComment::class)->orderBy('created_at', 'desc');
    }

    public function approvedComments()
    {
        return $this->hasMany(BeritaComment::class)->where('status', 'approved')->orderBy('created_at', 'desc');
    }

    // Scopes
    public function scopeTerbit($query)
    {
        return $query->where('status', 'Terbit')
                    ->where('tanggal_terbit', '<=', now());
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori', $kategori);
    }

    public function scopePopular($query, $limit = 5)
    {
        return $query->orderBy('views', 'desc')->limit($limit);
    }

    public function scopeTerbaru($query, $limit = 5)
    {
        return $query->orderBy('tanggal_terbit', 'desc')->limit($limit);
    }

    // Mutators
    public function setJudulAttribute($value)
    {
        $this->attributes['judul'] = $value;
        $this->attributes['slug'] = Str::slug($value);
    }

    // Accessors
    public function getExcerptAttribute($length = 150)
    {
        return $this->ringkasan ?: Str::limit(strip_tags($this->konten), $length);
    }

    public function getReadTimeAttribute()
    {
        $words = str_word_count(strip_tags($this->konten));
        $minutes = ceil($words / 200); // Rata-rata 200 kata per menit
        return $minutes . ' menit baca';
    }

    public function getKategoriColorAttribute()
    {
        $colors = [
            'Pengumuman' => 'primary',
            'Kegiatan' => 'success',
            'Pembangunan' => 'warning',
            'Kesehatan' => 'danger',
            'Pendidikan' => 'info',
            'UMKM' => 'secondary',
            'Sosial' => 'dark',
            'Keamanan' => 'danger',
            'Lainnya' => 'secondary'
        ];
        
        return $colors[$this->kategori] ?? 'secondary';
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'Terbit' => 'success',
            'Draft' => 'warning',
            'Arsip' => 'secondary',
            'Dijadwalkan' => 'info'
        ];
        
        return $colors[$this->status] ?? 'secondary';
    }

    // Alias untuk relasi user sebagai author
    public function author()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    // Methods
    public function incrementViews()
    {
        $this->increment('views');
    }

    public function incrementLikes()
    {
        $this->increment('likes_count');
    }

    public function decrementLikes()
    {
        $this->decrement('likes_count');
    }

    public function incrementComments()
    {
        $this->increment('comments_count');
    }

    public function decrementComments()
    {
        $this->decrement('comments_count');
    }

    public function incrementShares()
    {
        $this->increment('shares_count');
    }
} 
