<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class InventarisDesa extends Model
{
    use HasFactory;

    protected $table = 'inventaris_desa';

    protected $fillable = [
        'kode_inventaris',
        'nama_barang',
        'kategori',
        'deskripsi',
        'merk_tipe',
        'tahun_perolehan',
        'harga_perolehan',
        'nilai_sekarang',
        'sumber_dana',
        'kondisi',
        'status_pemakaian',
        'lokasi_penyimpanan',
        'alamat_detail',
        'penanggung_jawab',
        'jabatan_pj',
        'latitude',
        'longitude',
        'spesifikasi',
        'terakhir_maintenance',
        'jadwal_maintenance_berikutnya',
        'biaya_maintenance_per_tahun',
        'foto_barang',
        'dokumen_pendukung',
        'qr_code',
        'keterangan',
        'created_by'
    ];

    protected $casts = [
        'harga_perolehan' => 'decimal:2',
        'nilai_sekarang' => 'decimal:2',
        'biaya_maintenance_per_tahun' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'spesifikasi' => 'array',
        'terakhir_maintenance' => 'date',
        'jadwal_maintenance_berikutnya' => 'date'
    ];

    // Relationships
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    // Scope methods
    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori', $kategori);
    }

    public function scopeByKondisi($query, $kondisi)
    {
        return $query->where('kondisi', $kondisi);
    }

    public function scopeByStatus($query, $status)
    {
        return $query->where('status_pemakaian', $status);
    }

    public function scopeBaik($query)
    {
        return $query->where('kondisi', 'Baik');
    }

    public function scopeRusak($query)
    {
        return $query->whereIn('kondisi', ['Rusak Ringan', 'Rusak Berat']);
    }

    public function scopeDigunakan($query)
    {
        return $query->where('status_pemakaian', 'Digunakan');
    }

    public function scopePerluMaintenance($query)
    {
        return $query->where('jadwal_maintenance_berikutnya', '<=', now());
    }

    // Method untuk generate kode inventaris
    public static function generateKodeInventaris($kategori)
    {
        $kodeKategori = self::getKodeKategori($kategori);
        $tahun = now()->format('Y');
        $counter = self::where('kategori', $kategori)
                      ->whereYear('created_at', now()->year)
                      ->count() + 1;
        
        return $kodeKategori . '-' . $tahun . '-' . str_pad($counter, 3, '0', STR_PAD_LEFT);
    }

    // Helper method untuk kode kategori
    private static function getKodeKategori($kategori)
    {
        return match($kategori) {
            'Tanah' => 'TNH',
            'Bangunan' => 'BGN',
            'Kendaraan' => 'KND',
            'Elektronik' => 'ELK',
            'Furniture' => 'FRN',
            'Alat Tulis Kantor' => 'ATK',
            'Alat Kesehatan' => 'AKS',
            'Alat Pertanian' => 'APT',
            'Alat Olahraga' => 'AOR',
            'Buku dan Perpustakaan' => 'BKU',
            default => 'INV'
        };
    }

    // Method untuk generate QR Code
    public function generateQrCode()
    {
        if (!$this->qr_code) {
            // Generate unique QR code data
            $qrData = json_encode([
                'kode' => $this->kode_inventaris,
                'nama' => $this->nama_barang,
                'kategori' => $this->kategori,
                'lokasi' => $this->lokasi_penyimpanan,
                'timestamp' => now()->timestamp
            ]);
            
            $this->qr_code = base64_encode($qrData);
            $this->save();
        }
        
        return $this->qr_code;
    }

    // Method untuk update nilai sekarang berdasarkan penyusutan
    public function updateNilaiSekarang($metodePenyusutan = 'linear', $umurEkonomis = 10)
    {
        if (!$this->tahun_perolehan || !$this->harga_perolehan) {
            return $this->nilai_sekarang;
        }

        $umurAset = now()->year - $this->tahun_perolehan;
        
        switch ($metodePenyusutan) {
            case 'linear':
                $penyusutanPerTahun = $this->harga_perolehan / $umurEkonomis;
                $totalPenyusutan = min($penyusutanPerTahun * $umurAset, $this->harga_perolehan);
                $this->nilai_sekarang = max(0, $this->harga_perolehan - $totalPenyusutan);
                break;
                
            case 'saldo_menurun':
                $tarifPenyusutan = 2 / $umurEkonomis; // Double declining balance
                $nilaiSisa = $this->harga_perolehan;
                for ($i = 0; $i < $umurAset; $i++) {
                    $nilaiSisa *= (1 - $tarifPenyusutan);
                }
                $this->nilai_sekarang = max(0, $nilaiSisa);
                break;
                
            default:
                // No depreciation
                $this->nilai_sekarang = $this->harga_perolehan;
        }
        
        $this->save();
        return $this->nilai_sekarang;
    }

    // Method untuk cek apakah perlu maintenance
    public function isPerluMaintenance()
    {
        return $this->jadwal_maintenance_berikutnya && 
               now()->isAfter($this->jadwal_maintenance_berikutnya) &&
               $this->status_pemakaian === 'Digunakan';
    }

    // Method untuk schedule maintenance berikutnya
    public function scheduleMaintenance($intervalBulan = 6)
    {
        $this->jadwal_maintenance_berikutnya = now()->addMonths($intervalBulan);
        $this->save();
    }

    // Method untuk record maintenance
    public function recordMaintenance($biaya = 0, $keterangan = null)
    {
        $this->terakhir_maintenance = now();
        $this->scheduleMaintenance();
        
        if ($biaya > 0) {
            $this->biaya_maintenance_per_tahun += $biaya;
        }
        
        if ($keterangan) {
            $this->keterangan = ($this->keterangan ?? '') . "\n" . 
                               now()->format('Y-m-d') . ": " . $keterangan;
        }
        
        $this->save();
    }

    // Attribute untuk badge colors
    public function getKondisiBadgeColorAttribute()
    {
        return match($this->kondisi) {
            'Baik' => 'success',
            'Rusak Ringan' => 'warning',
            'Rusak Berat' => 'danger',
            'Tidak Dapat Digunakan' => 'dark',
            default => 'secondary'
        };
    }

    public function getStatusBadgeColorAttribute()
    {
        return match($this->status_pemakaian) {
            'Digunakan' => 'success',
            'Tidak Digunakan' => 'secondary',
            'Dipinjamkan' => 'info',
            'Dalam Perbaikan' => 'warning',
            'Hilang' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk get total nilai inventaris
    public static function getTotalNilai()
    {
        return self::sum('nilai_sekarang');
    }

    // Method untuk get statistik kondisi
    public static function getStatistikKondisi()
    {
        return self::selectRaw('kondisi, COUNT(*) as total, SUM(nilai_sekarang) as nilai_total')
                  ->groupBy('kondisi')
                  ->get();
    }

    // Method untuk get inventaris yang perlu maintenance
    public static function getPerluMaintenance()
    {
        return self::perluMaintenance()
                  ->where('status_pemakaian', 'Digunakan')
                  ->orderBy('jadwal_maintenance_berikutnya')
                  ->get();
    }
} 