<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class NotifikasiWa extends Model
{
    use HasFactory;

    protected $table = 'notifikasi_wa';

    protected $fillable = [
        // New structure columns
        'nik',
        'phone_number',
        'message',
        'message_type',
        'is_read',
        'sent_at',
        'response_data',
        'status',
        
        // Legacy columns (kept for backward compatibility)
        'no_wa_legacy',
        'pesan_legacy',
        'jenis_legacy',
        'reference_id',
        'reference_type',
        'response',
        'tanggal_kirim'
    ];

    protected $casts = [
        'is_read' => 'boolean',
        'sent_at' => 'datetime',
        'tanggal_kirim' => 'datetime',
        'response_data' => 'array'
    ];

    // Relationships
    public function penduduk()
    {
        return $this->belongsTo(Penduduk::class, 'nik', 'nik');
    }

    public function reference()
    {
        return $this->morphTo();
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeUnread($query)
    {
        return $query->where('is_read', false);
    }

    public function scopeByMessageType($query, $type)
    {
        return $query->where('message_type', $type);
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', today());
    }

    public function scopeRecent($query, $days = 7)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }

    // Legacy scopes for backward compatibility
    public function scopeTerkirim($query)
    {
        return $query->where('status', 'Terkirim');
    }

    public function scopeGagal($query)
    {
        return $query->where('status', 'Gagal');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_legacy', $jenis);
    }

    // Accessors
    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'pending' => 'warning',
            'sent' => 'success',
            'failed' => 'danger',
            // Legacy statuses
            'Pending' => 'warning',
            'Terkirim' => 'success',
            'Gagal' => 'danger',
            default => 'secondary'
        };
    }

    public function getFormattedPhoneAttribute()
    {
        $phone = $this->phone_number ?: $this->no_wa_legacy;
        
        if (!$phone) return null;
        
        // Format phone number (add +62 if needed)
        if (!str_starts_with($phone, '+')) {
            if (str_starts_with($phone, '0')) {
                $phone = '+62' . substr($phone, 1);
            } elseif (str_starts_with($phone, '62')) {
                $phone = '+' . $phone;
            } else {
                $phone = '+62' . $phone;
            }
        }
        
        return $phone;
    }

    public function getMessageContentAttribute()
    {
        return $this->message ?: $this->pesan_legacy;
    }

    public function getTypeAttribute()
    {
        return $this->message_type ?: $this->jenis_legacy;
    }

    // Methods
    public function markAsRead()
    {
        $this->update(['is_read' => true]);
        return $this;
    }

    public function markAsSent($responseData = null)
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => now(),
            'response_data' => $responseData
        ]);
        return $this;
    }

    public function markAsFailed($responseData = null)
    {
        $this->update([
            'status' => 'failed',
            'response_data' => $responseData
        ]);
        return $this;
    }

    // Legacy methods for backward compatibility
    public function markAsTerkirim($response = null)
    {
        $this->update([
            'status' => 'sent', // Map to new status
            'response' => $response,
            'tanggal_kirim' => now(),
            'sent_at' => now()
        ]);
        return $this;
    }

    public function markAsGagal($response = null)
    {
        $this->update([
            'status' => 'failed', // Map to new status
            'response' => $response
        ]);
        return $this;
    }

    // Static helper methods
    public static function createNotification($data)
    {
        return static::create([
            'nik' => $data['nik'] ?? null,
            'phone_number' => $data['phone_number'] ?? $data['no_wa'] ?? null,
            'message' => $data['message'] ?? $data['pesan'] ?? null,
            'message_type' => $data['message_type'] ?? $data['jenis'] ?? 'general',
            'status' => $data['status'] ?? 'pending',
            'reference_id' => $data['reference_id'] ?? null,
            'reference_type' => $data['reference_type'] ?? null,
        ]);
    }

    public static function getUnreadCount()
    {
        return static::where('is_read', false)->count();
    }

    public static function getStatusCounts()
    {
        return [
            'total' => static::count(),
            'unread' => static::where('is_read', false)->count(),
            'pending' => static::where('status', 'pending')->count(),
            'sent' => static::where('status', 'sent')->count(),
            'failed' => static::where('status', 'failed')->count(),
            'today' => static::whereDate('created_at', today())->count(),
        ];
    }
} 
