<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Pbb extends Model
{
    use HasFactory;

    protected $table = 'pbb';

    protected $fillable = [
        'nop',
        'nik',
        'nama_pemilik',
        'alamat_objek',
        'dusun',
        'rt',
        'rw',
        'luas_tanah',
        'luas_bangunan',
        'jenis_tanah',
        'jenis_bangunan',
        'njop_tanah',
        'njop_bangunan',
        'njop_total',
        'njkp',
        'pbb_terhutang',
        'status_objek',
        'keterangan',
        'latitude',
        'longitude'
    ];

    protected $casts = [
        'luas_tanah' => 'decimal:2',
        'luas_bangunan' => 'decimal:2',
        'njop_tanah' => 'decimal:2',
        'njop_bangunan' => 'decimal:2',
        'njop_total' => 'decimal:2',
        'njkp' => 'decimal:2',
        'pbb_terhutang' => 'decimal:2',
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8'
    ];

    // Relationships
    public function penduduk()
    {
        return $this->belongsTo(Penduduk::class, 'nik', 'nik');
    }

    public function tagihanPbb()
    {
        return $this->hasMany(TagihanPbb::class);
    }

    // Scope methods
    public function scopeAktif($query)
    {
        return $query->where('status_objek', 'Aktif');
    }

    public function scopeByDusun($query, $dusun)
    {
        return $query->where('dusun', $dusun);
    }

    public function scopeByRtRw($query, $rt = null, $rw = null)
    {
        if ($rt) $query->where('rt', $rt);
        if ($rw) $query->where('rw', $rw);
        return $query;
    }

    // Method untuk hitung pajak
    public function hitungPajak($njopPerM2Tanah = null, $njopPerM2Bangunan = null)
    {
        // Update NJOP jika ada parameter baru
        if ($njopPerM2Tanah) {
            $this->njop_tanah = $this->luas_tanah * $njopPerM2Tanah;
        }
        
        if ($njopPerM2Bangunan) {
            $this->njop_bangunan = $this->luas_bangunan * $njopPerM2Bangunan;
        }

        // Hitung total NJOP
        $this->njop_total = $this->njop_tanah + $this->njop_bangunan;

        // Hitung NJKP (biasanya 20% dari NJOP untuk rumah tinggal)
        $persentaseNjkp = 0.20; // Bisa disesuaikan berdasarkan jenis objek
        $this->njkp = $this->njop_total * $persentaseNjkp;

        // Hitung PBB (0.5% dari NJKP)
        $tarifPbb = 0.005;
        $this->pbb_terhutang = $this->njkp * $tarifPbb;

        $this->save();

        return $this->pbb_terhutang;
    }

    // Method untuk generate tagihan tahunan
    public function generateTagihanTahunan($tahun = null)
    {
        if (!$tahun) {
            $tahun = now()->year;
        }

        // Cek apakah sudah ada tagihan untuk tahun ini
        $existingTagihan = $this->tagihanPbb()->where('tahun_pajak', $tahun)->first();
        if ($existingTagihan) {
            return $existingTagihan;
        }

        // Buat tagihan baru
        $tagihan = TagihanPbb::create([
            'pbb_id' => $this->id,
            'nop' => $this->nop,
            'tahun_pajak' => $tahun,
            'nilai_pajak' => $this->pbb_terhutang,
            'denda' => 0,
            'total_tagihan' => $this->pbb_terhutang,
            'jatuh_tempo' => now()->setMonth(8)->setDay(31), // 31 Agustus
            'sisa_tagihan' => $this->pbb_terhutang
        ]);

        return $tagihan;
    }

    // Method untuk validasi NOP
    public static function isValidNop($nop)
    {
        // NOP format: xx.xx.xxx.xxx.xxx-xxxx.x
        // Total 18 digit (tanpa titik dan strip)
        $cleanNop = preg_replace('/[^0-9]/', '', $nop);
        return strlen($cleanNop) === 18;
    }

    // Attribute untuk format NOP yang rapi
    public function getFormattedNopAttribute()
    {
        $nop = $this->nop;
        if (strlen($nop) === 18) {
            return substr($nop, 0, 2) . '.' . 
                   substr($nop, 2, 2) . '.' . 
                   substr($nop, 4, 3) . '.' . 
                   substr($nop, 7, 3) . '.' . 
                   substr($nop, 10, 3) . '-' . 
                   substr($nop, 13, 4) . '.' . 
                   substr($nop, 17, 1);
        }
        return $nop;
    }

    // Method untuk status pembayaran terkini
    public function getStatusPembayaranTerkini()
    {
        $tagihanTerkini = $this->tagihanPbb()
                              ->where('tahun_pajak', now()->year)
                              ->first();

        if (!$tagihanTerkini) {
            return 'Belum Ada Tagihan';
        }

        return $tagihanTerkini->status_bayar;
    }

    // Method untuk total tunggakan
    public function getTotalTunggakan()
    {
        return $this->tagihanPbb()
                   ->where('status_bayar', '!=', 'Lunas')
                   ->sum('sisa_tagihan');
    }
} 