<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;
use App\Helpers\SharedHostingHelper;

class ProdukUmkm extends Model
{
    use HasFactory;

    protected $table = 'produk_umkm';

    protected $fillable = [
        'umkm_id',
        'nama_produk',
        'sku',
        'deskripsi',
        'deskripsi_singkat',
        'kategori_produk',
        'tags',
        'harga_jual',
        'harga_modal',
        'harga_diskon',
        'tanggal_mulai_diskon',
        'tanggal_selesai_diskon',
        'stok',
        'minimal_stok',
        'unlimited_stok',
        'satuan',
        'berat',
        'dimensi',
        'warna',
        'ukuran',
        'bahan',
        'cara_penggunaan',
        'tanggal_kadaluarsa',
        'foto_utama',
        'galeri_foto',
        'video_produk',
        'sertifikat_produk',
        'bisa_dikirim',
        'hanya_cod',
        'ongkir_tambahan',
        'estimasi_pengiriman_hari',
        'catatan_pengiriman',
        'rating_rata_rata',
        'jumlah_review',
        'total_terjual',
        'view_count',
        'wishlist_count',
        'status',
        'is_featured',
        'is_bestseller',
        'is_new',
        'urutan_tampil',
        'slug',
        'meta_title',
        'meta_description',
        'keyword',
        'eligible_for_promo',
        'minimal_pembelian',
        'maksimal_pembelian',
        'syarat_ketentuan',
        'user_id'
    ];

    protected $casts = [
        'tags' => 'array',
        'dimensi' => 'array',
        'galeri_foto' => 'array',
        'sertifikat_produk' => 'array',
        'keyword' => 'array',
        'harga_jual' => 'decimal:2',
        'harga_modal' => 'decimal:2',
        'harga_diskon' => 'decimal:2',
        'ongkir_tambahan' => 'decimal:2',
        'rating_rata_rata' => 'decimal:2',
        'berat' => 'decimal:2',
        'tanggal_mulai_diskon' => 'date',
        'tanggal_selesai_diskon' => 'date',
        'tanggal_kadaluarsa' => 'date',
        'unlimited_stok' => 'boolean',
        'bisa_dikirim' => 'boolean',
        'hanya_cod' => 'boolean',
        'is_featured' => 'boolean',
        'is_bestseller' => 'boolean',
        'is_new' => 'boolean',
        'eligible_for_promo' => 'boolean',
    ];

    // Boot method untuk auto-generate slug dan SKU
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($produk) {
            if (empty($produk->slug)) {
                $produk->slug = Str::slug($produk->nama_produk);
                
                // Ensure unique slug
                $originalSlug = $produk->slug;
                $counter = 1;
                while (static::where('slug', $produk->slug)->exists()) {
                    $produk->slug = $originalSlug . '-' . $counter;
                    $counter++;
                }
            }
            
            if (empty($produk->sku)) {
                $produk->sku = 'PRD-' . strtoupper(Str::random(8));
            }
        });
    }

    // Relationships
    public function umkm(): BelongsTo
    {
        return $this->belongsTo(Umkm::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function transaksi()
    {
        return $this->morphMany(Transaksi::class, 'transactionable');
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeBestseller($query)
    {
        return $query->where('is_bestseller', true);
    }

    public function scopeAvailable($query)
    {
        return $query->where(function($q) {
            $q->where('unlimited_stok', true)
              ->orWhere('stok', '>', 0);
        });
    }

    public function scopeByKategori($query, $kategori)
    {
        return $query->where('kategori_produk', $kategori);
    }

    // Accessors & Mutators
    public function getFotoUtamaUrlAttribute()
    {
        return SharedHostingHelper::getProductPhotoUrl($this->foto_utama);
    }

    public function getGaleriFotoUrlsAttribute()
    {
        if ($this->galeri_foto) {
            return collect($this->galeri_foto)->map(function ($foto) {
                return SharedHostingHelper::getFileUrl('produk/galeri/' . $foto, 'images/default-umkm-logo.png');
            })->toArray();
        }
        return [];
    }

    public function getHargaAkhirAttribute()
    {
        if ($this->isDiskonAktif()) {
            return $this->harga_diskon;
        }
        return $this->harga_jual;
    }

    public function getPersentaseDiskonAttribute()
    {
        if ($this->isDiskonAktif()) {
            return round((($this->harga_jual - $this->harga_diskon) / $this->harga_jual) * 100);
        }
        return 0;
    }

    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'nonaktif' => '<span class="badge bg-secondary">Non-aktif</span>',
            'habis_stok' => '<span class="badge bg-danger">Habis Stok</span>',
            'pending_review' => '<span class="badge bg-warning">Pending Review</span>',
        ];
        
        return $badges[$this->status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getKategoriLabelAttribute()
    {
        $labels = [
            'makanan_segar' => 'Makanan Segar',
            'makanan_olahan' => 'Makanan Olahan',
            'minuman' => 'Minuman',
            'kerajinan_kayu' => 'Kerajinan Kayu',
            'kerajinan_bambu' => 'Kerajinan Bambu',
            'kerajinan_tekstil' => 'Kerajinan Tekstil',
            'fashion_pakaian' => 'Fashion Pakaian',
            'fashion_aksesoris' => 'Fashion Aksesoris',
            'elektronik' => 'Elektronik',
            'peralatan_rumah' => 'Peralatan Rumah',
            'kosmetik_kecantikan' => 'Kosmetik & Kecantikan',
            'obat_herbal' => 'Obat Herbal',
            'tanaman_hias' => 'Tanaman Hias',
            'hasil_pertanian' => 'Hasil Pertanian',
            'hasil_peternakan' => 'Hasil Peternakan',
            'jasa_digital' => 'Jasa Digital',
            'jasa_fisik' => 'Jasa Fisik',
            'lainnya' => 'Lainnya'
        ];
        
        return $labels[$this->kategori_produk] ?? 'Unknown';
    }

    // Methods
    public function isDiskonAktif()
    {
        if (!$this->harga_diskon || !$this->tanggal_mulai_diskon || !$this->tanggal_selesai_diskon) {
            return false;
        }
        
        $now = now()->toDateString();
        return $now >= $this->tanggal_mulai_diskon && $now <= $this->tanggal_selesai_diskon;
    }

    public function isAvailable()
    {
        return $this->unlimited_stok || $this->stok > 0;
    }

    public function isStokMenipis()
    {
        return !$this->unlimited_stok && $this->stok <= $this->minimal_stok;
    }

    public function incrementViewCount()
    {
        $this->increment('view_count');
    }

    public function incrementWishlist()
    {
        $this->increment('wishlist_count');
    }

    public function decrementWishlist()
    {
        $this->decrement('wishlist_count');
    }

    public function updateStok($jumlah, $operasi = 'kurang')
    {
        if ($this->unlimited_stok) {
            return true;
        }
        
        if ($operasi === 'kurang') {
            if ($this->stok >= $jumlah) {
                $this->decrement('stok', $jumlah);
                $this->increment('total_terjual', $jumlah);
                
                // Update status jika stok habis
                if ($this->stok <= 0) {
                    $this->update(['status' => 'habis_stok']);
                }
                
                return true;
            }
            return false;
        } else {
            $this->increment('stok', $jumlah);
            
            // Update status jika stok tersedia kembali
            if ($this->status === 'habis_stok' && $this->stok > 0) {
                $this->update(['status' => 'aktif']);
            }
            
            return true;
        }
    }

    public function updateRating($rating, $increment = true)
    {
        if ($increment) {
            $totalRating = ($this->rating_rata_rata * $this->jumlah_review) + $rating;
            $newJumlahReview = $this->jumlah_review + 1;
            $newRating = $totalRating / $newJumlahReview;
            
            $this->update([
                'rating_rata_rata' => round($newRating, 2),
                'jumlah_review' => $newJumlahReview
            ]);
        }
        
        // Update rating UMKM
        $this->umkm->updateRating();
    }

    public function getTotalPendapatan()
    {
        return $this->transaksi()
            ->where('status', 'PAID')
            ->sum('amount');
    }
}
