<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class TagihanPbb extends Model
{
    use HasFactory;

    protected $table = 'tagihan_pbb';

    protected $fillable = [
        'pbb_id',
        'nop',
        'tahun_pajak',
        'nilai_pajak',
        'denda',
        'total_tagihan',
        'status_bayar',
        'jatuh_tempo',
        'tanggal_bayar',
        'jumlah_bayar',
        'sisa_tagihan',
        'metode_bayar',
        'referensi_bayar',
        'bukti_bayar',
        'nomor_kwitansi',
        'file_kwitansi',
        'keterangan'
    ];

    protected $casts = [
        'nilai_pajak' => 'decimal:2',
        'denda' => 'decimal:2',
        'total_tagihan' => 'decimal:2',
        'jumlah_bayar' => 'decimal:2',
        'sisa_tagihan' => 'decimal:2',
        'jatuh_tempo' => 'date',
        'tanggal_bayar' => 'date'
    ];

    // Relationships
    public function pbb()
    {
        return $this->belongsTo(Pbb::class);
    }

    // Scope methods
    public function scopeBelumLunas($query)
    {
        return $query->where('status_bayar', '!=', 'Lunas');
    }

    public function scopeLunas($query)
    {
        return $query->where('status_bayar', 'Lunas');
    }

    public function scopeJatuhTempo($query)
    {
        return $query->where('jatuh_tempo', '<', now())->where('status_bayar', '!=', 'Lunas');
    }

    public function scopeTahunPajak($query, $tahun)
    {
        return $query->where('tahun_pajak', $tahun);
    }

    // Method untuk hitung denda
    public function hitungDenda()
    {
        if ($this->status_bayar === 'Lunas' || !$this->jatuh_tempo) {
            return 0;
        }

        $hariTerlambat = now()->diffInDays($this->jatuh_tempo, false);
        
        if ($hariTerlambat <= 0) {
            return 0; // Belum jatuh tempo
        }

        // Denda 2% per bulan, maksimal 24 bulan
        $bulanTerlambat = ceil($hariTerlambat / 30);
        $bulanTerlambat = min($bulanTerlambat, 24);
        
        $denda = $this->nilai_pajak * 0.02 * $bulanTerlambat;
        
        // Update denda dan total tagihan
        $this->denda = $denda;
        $this->total_tagihan = $this->nilai_pajak + $denda;
        $this->sisa_tagihan = $this->total_tagihan - $this->jumlah_bayar;
        $this->save();

        return $denda;
    }

    // Method untuk proses pembayaran
    public function prosesPembayaran($jumlahBayar, $metodeBayar, $referensi = null)
    {
        // Hitung denda terkini
        $this->hitungDenda();

        $this->jumlah_bayar += $jumlahBayar;
        $this->sisa_tagihan = $this->total_tagihan - $this->jumlah_bayar;
        
        if ($this->sisa_tagihan <= 0) {
            $this->status_bayar = 'Lunas';
            $this->tanggal_bayar = now();
            $this->sisa_tagihan = 0;
        } else if ($this->jumlah_bayar > 0) {
            $this->status_bayar = 'Dicicil';
        }

        $this->metode_bayar = $metodeBayar;
        $this->referensi_bayar = $referensi;
        $this->save();

        // Generate kwitansi jika lunas
        if ($this->status_bayar === 'Lunas') {
            $this->generateKwitansi();
        }

        return $this;
    }

    // Method untuk generate nomor kwitansi
    public function generateKwitansi()
    {
        if (!$this->nomor_kwitansi) {
            $counter = self::whereNotNull('nomor_kwitansi')
                          ->whereYear('created_at', now()->year)
                          ->count() + 1;
            
            $this->nomor_kwitansi = 'KWT-PBB-' . now()->format('Y') . '-' . str_pad($counter, 4, '0', STR_PAD_LEFT);
            $this->save();
        }

        return $this->nomor_kwitansi;
    }

    // Attribute untuk status badge color
    public function getStatusBadgeColorAttribute()
    {
        return match($this->status_bayar) {
            'Lunas' => 'success',
            'Dicicil' => 'warning',
            'Belum Lunas' => 'danger',
            default => 'secondary'
        };
    }

    // Method untuk cek apakah sudah jatuh tempo
    public function isJatuhTempo()
    {
        return $this->jatuh_tempo && now()->isAfter($this->jatuh_tempo) && $this->status_bayar !== 'Lunas';
    }

    // Method untuk sisa hari sebelum jatuh tempo
    public function sisaHariJatuhTempo()
    {
        if (!$this->jatuh_tempo || $this->status_bayar === 'Lunas') {
            return null;
        }

        return now()->diffInDays($this->jatuh_tempo, false);
    }
} 