<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Str;

class Transaksi extends Model
{
    use HasFactory;

    protected $table = 'transaksi';

    protected $fillable = [
        'kode_transaksi',
        'merchant_ref',
        'tripay_reference',
        'jenis_transaksi',
        'transactionable_type',
        'transactionable_id',
        'customer_name',
        'customer_email',
        'customer_phone',
        'customer_address',
        'amount',
        'fee_merchant',
        'fee_customer',
        'total_fee',
        'amount_received',
        'payment_method',
        'payment_method_name',
        'payment_name',
        'payment_code',
        'checkout_url',
        'qr_url',
        'status',
        'paid_at',
        'expired_time',
        'settled_at',
        'payment_timeout',
        'callback_data',
        'callback_received_at',
        'callback_signature',
        'is_callback_verified',
        'payment_instructions',
        'order_items',
        'note',
        'cancel_reason',
        'refund_amount',
        'refund_at',
        'refund_reference',
        'refund_reason',
        'user_agent',
        'ip_address',
        'source',
        'metadata',
        'email_sent',
        'whatsapp_sent',
        'notification_sent_at',
        'notification_attempts',
        'user_id',
        'admin_notes',
        'verified_at',
        'verified_by'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'fee_merchant' => 'decimal:2',
        'fee_customer' => 'decimal:2',
        'total_fee' => 'decimal:2',
        'amount_received' => 'decimal:2',
        'refund_amount' => 'decimal:2',
        'paid_at' => 'datetime',
        'expired_time' => 'datetime',
        'settled_at' => 'datetime',
        'callback_received_at' => 'datetime',
        'refund_at' => 'datetime',
        'notification_sent_at' => 'datetime',
        'verified_at' => 'datetime',
        'callback_data' => 'array',
        'payment_instructions' => 'array',
        'order_items' => 'array',
        'metadata' => 'array',
        'is_callback_verified' => 'boolean',
        'email_sent' => 'boolean',
        'whatsapp_sent' => 'boolean',
    ];

    // Boot method untuk auto-generate kode transaksi
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($transaksi) {
            if (empty($transaksi->kode_transaksi)) {
                $transaksi->kode_transaksi = 'TRX-' . date('Ymd') . '-' . strtoupper(Str::random(8));
                
                // Ensure unique transaction code
                while (static::where('kode_transaksi', $transaksi->kode_transaksi)->exists()) {
                    $transaksi->kode_transaksi = 'TRX-' . date('Ymd') . '-' . strtoupper(Str::random(8));
                }
            }
            
            if (empty($transaksi->merchant_ref)) {
                $transaksi->merchant_ref = 'MRC-' . time() . '-' . strtoupper(Str::random(6));
            }
            
            // Set default expired time (24 hours)
            if (empty($transaksi->expired_time)) {
                $transaksi->expired_time = now()->addHours(24);
            }
            
            // Set default source
            if (empty($transaksi->source)) {
                $transaksi->source = 'website';
            }
            
            // Set user agent and IP
            if (empty($transaksi->user_agent)) {
                $transaksi->user_agent = request()->header('User-Agent');
            }
            
            if (empty($transaksi->ip_address)) {
                $transaksi->ip_address = request()->ip();
            }
        });
    }

    // Relationships
    public function transactionable(): MorphTo
    {
        return $this->morphTo();
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function verifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    // Scopes
    public function scopePaid($query)
    {
        return $query->where('status', 'PAID');
    }

    public function scopeUnpaid($query)
    {
        return $query->where('status', 'UNPAID');
    }

    public function scopeSettled($query)
    {
        return $query->where('status', 'SETTLED');
    }

    public function scopeExpired($query)
    {
        return $query->where('status', 'EXPIRED');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'FAILED');
    }

    public function scopeCancelled($query)
    {
        return $query->where('status', 'CANCELLED');
    }

    public function scopeRefund($query)
    {
        return $query->where('status', 'REFUND');
    }

    public function scopeByJenisTransaksi($query, $jenis)
    {
        return $query->where('jenis_transaksi', $jenis);
    }

    public function scopeByPaymentMethod($query, $method)
    {
        return $query->where('payment_method', $method);
    }

    public function scopeToday($query)
    {
        return $query->whereDate('created_at', now()->toDateString());
    }

    public function scopeThisMonth($query)
    {
        return $query->whereMonth('created_at', now()->month)
                    ->whereYear('created_at', now()->year);
    }

    public function scopeThisYear($query)
    {
        return $query->whereYear('created_at', now()->year);
    }

    public function scopeByDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('created_at', [$startDate, $endDate]);
    }

    // Accessors & Mutators
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'UNPAID' => '<span class="badge bg-warning">Belum Bayar</span>',
            'PAID' => '<span class="badge bg-success">Sudah Bayar</span>',
            'SETTLED' => '<span class="badge bg-primary">Settled</span>',
            'EXPIRED' => '<span class="badge bg-danger">Expired</span>',
            'FAILED' => '<span class="badge bg-danger">Gagal</span>',
            'CANCELLED' => '<span class="badge bg-secondary">Dibatalkan</span>',
            'REFUND' => '<span class="badge bg-info">Refund</span>',
        ];
        
        return $badges[$this->status] ?? '<span class="badge bg-secondary">Unknown</span>';
    }

    public function getJenisTransaksiLabelAttribute()
    {
        $labels = [
            'tiket_wisata' => 'Tiket Wisata',
            'produk_umkm' => 'Produk UMKM',
            'marketplace' => 'Marketplace Desa',
            'donasi' => 'Donasi',
            'layanan_desa' => 'Layanan Desa',
            'lainnya' => 'Lainnya'
        ];
        
        return $labels[$this->jenis_transaksi] ?? 'Unknown';
    }

    public function getPaymentMethodLabelAttribute()
    {
        $labels = [
            'MYBVA' => 'Maybank VA',
            'PERMATAVA' => 'Permata VA',
            'BNIVA' => 'BNI VA',
            'BRIVA' => 'BRI VA',
            'MANDIRIVA' => 'Mandiri VA',
            'BCAVA' => 'BCA VA',
            'SMSVA' => 'SMS VA',
            'MUAMALATVA' => 'Muamalat VA',
            'CIMBVA' => 'CIMB VA',
            'BSIVA' => 'BSI VA',
            'ALFAMART' => 'Alfamart',
            'INDOMARET' => 'Indomaret',
            'ALFAMIDI' => 'Alfamidi',
            'OVO' => 'OVO',
            'DANA' => 'DANA',
            'SHOPEEPAY' => 'ShopeePay',
            'LINKAJA' => 'LinkAja',
            'GOPAY' => 'GoPay',
            'QRIS' => 'QRIS',
            'QRISC' => 'QRIS (Customizable)',
            'QRISD' => 'QRIS (Dynamic)',
        ];
        
        return $labels[$this->payment_method] ?? $this->payment_method_name ?? 'Unknown';
    }

    public function getTotalAmountAttribute()
    {
        return $this->amount + $this->fee_customer;
    }

    public function getNetAmountAttribute()
    {
        return $this->amount - $this->fee_merchant;
    }

    public function getIsExpiredAttribute()
    {
        return $this->expired_time && now()->gt($this->expired_time) && $this->status === 'UNPAID';
    }

    public function getIsPaidAttribute()
    {
        return in_array($this->status, ['PAID', 'SETTLED']);
    }

    public function getIsRefundableAttribute()
    {
        return $this->is_paid && !$this->refund_at;
    }

    public function getFormattedAmountAttribute()
    {
        return 'Rp ' . number_format($this->amount, 0, ',', '.');
    }

    public function getFormattedTotalAmountAttribute()
    {
        return 'Rp ' . number_format($this->total_amount, 0, ',', '.');
    }

    public function getPaymentInstructionsTextAttribute()
    {
        if (!$this->payment_instructions) {
            return null;
        }
        
        $instructions = [];
        foreach ($this->payment_instructions as $instruction) {
            if (isset($instruction['title']) && isset($instruction['steps'])) {
                $instructions[] = $instruction['title'] . ': ' . implode(', ', $instruction['steps']);
            }
        }
        
        return implode(' | ', $instructions);
    }

    // Methods
    public function markAsPaid($paidAt = null, $reference = null)
    {
        $this->update([
            'status' => 'PAID',
            'paid_at' => $paidAt ?? now(),
            'tripay_reference' => $reference ?? $this->tripay_reference
        ]);
        
        // Trigger post-payment actions
        $this->handlePostPayment();
        
        return true;
    }

    public function markAsSettled($settledAt = null)
    {
        if (!$this->is_paid) {
            return false;
        }
        
        $this->update([
            'status' => 'SETTLED',
            'settled_at' => $settledAt ?? now()
        ]);
        
        return true;
    }

    public function markAsExpired()
    {
        if ($this->status !== 'UNPAID') {
            return false;
        }
        
        $this->update(['status' => 'EXPIRED']);
        
        // Handle post-expiry actions
        $this->handlePostExpiry();
        
        return true;
    }

    public function markAsFailed($reason = null)
    {
        $this->update([
            'status' => 'FAILED',
            'cancel_reason' => $reason
        ]);
        
        return true;
    }

    public function cancel($reason = null)
    {
        if ($this->is_paid) {
            return false;
        }
        
        $this->update([
            'status' => 'CANCELLED',
            'cancel_reason' => $reason
        ]);
        
        return true;
    }

    public function refund($amount = null, $reason = null, $reference = null)
    {
        if (!$this->is_refundable) {
            return false;
        }
        
        $refundAmount = $amount ?? $this->amount;
        
        $this->update([
            'status' => 'REFUND',
            'refund_amount' => $refundAmount,
            'refund_at' => now(),
            'refund_reason' => $reason,
            'refund_reference' => $reference
        ]);
        
        // Handle post-refund actions
        $this->handlePostRefund();
        
        return true;
    }

    public function updateFromCallback($callbackData, $signature = null)
    {
        $this->update([
            'callback_data' => $callbackData,
            'callback_received_at' => now(),
            'callback_signature' => $signature,
            'is_callback_verified' => true
        ]);
        
        // Update status based on callback
        if (isset($callbackData['status'])) {
            $this->updateStatusFromCallback($callbackData['status']);
        }
        
        return true;
    }

    public function updateStatusFromCallback($status)
    {
        switch (strtoupper($status)) {
            case 'PAID':
                $this->markAsPaid();
                break;
            case 'SETTLED':
                $this->markAsSettled();
                break;
            case 'EXPIRED':
                $this->markAsExpired();
                break;
            case 'FAILED':
                $this->markAsFailed();
                break;
        }
    }

    public function sendNotification($type = 'payment_created')
    {
        // Implementation for sending notifications
        switch ($type) {
            case 'payment_created':
                // Send payment link
                break;
            case 'payment_success':
                // Send payment success notification
                break;
            case 'payment_expired':
                // Send payment expired notification
                break;
            case 'payment_failed':
                // Send payment failed notification
                break;
        }
        
        $this->update([
            'notification_sent_at' => now(),
            'notification_attempts' => $this->notification_attempts + 1
        ]);
    }

    public function sendWhatsAppNotification($message)
    {
        // Implementation for WhatsApp notification
        $this->update(['whatsapp_sent' => true]);
    }

    public function sendEmailNotification($subject, $message)
    {
        // Implementation for email notification
        $this->update(['email_sent' => true]);
    }

    public function extendExpiry($hours = 24)
    {
        if ($this->status !== 'UNPAID') {
            return false;
        }
        
        $this->update(['expired_time' => now()->addHours($hours)]);
        return true;
    }

    public function addAdminNote($note, $userId = null)
    {
        $existingNotes = $this->admin_notes ? $this->admin_notes . "\n\n" : '';
        $timestamp = now()->format('Y-m-d H:i:s');
        $user = $userId ? " (User ID: {$userId})" : '';
        
        $this->update([
            'admin_notes' => $existingNotes . "[{$timestamp}]{$user}: {$note}"
        ]);
    }

    public function verify($userId = null)
    {
        $this->update([
            'verified_at' => now(),
            'verified_by' => $userId
        ]);
    }

    // Private methods for handling post-transaction actions
    private function handlePostPayment()
    {
        // Update related model (e.g., ticket status, product stock)
        if ($this->transactionable) {
            switch ($this->jenis_transaksi) {
                case 'tiket_wisata':
                    $this->transactionable->update(['status_tiket' => 'paid']);
                    break;
                case 'produk_umkm':
                    // Update stock, sales count, etc.
                    break;
            }
        }
        
        // Send success notification
        $this->sendNotification('payment_success');
    }

    private function handlePostExpiry()
    {
        // Handle expired transaction
        if ($this->transactionable) {
            switch ($this->jenis_transaksi) {
                case 'tiket_wisata':
                    $this->transactionable->update(['status_tiket' => 'expired']);
                    break;
            }
        }
        
        // Send expiry notification
        $this->sendNotification('payment_expired');
    }

    private function handlePostRefund()
    {
        // Handle refund actions
        if ($this->transactionable) {
            switch ($this->jenis_transaksi) {
                case 'tiket_wisata':
                    $this->transactionable->update(['status_tiket' => 'refunded']);
                    break;
                case 'produk_umkm':
                    // Restore stock if needed
                    break;
            }
        }
    }

    // Static methods for statistics
    public static function getTotalRevenue($dateRange = null)
    {
        $query = static::paid();
        
        if ($dateRange) {
            $query->byDateRange($dateRange['start'], $dateRange['end']);
        }
        
        return $query->sum('amount');
    }

    public static function getTransactionCount($status = null, $dateRange = null)
    {
        $query = static::query();
        
        if ($status) {
            $query->where('status', $status);
        }
        
        if ($dateRange) {
            $query->byDateRange($dateRange['start'], $dateRange['end']);
        }
        
        return $query->count();
    }

    public static function getPopularPaymentMethods($limit = 10)
    {
        return static::paid()
            ->selectRaw('payment_method, payment_method_name, COUNT(*) as total')
            ->groupBy('payment_method', 'payment_method_name')
            ->orderByDesc('total')
            ->limit($limit)
            ->get();
    }

    public static function getDailyStats($date = null)
    {
        $date = $date ?? now()->toDateString();
        
        return [
            'total_transactions' => static::whereDate('created_at', $date)->count(),
            'total_revenue' => static::whereDate('created_at', $date)->paid()->sum('amount'),
            'success_rate' => static::whereDate('created_at', $date)->paid()->count() / 
                            max(static::whereDate('created_at', $date)->count(), 1) * 100,
            'average_amount' => static::whereDate('created_at', $date)->paid()->avg('amount')
        ];
    }
}
