<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasApiTokens;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'role',
        'role_id',
        'is_active',
        'is_admin',
        'last_login_at',
        'phone',
        'alamat',
        'tanggal_lahir',
        'jenis_kelamin',
        'nik',
        'verified_at',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'is_active' => 'boolean',
            'is_admin' => 'boolean',
            'last_login_at' => 'datetime',
            'tanggal_lahir' => 'date',
            'verified_at' => 'datetime',
        ];
    }

    /**
     * Get the role that owns the user
     */
    public function userRole()
    {
        return $this->belongsTo(Role::class, 'role_id');
    }

    /**
     * Check if user has permission
     */
    public function hasPermission($permission)
    {
        if (!$this->userRole) {
            return false;
        }
        
        return $this->userRole->hasPermission($permission);
    }

    /**
     * Check if user has role
     */
    public function hasRole($roleName)
    {
        if ($this->role === $roleName) {
            return true;
        }
        
        if (!$this->userRole) {
            return false;
        }
        
        return $this->userRole->name === $roleName;
    }

    /**
     * Check if user is super admin
     */
    public function isSuperAdmin()
    {
        return in_array($this->role, ['superadmin', 'super_admin']) || 
               ($this->userRole && $this->userRole->name === 'superadmin');
    }

    /**
     * Check if user is admin
     */
    public function isAdmin()
    {
        return $this->is_admin || 
               $this->isSuperAdmin() || 
               $this->hasRole('admin') || 
               in_array($this->role, ['admin', 'superadmin', 'super_admin']);
    }

    /**
     * Check if user can access admin panel
     */
    public function canAccessAdmin()
    {
        return $this->is_active && $this->isAdmin();
    }

    /**
     * Assign role to user
     */
    public function assignRole($roleName)
    {
        $this->role = $roleName;
        
        // Find role in roles table if exists
        if (class_exists('App\\Models\\Role')) {
            $roleModel = \App\Models\Role::where('name', $roleName)->first();
            if ($roleModel) {
                $this->role_id = $roleModel->id;
            }
        }
        
        // Set admin flags
        if (in_array($roleName, ['admin', 'superadmin', 'super_admin'])) {
            $this->is_admin = true;
        }
        
        $this->save();
        return $this;
    }

    /**
     * Get user's display role
     */
    public function getDisplayRoleAttribute()
    {
        if ($this->userRole) {
            return $this->userRole->display_name ?? $this->userRole->name;
        }
        
        return ucfirst($this->role ?? 'user');
    }

    /**
     * Boot the model
     */
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($user) {
            if (empty($user->is_active)) {
                $user->is_active = true;
            }
            
            if (empty($user->role)) {
                $user->role = 'user';
            }
        });
    }
}
