<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class WilayahAdministratif extends Model
{
    use HasFactory;

    protected $table = 'wilayah_administratif';

    protected $fillable = [
        'jenis_wilayah',
        'kode_wilayah',
        'nama_wilayah',
        'parent_id',
        'kepala_wilayah',
        'nik_kepala',
        'nomor_hp_kepala',
        'alamat_kantor',
        'latitude',
        'longitude',
        'batas_wilayah',
        'luas_wilayah',
        'jumlah_kk',
        'jumlah_penduduk',
        'status',
        'keterangan',
        'urutan_tampil',
        'user_id'
    ];

    protected $casts = [
        'latitude' => 'decimal:8',
        'longitude' => 'decimal:8',
        'luas_wilayah' => 'decimal:2',
        'jumlah_kk' => 'integer',
        'jumlah_penduduk' => 'integer',
        'urutan_tampil' => 'integer'
    ];

    // Relationships
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(WilayahAdministratif::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(WilayahAdministratif::class, 'parent_id');
    }

    public function penduduk(): HasMany
    {
        return $this->hasMany(Penduduk::class, 'wilayah_id');
    }

    // Scopes
    public function scopeAktif($query)
    {
        return $query->where('status', 'aktif');
    }

    public function scopeByJenis($query, $jenis)
    {
        return $query->where('jenis_wilayah', $jenis);
    }

    public function scopeDusun($query)
    {
        return $query->where('jenis_wilayah', 'dusun');
    }

    public function scopeRw($query)
    {
        return $query->where('jenis_wilayah', 'rw');
    }

    public function scopeRt($query)
    {
        return $query->where('jenis_wilayah', 'rt');
    }

    public function scopeOrderByUrutan($query)
    {
        return $query->orderBy('urutan_tampil', 'asc');
    }

    // Accessors
    public function getStatusBadgeAttribute()
    {
        $badges = [
            'aktif' => '<span class="badge bg-success">Aktif</span>',
            'nonaktif' => '<span class="badge bg-secondary">Non-Aktif</span>'
        ];

        return $badges[$this->status] ?? '';
    }

    public function getJenisWilayahBadgeAttribute()
    {
        $badges = [
            'dusun' => '<span class="badge bg-primary">Dusun</span>',
            'rw' => '<span class="badge bg-info">RW</span>',
            'rt' => '<span class="badge bg-warning">RT</span>'
        ];

        return $badges[$this->jenis_wilayah] ?? '';
    }

    public function getFormattedLuasWilayahAttribute()
    {
        if ($this->luas_wilayah) {
            return number_format($this->luas_wilayah, 2, ',', '.') . ' Ha';
        }
        return '-';
    }

    public function getKepadatanPendudukAttribute()
    {
        if ($this->luas_wilayah > 0) {
            return round($this->jumlah_penduduk / $this->luas_wilayah, 2);
        }
        return 0;
    }

    public function getNamaLengkapAttribute()
    {
        $prefix = strtoupper($this->jenis_wilayah);
        return $prefix . ' ' . $this->nama_wilayah;
    }

    public function getHierarkiAttribute()
    {
        $hierarki = [];
        
        if ($this->parent) {
            $hierarki[] = $this->parent->nama_lengkap;
            if ($this->parent->parent) {
                $hierarki[] = $this->parent->parent->nama_lengkap;
            }
        }
        
        $hierarki[] = $this->nama_lengkap;
        
        return implode(' > ', array_reverse($hierarki));
    }

    // Methods
    public function updateStatistikPenduduk()
    {
        $jumlahKK = 0;
        $jumlahPenduduk = 0;

        if ($this->jenis_wilayah === 'dusun') {
            // Hitung langsung dari data penduduk berdasarkan dusun
            $jumlahKK = Keluarga::where('dusun', $this->nama_wilayah)->count();
            $jumlahPenduduk = Penduduk::where('dusun', $this->nama_wilayah)
                                    ->where('status_hidup', 'hidup')
                                    ->count();
        } elseif ($this->jenis_wilayah === 'rw') {
            // Hitung langsung dari data penduduk berdasarkan RW
            // Cari dusun parent untuk filter yang tepat
            $dusunParent = $this->parent ? $this->parent->nama_wilayah : null;
            
            $kkQuery = Keluarga::where('rw', $this->nama_wilayah);
            $pendudukQuery = Penduduk::where('rw', $this->nama_wilayah)->where('status_hidup', 'hidup');
            
            if ($dusunParent) {
                $kkQuery->where('dusun', $dusunParent);
                $pendudukQuery->where('dusun', $dusunParent);
            }
            
            $jumlahKK = $kkQuery->count();
            $jumlahPenduduk = $pendudukQuery->count();
        } else {
            // RT: hitung langsung dari data penduduk berdasarkan RT
            // Cari RW parent dan dusun untuk filter yang tepat
            $rwParent = $this->parent ? $this->parent->nama_wilayah : null;
            $dusunParent = $this->parent && $this->parent->parent ? $this->parent->parent->nama_wilayah : null;
            
            $kkQuery = Keluarga::where('rt', $this->nama_wilayah);
            $pendudukQuery = Penduduk::where('rt', $this->nama_wilayah)->where('status_hidup', 'hidup');
            
            if ($rwParent) {
                $kkQuery->where('rw', $rwParent);
                $pendudukQuery->where('rw', $rwParent);
            }
            
            if ($dusunParent) {
                $kkQuery->where('dusun', $dusunParent);
                $pendudukQuery->where('dusun', $dusunParent);
            }
            
            $jumlahKK = $kkQuery->count();
            $jumlahPenduduk = $pendudukQuery->count();
        }

        $this->update([
            'jumlah_kk' => $jumlahKK,
            'jumlah_penduduk' => $jumlahPenduduk
        ]);
        
        // Clear cache after update
        \Illuminate\Support\Facades\Cache::forget('wilayah_statistik');
        
        return [
            'jumlah_kk' => $jumlahKK,
            'jumlah_penduduk' => $jumlahPenduduk
        ];
    }

    public function nonaktifkan()
    {
        $this->update(['status' => 'nonaktif']);
        
        // Nonaktifkan juga anak-anaknya
        $this->children()->update(['status' => 'nonaktif']);
    }

    public function aktifkan()
    {
        $this->update(['status' => 'aktif']);
    }

    public static function getJenisWilayahOptions()
    {
        return [
            'dusun' => 'Dusun',
            'rw' => 'RW (Rukun Warga)',
            'rt' => 'RT (Rukun Tetangga)'
        ];
    }

    public static function getStatusOptions()
    {
        return [
            'aktif' => 'Aktif',
            'nonaktif' => 'Non-Aktif'
        ];
    }

    public static function getTreeStructure()
    {
        $dusun = self::dusun()->aktif()->orderByUrutan()->get();
        $tree = [];

        foreach ($dusun as $dus) {
            $tree[$dus->id] = [
                'data' => $dus,
                'rw' => []
            ];

            $rwList = $dus->children()->rw()->aktif()->orderByUrutan()->get();
            foreach ($rwList as $rw) {
                $tree[$dus->id]['rw'][$rw->id] = [
                    'data' => $rw,
                    'rt' => $rw->children()->rt()->aktif()->orderByUrutan()->get()
                ];
            }
        }

        return $tree;
    }

    public static function getStatistikWilayah()
    {
        return \Illuminate\Support\Facades\Cache::remember('wilayah_statistik', 3600, function () {
            return [
                'total_dusun' => self::dusun()->aktif()->count(),
                'total_rw' => self::rw()->aktif()->count(),
                'total_rt' => self::rt()->aktif()->count(),
                'total_kk' => self::aktif()->sum('jumlah_kk'),
                'total_penduduk' => self::aktif()->sum('jumlah_penduduk'),
                'rata_rata_kk_per_rt' => self::rt()->aktif()->avg('jumlah_kk'),
                'rata_rata_penduduk_per_rt' => self::rt()->aktif()->avg('jumlah_penduduk'),
                'luas_total' => self::aktif()->sum('luas_wilayah')
            ];
        });
    }

    /**
     * Get real-time statistics directly from penduduk and keluarga tables
     */
    public static function getStatistikRealTime()
    {
        // Get real counts directly from source tables
        $totalKKReal = Keluarga::count();
        $totalPendudukReal = Penduduk::where('status_hidup', 'hidup')->count();
        
        // Get counts from wilayah_administratif table (stored/cached data)
        $totalKKWilayah = self::aktif()->sum('jumlah_kk');
        $totalPendudukWilayah = self::aktif()->sum('jumlah_penduduk');
        
        return [
            'realtime' => [
                'total_kk_real' => $totalKKReal,
                'total_penduduk_real' => $totalPendudukReal,
                'total_kk_wilayah' => $totalKKWilayah,
                'total_penduduk_wilayah' => $totalPendudukWilayah,
                'selisih_kk' => $totalKKReal - $totalKKWilayah,
                'selisih_penduduk' => $totalPendudukReal - $totalPendudukWilayah,
                'sinkron' => ($totalKKReal == $totalKKWilayah && $totalPendudukReal == $totalPendudukWilayah)
            ],
            'detail_per_dusun' => self::getDetailPerDusun()
        ];
    }

    /**
     * Get detailed statistics per dusun
     */
    private static function getDetailPerDusun()
    {
        $dusunList = self::dusun()->aktif()->get();
        $detail = [];
        
        foreach ($dusunList as $dusun) {
            $kkReal = Keluarga::where('dusun', $dusun->nama_wilayah)->count();
            $pendudukReal = Penduduk::where('dusun', $dusun->nama_wilayah)
                                  ->where('status_hidup', 'hidup')
                                  ->count();
            
            $detail[] = [
                'nama_dusun' => $dusun->nama_wilayah,
                'kk_real' => $kkReal,
                'penduduk_real' => $pendudukReal,
                'kk_tersimpan' => $dusun->jumlah_kk,
                'penduduk_tersimpan' => $dusun->jumlah_penduduk,
                'selisih_kk' => $kkReal - $dusun->jumlah_kk,
                'selisih_penduduk' => $pendudukReal - $dusun->jumlah_penduduk
            ];
        }
        
        return $detail;
    }

    public static function getWilayahByJenis($jenis)
    {
        return self::where('jenis_wilayah', $jenis)
                   ->aktif()
                   ->orderByUrutan()
                   ->with(['parent', 'children'])
                   ->get();
    }

    public static function getDusunWithRwRt()
    {
        return self::dusun()
                   ->aktif()
                   ->orderByUrutan()
                   ->with(['children' => function($query) {
                       $query->rw()->aktif()->orderByUrutan()->with(['children' => function($q) {
                           $q->rt()->aktif()->orderByUrutan();
                       }]);
                   }])
                   ->get();
    }


} 