<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class OttopayService
{
    private $baseUrl;
    private $apiKey;
    private $secretKey;
    private $merchantCode;
    private $mode;

    public function __construct()
    {
        $settings = \App\Models\Setting::first();
        
        $this->apiKey = $settings->ottopay_api_key ?? config('ottopay.api_key');
        $this->secretKey = $settings->ottopay_secret_key ?? config('ottopay.secret_key');
        $this->merchantCode = $settings->ottopay_merchant_code ?? config('ottopay.merchant_code');
        $this->mode = $settings->ottopay_mode ?? config('ottopay.mode', 'sandbox');
        
        $this->baseUrl = $this->mode === 'production' 
            ? 'https://api.ottopay.id' 
            : 'https://sandbox-api.ottopay.id';
    }

    /**
     * Test koneksi ke OttoPay
     */
    public function testConnection()
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ])->get($this->baseUrl . '/api/v1/merchant/profile');

            if ($response->successful()) {
                return [
                    'success' => true,
                    'message' => 'Koneksi berhasil',
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Koneksi gagal: ' . $response->body()
            ];
        } catch (\Exception $e) {
            Log::error('OttoPay connection test failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Buat transaksi pembayaran
     */
    public function createPayment($data)
    {
        try {
            $payload = [
                'merchant_code' => $this->merchantCode,
                'order_id' => $data['order_id'],
                'amount' => $data['amount'],
                'currency' => 'IDR',
                'description' => $data['description'] ?? 'Pembayaran',
                'customer_name' => $data['customer_name'],
                'customer_email' => $data['customer_email'],
                'customer_phone' => $data['customer_phone'] ?? null,
                'return_url' => $data['return_url'] ?? null,
                'callback_url' => $data['callback_url'] ?? null,
                'payment_methods' => $data['payment_methods'] ?? ['VA', 'QRIS', 'EWALLET'],
                'expired_time' => $data['expired_time'] ?? 24, // hours
            ];

            $signature = $this->generateSignature($payload);
            
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'X-Signature' => $signature,
            ])->post($this->baseUrl . '/api/v1/payment/create', $payload);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Gagal membuat pembayaran: ' . $response->body(),
                'error_code' => $response->status()
            ];
        } catch (\Exception $e) {
            Log::error('OttoPay create payment failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Cek status pembayaran
     */
    public function checkPaymentStatus($orderId)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ])->get($this->baseUrl . '/api/v1/payment/status/' . $orderId);

            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            }

            return [
                'success' => false,
                'message' => 'Gagal mengecek status: ' . $response->body()
            ];
        } catch (\Exception $e) {
            Log::error('OttoPay check status failed: ' . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Generate signature untuk keamanan
     */
    private function generateSignature($data)
    {
        $string = '';
        ksort($data);
        
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $value = json_encode($value);
            }
            $string .= $key . '=' . $value . '&';
        }
        
        $string = rtrim($string, '&');
        $string .= $this->secretKey;
        
        return hash('sha256', $string);
    }

    /**
     * Verify callback signature
     */
    public function verifyCallback($data, $signature)
    {
        $generatedSignature = $this->generateSignature($data);
        return hash_equals($generatedSignature, $signature);
    }

    /**
     * Get available payment methods
     */
    public function getPaymentMethods()
    {
        return [
            'VA' => [
                'name' => 'Virtual Account',
                'banks' => ['BCA', 'BRI', 'BNI', 'Mandiri', 'Permata', 'CIMB']
            ],
            'QRIS' => [
                'name' => 'QRIS',
                'description' => 'Pembayaran dengan QR Code'
            ],
            'EWALLET' => [
                'name' => 'E-Wallet',
                'providers' => ['OVO', 'DANA', 'GoPay', 'ShopeePay', 'LinkAja']
            ],
            'CREDIT_CARD' => [
                'name' => 'Kartu Kredit',
                'providers' => ['Visa', 'Mastercard', 'JCB']
            ]
        ];
    }

    /**
     * Get transaction fees
     */
    public function getTransactionFees()
    {
        return [
            'VA' => 4000, // Rp 4.000
            'QRIS' => '0.7%', // 0.7% dari nominal
            'EWALLET' => '2%', // 2% dari nominal
            'CREDIT_CARD' => '2.9%' // 2.9% dari nominal
        ];
    }
} 