<?php

namespace App\Services;

use App\Models\Setting;
use App\Services\PaymentGatewayService;

class PaymentMethodSyncService
{
    protected $paymentGatewayService;
    protected $setting;

    public function __construct(PaymentGatewayService $paymentGatewayService)
    {
        $this->paymentGatewayService = $paymentGatewayService;
        $this->setting = Setting::first();
    }

    /**
     * Get all available payment methods from configured gateways
     */
    public function getAvailablePaymentMethods()
    {
        $methods = [];

        // Get methods from Tripay
        if ($this->isTripayConfigured()) {
            $tripayMethods = $this->getTripayPaymentMethods();
            $methods = array_merge($methods, $tripayMethods);
        }

        // Get methods from OttoPay
        if ($this->isOttopayConfigured()) {
            $ottopayMethods = $this->getOttopayPaymentMethods();
            $methods = array_merge($methods, $ottopayMethods);
        }

        // Add manual/offline methods
        $methods = array_merge($methods, $this->getManualPaymentMethods());

        return $methods;
    }

    /**
     * Get payment methods for tourism module
     */
    public function getTourismPaymentMethods()
    {
        $methods = $this->getAvailablePaymentMethods();
        
        // Format for tourism frontend
        return array_map(function($method) {
            return [
                'code' => $method['code'],
                'name' => $method['name'],
                'type' => $method['type'],
                'icon' => $method['icon'],
                'description' => $method['description'],
                'fee' => $method['fee'],
                'gateway' => $method['gateway']
            ];
        }, $methods);
    }

    /**
     * Get payment methods for UMKM/checkout module
     */
    public function getUmkmPaymentMethods()
    {
        $methods = $this->getAvailablePaymentMethods();
        
        // Format for UMKM checkout
        return array_map(function($method) {
            return [
                'code' => $method['code'],
                'name' => $method['name'],
                'type' => $method['type'],
                'icon' => $method['icon'],
                'description' => $method['description'],
                'fee' => $method['fee'],
                'gateway' => $method['gateway'],
                'group' => $method['group'] // For grouping in UI
            ];
        }, $methods);
    }

    /**
     * Get Tripay payment methods
     */
    private function getTripayPaymentMethods()
    {
        $methods = [];
        $channels = $this->setting->tripay_channels ?? [];

        $tripayChannelMap = [
            'BRIVA' => [
                'name' => 'BRI Virtual Account',
                'type' => 'virtual_account',
                'icon' => 'https://tripay.co.id/upload/payment-icon/YgCdBxY21649654936.png',
                'group' => 'Virtual Account',
                'fee' => 4000
            ],
            'BCAVA' => [
                'name' => 'BCA Virtual Account',
                'type' => 'virtual_account',
                'icon' => 'https://tripay.co.id/upload/payment-icon/1ILuatq21649654936.png',
                'group' => 'Virtual Account',
                'fee' => 4000
            ],
            'BNIVA' => [
                'name' => 'BNI Virtual Account',
                'type' => 'virtual_account',
                'icon' => 'https://tripay.co.id/upload/payment-icon/X6MgCnl51649654936.png',
                'group' => 'Virtual Account',
                'fee' => 4000
            ],
            'MANDIRIVA' => [
                'name' => 'Mandiri Virtual Account',
                'type' => 'virtual_account',
                'icon' => 'https://tripay.co.id/upload/payment-icon/mandiri-icon.png',
                'group' => 'Virtual Account',
                'fee' => 4000
            ],
            'OVO' => [
                'name' => 'OVO',
                'type' => 'ewallet',
                'icon' => 'https://tripay.co.id/upload/payment-icon/qsE9n2l21649654936.png',
                'group' => 'E-Wallet',
                'fee' => 2500
            ],
            'DANA' => [
                'name' => 'DANA',
                'type' => 'ewallet',
                'icon' => 'https://tripay.co.id/upload/payment-icon/jRyO7gl21649654936.png',
                'group' => 'E-Wallet',
                'fee' => 2500
            ],
            'SHOPEEPAY' => [
                'name' => 'ShopeePay',
                'type' => 'ewallet',
                'icon' => 'https://tripay.co.id/upload/payment-icon/2jCBWkJ21649654936.png',
                'group' => 'E-Wallet',
                'fee' => 2500
            ],
            'GOPAY' => [
                'name' => 'GoPay',
                'type' => 'ewallet',
                'icon' => 'https://tripay.co.id/upload/payment-icon/gopay-icon.png',
                'group' => 'E-Wallet',
                'fee' => 2500
            ],
            'LINKAJA' => [
                'name' => 'LinkAja',
                'type' => 'ewallet',
                'icon' => 'https://tripay.co.id/upload/payment-icon/linkaja-icon.png',
                'group' => 'E-Wallet',
                'fee' => 2500
            ],
            'QRIS' => [
                'name' => 'QRIS',
                'type' => 'qris',
                'icon' => 'https://tripay.co.id/upload/payment-icon/qris-icon.png',
                'group' => 'QRIS',
                'fee' => 1500
            ]
        ];

        foreach ($channels as $channel) {
            if (isset($tripayChannelMap[$channel])) {
                $channelData = $tripayChannelMap[$channel];
                $methods[] = [
                    'code' => $channel,
                    'name' => $channelData['name'],
                    'type' => $channelData['type'],
                    'icon' => $channelData['icon'],
                    'description' => 'Pembayaran via ' . $channelData['name'],
                    'fee' => $channelData['fee'],
                    'gateway' => 'tripay',
                    'group' => $channelData['group']
                ];
            }
        }

        return $methods;
    }

    /**
     * Get OttoPay payment methods
     */
    private function getOttopayPaymentMethods()
    {
        $methods = [];
        $channels = $this->setting->ottopay_channels ?? [];

        $ottopayChannelMap = [
            'VA' => [
                'name' => 'Virtual Account (OttoPay)',
                'type' => 'virtual_account',
                'icon' => '/images/ottopay-va.png',
                'group' => 'Virtual Account',
                'fee' => 4000,
                'banks' => ['BCA', 'BRI', 'BNI', 'Mandiri', 'Permata', 'CIMB']
            ],
            'QRIS' => [
                'name' => 'QRIS (OttoPay)',
                'type' => 'qris',
                'icon' => '/images/ottopay-qris.png',
                'group' => 'QRIS',
                'fee_percentage' => 0.7
            ],
            'EWALLET' => [
                'name' => 'E-Wallet (OttoPay)',
                'type' => 'ewallet',
                'icon' => '/images/ottopay-ewallet.png',
                'group' => 'E-Wallet',
                'fee_percentage' => 2,
                'providers' => ['OVO', 'DANA', 'GoPay', 'ShopeePay', 'LinkAja']
            ],
            'CREDIT_CARD' => [
                'name' => 'Kartu Kredit',
                'type' => 'credit_card',
                'icon' => '/images/ottopay-cc.png',
                'group' => 'Kartu Kredit',
                'fee_percentage' => 2.9,
                'providers' => ['Visa', 'Mastercard', 'JCB']
            ]
        ];

        foreach ($channels as $channel) {
            if (isset($ottopayChannelMap[$channel])) {
                $channelData = $ottopayChannelMap[$channel];
                $methods[] = [
                    'code' => 'OTTOPAY_' . $channel,
                    'name' => $channelData['name'],
                    'type' => $channelData['type'],
                    'icon' => $channelData['icon'],
                    'description' => 'Pembayaran via ' . $channelData['name'],
                    'fee' => $channelData['fee'] ?? null,
                    'fee_percentage' => $channelData['fee_percentage'] ?? null,
                    'gateway' => 'ottopay',
                    'group' => $channelData['group'],
                    'providers' => $channelData['providers'] ?? null
                ];
            }
        }

        return $methods;
    }

    /**
     * Get manual/offline payment methods
     */
    private function getManualPaymentMethods()
    {
        return [
            [
                'code' => 'cash',
                'name' => 'Bayar di Tempat',
                'type' => 'manual',
                'icon' => '/images/cash-icon.png',
                'description' => 'Bayar tunai saat berkunjung/pengambilan',
                'fee' => 0,
                'gateway' => 'manual',
                'group' => 'Manual'
            ],
            [
                'code' => 'bank_transfer',
                'name' => 'Transfer Bank',
                'type' => 'manual',
                'icon' => '/images/bank-transfer-icon.png',
                'description' => 'Transfer manual ke rekening',
                'fee' => 0,
                'gateway' => 'manual',
                'group' => 'Manual'
            ]
        ];
    }

    /**
     * Calculate payment fee
     */
    public function calculatePaymentFee($amount, $paymentCode)
    {
        $methods = $this->getAvailablePaymentMethods();
        
        foreach ($methods as $method) {
            if ($method['code'] === $paymentCode) {
                if (isset($method['fee']) && $method['fee'] > 0) {
                    return $method['fee']; // Fixed fee
                }
                
                if (isset($method['fee_percentage']) && $method['fee_percentage'] > 0) {
                    return ($amount * $method['fee_percentage']) / 100; // Percentage fee
                }
                
                return 0; // No fee
            }
        }
        
        return 0;
    }

    /**
     * Get payment method details by code
     */
    public function getPaymentMethodDetails($paymentCode)
    {
        $methods = $this->getAvailablePaymentMethods();
        
        foreach ($methods as $method) {
            if ($method['code'] === $paymentCode) {
                return $method;
            }
        }
        
        return null;
    }

    /**
     * Check if Tripay is configured
     */
    private function isTripayConfigured()
    {
        return !empty($this->setting->tripay_api_key) && 
               !empty($this->setting->tripay_private_key) && 
               !empty($this->setting->tripay_merchant_code);
    }

    /**
     * Check if OttoPay is configured
     */
    private function isOttopayConfigured()
    {
        return !empty($this->setting->ottopay_api_key) && 
               !empty($this->setting->ottopay_secret_key) && 
               !empty($this->setting->ottopay_merchant_code);
    }

    /**
     * Generate payment options HTML for frontend
     */
    public function generatePaymentOptionsHtml($type = 'tourism')
    {
        $methods = $type === 'tourism' ? $this->getTourismPaymentMethods() : $this->getUmkmPaymentMethods();
        $groupedMethods = [];
        
        // Group methods by type
        foreach ($methods as $method) {
            $groupedMethods[$method['group']][] = $method;
        }
        
        $html = '';
        
        foreach ($groupedMethods as $groupName => $groupMethods) {
            $html .= '<div class="payment-group mb-4">';
            $html .= '<h4 class="text-sm font-medium text-gray-700 mb-3">' . $groupName . '</h4>';
            $html .= '<div class="space-y-2">';
            
            foreach ($groupMethods as $method) {
                $feeText = '';
                if ($method['fee'] > 0) {
                    $feeText = 'Biaya admin: Rp ' . number_format($method['fee'], 0, ',', '.');
                } elseif (isset($method['fee_percentage']) && $method['fee_percentage'] > 0) {
                    $feeText = 'Biaya admin: ' . $method['fee_percentage'] . '%';
                } else {
                    $feeText = 'Tanpa biaya admin';
                }
                
                $html .= '<label class="flex items-center p-3 border border-gray-300 rounded-lg cursor-pointer hover:bg-gray-50">';
                $html .= '<input type="radio" name="metode_pembayaran" value="' . $method['code'] . '" class="mr-3">';
                $html .= '<div class="flex items-center">';
                $html .= '<div class="w-8 h-8 bg-gray-100 rounded-lg flex items-center justify-center mr-3">';
                $html .= '<img src="' . $method['icon'] . '" alt="' . $method['name'] . '" class="w-6 h-6 object-contain">';
                $html .= '</div>';
                $html .= '<div>';
                $html .= '<div class="font-medium">' . $method['name'] . '</div>';
                $html .= '<div class="text-sm text-gray-500">' . $feeText . '</div>';
                $html .= '</div>';
                $html .= '</div>';
                $html .= '</label>';
            }
            
            $html .= '</div>';
            $html .= '</div>';
        }
        
        return $html;
    }
} 