<?php

namespace App\Services;

use App\Models\NotifikasiWa;
use App\Models\Penduduk;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    protected $apiKey;
    protected $baseUrl;

    public function __construct()
    {
        $this->apiKey = config('services.whatsapp.api_key');
        $this->baseUrl = config('services.whatsapp.base_url');
    }

    /**
     * Format nomor HP ke format WhatsApp
     */
    public function formatPhoneNumber($phone)
    {
        // Hapus karakter non-numeric
        $phone = preg_replace('/[^0-9]/', '', $phone);
        
        // Jika diawali 0, ganti dengan 62
        if (substr($phone, 0, 1) === '0') {
            $phone = '62' . substr($phone, 1);
        }
        
        // Jika tidak diawali 62, tambahkan 62
        if (substr($phone, 0, 2) !== '62') {
            $phone = '62' . $phone;
        }
        
        return $phone;
    }

    /**
     * Kirim pesan WhatsApp
     */
    public function sendMessage($phone, $message, $type = 'manual', $relatedId = null, $relatedType = null)
    {
        try {
            $formattedPhone = $this->formatPhoneNumber($phone);
            
            $response = Http::post($this->baseUrl . '/api/send', [
                'token' => $this->apiKey,
                'number' => $formattedPhone,
                'message' => $message
            ]);

            $responseData = $response->json();
            $status = $response->successful() && ($responseData['status'] ?? false) ? 'Berhasil' : 'Gagal';

            // Simpan log notifikasi
            NotifikasiWa::create([
                'nomor_tujuan' => $formattedPhone,
                'pesan' => $message,
                'tipe' => $type,
                'status' => $status,
                'response' => json_encode($responseData),
                'related_id' => $relatedId,
                'related_type' => $relatedType,
                'tanggal_kirim' => now()
            ]);

            return [
                'success' => $status === 'Berhasil',
                'message' => $status === 'Berhasil' ? 'Pesan berhasil dikirim' : 'Gagal mengirim pesan',
                'data' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('WhatsApp Send Error: ' . $e->getMessage());
            
            // Simpan log error
            NotifikasiWa::create([
                'nomor_tujuan' => $phone,
                'pesan' => $message,
                'tipe' => $type,
                'status' => 'Gagal',
                'response' => json_encode(['error' => $e->getMessage()]),
                'related_id' => $relatedId,
                'related_type' => $relatedType,
                'tanggal_kirim' => now()
            ]);

            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengirim pesan',
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Broadcast pesan ke semua penduduk yang memiliki WhatsApp
     */
    public function broadcastMessage($message, $type = 'broadcast', $relatedId = null, $relatedType = null)
    {
        $penduduk = Penduduk::whereNotNull('no_wa')
                           ->where('no_wa', '!=', '')
                           ->get();

        $results = [
            'success' => 0,
            'failed' => 0,
            'total' => $penduduk->count()
        ];

        foreach ($penduduk as $p) {
            $result = $this->sendMessage($p->no_wa, $message, $type, $relatedId, $relatedType);
            
            if ($result['success']) {
                $results['success']++;
            } else {
                $results['failed']++;
            }
            
            // Delay untuk menghindari rate limiting
            sleep(1);
        }

        return $results;
    }

    /**
     * Template pesan untuk persetujuan surat
     */
    public function templateSuratDisetujui($nama, $jenisSurat, $nomorSurat)
    {
        return "🔔 *NOTIFIKASI PELAYANAN SURAT*\n\n"
             . "Yth. Bapak/Ibu *{$nama}*\n\n"
             . "Pengajuan {$jenisSurat} Anda telah *DISETUJUI*\n"
             . "Nomor Surat: {$nomorSurat}\n\n"
             . "Silakan datang ke kantor desa untuk mengambil surat.\n\n"
             . "Terima kasih.\n\n"
             . "_Pesan ini dikirim otomatis oleh sistem Aplikasi Desa_";
    }

    /**
     * Template pesan untuk pengaduan yang ditanggapi
     */
    public function templatePengaduanDitanggapi($nama, $judulPengaduan, $tanggapan)
    {
        return "🔔 *TANGGAPAN PENGADUAN*\n\n"
             . "Yth. Bapak/Ibu *{$nama}*\n\n"
             . "Pengaduan Anda telah ditanggapi:\n"
             . "*{$judulPengaduan}*\n\n"
             . "*Tanggapan:*\n{$tanggapan}\n\n"
             . "Terima kasih atas partisipasi Anda.\n\n"
             . "_Pesan ini dikirim otomatis oleh sistem Aplikasi Desa_";
    }

    /**
     * Template pesan untuk bantuan yang diterima
     */
    public function templateBantuanDiterima($nama, $jenisBantuan, $periode)
    {
        return "🔔 *NOTIFIKASI BANTUAN SOSIAL*\n\n"
             . "Yth. Bapak/Ibu *{$nama}*\n\n"
             . "Selamat! Anda telah terpilih sebagai penerima bantuan:\n"
             . "*{$jenisBantuan}*\n"
             . "Periode: {$periode}\n\n"
             . "Silakan datang ke kantor desa untuk informasi lebih lanjut mengenai distribusi bantuan.\n\n"
             . "Terima kasih.\n\n"
             . "_Pesan ini dikirim otomatis oleh sistem Aplikasi Desa_";
    }

    /**
     * Template pesan untuk berita baru
     */
    public function templateBeritaBaru($judul, $ringkasan, $link)
    {
        return "📰 *BERITA TERBARU*\n\n"
             . "*{$judul}*\n\n"
             . "{$ringkasan}\n\n"
             . "Baca selengkapnya: {$link}\n\n"
             . "_Tetap terhubung dengan informasi terkini dari desa Anda_";
    }

    /**
     * Template pesan custom
     */
    public function templateCustom($judul, $pesan)
    {
        return "📢 *{$judul}*\n\n"
             . "{$pesan}\n\n"
             . "_Pesan dari Pemerintah Desa_";
    }

    /**
     * Cek status API WhatsApp
     */
    public function checkStatus()
    {
        try {
            $response = Http::get($this->baseUrl . '/api/status', [
                'token' => $this->apiKey
            ]);

            return [
                'success' => $response->successful(),
                'data' => $response->json()
            ];

        } catch (\Exception $e) {
            Log::error('WhatsApp Status Check Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Dapatkan info device WhatsApp
     */
    public function getDeviceInfo()
    {
        try {
            $response = Http::get($this->baseUrl . '/api/device', [
                'token' => $this->apiKey
            ]);

            return [
                'success' => $response->successful(),
                'data' => $response->json()
            ];

        } catch (\Exception $e) {
            Log::error('WhatsApp Device Info Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Kirim pesan dengan gambar
     */
    public function sendImageMessage($phone, $message, $imageUrl)
    {
        try {
            $formattedPhone = $this->formatPhoneNumber($phone);
            
            $response = Http::post($this->baseUrl . '/api/send-media', [
                'token' => $this->apiKey,
                'number' => $formattedPhone,
                'message' => $message,
                'media' => $imageUrl,
                'type' => 'image'
            ]);

            $responseData = $response->json();
            $status = $response->successful() && ($responseData['status'] ?? false) ? 'Berhasil' : 'Gagal';

            // Simpan log notifikasi
            NotifikasiWa::create([
                'nomor_tujuan' => $formattedPhone,
                'pesan' => $message,
                'tipe' => 'media',
                'status' => $status,
                'response' => json_encode($responseData),
                'tanggal_kirim' => now()
            ]);

            return [
                'success' => $status === 'Berhasil',
                'message' => $status === 'Berhasil' ? 'Pesan dengan gambar berhasil dikirim' : 'Gagal mengirim pesan',
                'data' => $responseData
            ];

        } catch (\Exception $e) {
            Log::error('WhatsApp Send Image Error: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengirim pesan dengan gambar',
                'error' => $e->getMessage()
            ];
        }
    }
} 
