import 'package:flutter/foundation.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';

import '../services/api_service.dart';
import '../core/constants/app_constants.dart';

class AuthProvider extends ChangeNotifier {
  bool _isAuthenticated = false;
  bool _isLoading = false;
  String? _errorMessage;
  Map<String, dynamic>? _userData;
  String? _authToken;

  // Getters
  bool get isAuthenticated => _isAuthenticated;
  bool get isLoading => _isLoading;
  String? get errorMessage => _errorMessage;
  bool get hasError => _errorMessage != null;
  Map<String, dynamic>? get userData => _userData;
  String? get authToken => _authToken;
  String? get userName => _userData?['name'] ?? _userData?['nama'];
  String? get userEmail => _userData?['email'];

  AuthProvider() {
    _checkAuthStatus();
  }

  Future<void> _checkAuthStatus() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      _authToken = prefs.getString(AppConstants.authTokenKey);
      final userDataString = prefs.getString(AppConstants.userDataKey);
      
      if (_authToken != null && userDataString != null) {
        _userData = json.decode(userDataString);
        _isAuthenticated = true;
        notifyListeners();
      }
    } catch (e) {
      debugPrint('Error checking auth status: $e');
    }
  }

  Future<bool> login({
    required String email,
    required String password,
    String? captcha,
  }) async {
    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.login({
          'email': email,
          'password': password,
          if (captcha != null) 'captcha': captcha,
        }),
      );

      if (response.success && response.data != null) {
        final data = response.data!;
        _authToken = data['token'];
        _userData = data['user'];
        _isAuthenticated = true;

        // Save to storage
        await _saveAuthData();
        await ApiService.instance.saveAuthToken(_authToken!);
        
        _setLoading(false);
        return true;
      } else {
        _setError(response.message ?? 'Login gagal');
        return false;
      }
    } catch (e) {
      _setError(e.toString());
      return false;
    }
  }

  Future<bool> portalLogin({
    required String nik,
    required String phone,
    String? captcha,
  }) async {
    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.portalLogin({
          'nik': nik,
          'phone': phone,
          if (captcha != null) 'captcha': captcha,
        }),
      );

      if (response.success && response.data != null) {
        final data = response.data!;
        _authToken = data['token'];
        _userData = data['user'];
        _isAuthenticated = true;

        // Save to storage
        await _saveAuthData();
        await ApiService.instance.saveAuthToken(_authToken!);
        
        _setLoading(false);
        return true;
      } else {
        _setError(response.message ?? 'Login portal gagal');
        return false;
      }
    } catch (e) {
      _setError(e.toString());
      return false;
    }
  }

  Future<void> logout() async {
    _setLoading(true);

    try {
      // Call logout API
      await ApiService.instance.handleRequest(
        () => ApiService.instance.client.logout(),
      );
    } catch (e) {
      debugPrint('Error calling logout API: $e');
    }

    // Clear local data regardless of API response
    await _clearAuthData();
    await ApiService.instance.clearAuthToken();
    
    _isAuthenticated = false;
    _authToken = null;
    _userData = null;
    _setLoading(false);
  }

  Future<void> _saveAuthData() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      if (_authToken != null) {
        await prefs.setString(AppConstants.authTokenKey, _authToken!);
      }
      if (_userData != null) {
        await prefs.setString(AppConstants.userDataKey, json.encode(_userData));
      }
    } catch (e) {
      debugPrint('Error saving auth data: $e');
    }
  }

  Future<void> _clearAuthData() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      await prefs.remove(AppConstants.authTokenKey);
      await prefs.remove(AppConstants.userDataKey);
    } catch (e) {
      debugPrint('Error clearing auth data: $e');
    }
  }

  // OTP methods
  Future<bool> sendOtp({
    required String nik,
    required String phone,
  }) async {
    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.sendOtp({
          'nik': nik,
          'phone': phone,
        }),
      );

      _setLoading(false);
      
      if (response.success) {
        return true;
      } else {
        _setError(response.message ?? 'Gagal mengirim OTP');
        return false;
      }
    } catch (e) {
      _setError(e.toString());
      return false;
    }
  }

  Future<bool> verifyOtp({
    required String nik,
    required String phone,
    required String otp,
  }) async {
    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.verifyOtp({
          'nik': nik,
          'phone': phone,
          'otp': otp,
        }),
      );

      if (response.success && response.data != null) {
        final data = response.data!;
        _authToken = data['token'];
        _userData = data['user'];
        _isAuthenticated = true;

        // Save to storage
        await _saveAuthData();
        await ApiService.instance.saveAuthToken(_authToken!);
        
        _setLoading(false);
        return true;
      } else {
        _setError(response.message ?? 'Verifikasi OTP gagal');
        return false;
      }
    } catch (e) {
      _setError(e.toString());
      return false;
    }
  }

  void _setLoading(bool loading) {
    _isLoading = loading;
    notifyListeners();
  }

  void _setError(String error) {
    _errorMessage = error;
    _isLoading = false;
    notifyListeners();
  }

  void _clearError() {
    _errorMessage = null;
    notifyListeners();
  }

}
