import 'package:flutter/foundation.dart';
import 'package:shared_preferences/shared_preferences.dart';
import 'dart:convert';

import '../models/village_setting.dart';
import '../services/api_service.dart';
import '../core/constants/app_constants.dart';

class VillageProvider extends ChangeNotifier {
  VillageSetting? _villageSetting;
  bool _isLoading = false;
  String? _errorMessage;

  // Getters
  VillageSetting? get villageSetting => _villageSetting;
  bool get isLoading => _isLoading;
  String? get errorMessage => _errorMessage;
  bool get hasError => _errorMessage != null;

  // Default values
  String get villageName => _villageSetting?.displayName ?? AppConstants.appName;
  String get villageLocation => _villageSetting?.locationText ?? 'Desa Digital';
  String? get villageLogo => _villageSetting?.logoUrl;
  String? get youtubeVideoId => _villageSetting?.youtubeVideoId ?? AppConstants.defaultYouTubeVideoId;

  VillageProvider() {
    _loadCachedData();
    loadVillageSetting();
  }

  Future<void> _loadCachedData() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      final cachedData = prefs.getString(AppConstants.villageSettingKey);
      
      if (cachedData != null) {
        final jsonData = json.decode(cachedData);
        _villageSetting = VillageSetting.fromJson(jsonData['data']);
        
        // Check if cache is expired
        final cacheTime = DateTime.parse(jsonData['cached_at']);
        final now = DateTime.now();
        final difference = now.difference(cacheTime).inHours;
        
        if (difference < AppConstants.cacheVillageSettingDuration) {
          notifyListeners();
          return;
        }
      }
    } catch (e) {
      debugPrint('Error loading cached village setting: $e');
    }
  }

  Future<void> loadVillageSetting({bool forceRefresh = false}) async {
    if (_isLoading && !forceRefresh) return;

    _setLoading(true);
    _clearError();

    try {
      final response = await ApiService.instance.handleRequest(
        () => ApiService.instance.client.getVillageSetting(),
      );

      if (response.success && response.data != null) {
        _villageSetting = response.data;
        await _cacheData();
        _setLoading(false);
      } else {
        _setError(response.message ?? 'Gagal memuat pengaturan desa');
      }
    } catch (e) {
      _setError(e.toString());
    }
  }

  Future<void> _cacheData() async {
    if (_villageSetting == null) return;
    
    try {
      final prefs = await SharedPreferences.getInstance();
      final cacheData = {
        'data': _villageSetting!.toJson(),
        'cached_at': DateTime.now().toIso8601String(),
      };
      
      await prefs.setString(
        AppConstants.villageSettingKey,
        json.encode(cacheData),
      );
    } catch (e) {
      debugPrint('Error caching village setting: $e');
    }
  }

  void _setLoading(bool loading) {
    _isLoading = loading;
    notifyListeners();
  }

  void _setError(String error) {
    _errorMessage = error;
    _isLoading = false;
    notifyListeners();
  }

  void _clearError() {
    _errorMessage = null;
    notifyListeners();
  }

  // Refresh data
  Future<void> refresh() async {
    await loadVillageSetting(forceRefresh: true);
  }

  // Clear cache
  Future<void> clearCache() async {
    try {
      final prefs = await SharedPreferences.getInstance();
      await prefs.remove(AppConstants.villageSettingKey);
      _villageSetting = null;
      notifyListeners();
    } catch (e) {
      debugPrint('Error clearing village setting cache: $e');
    }
  }

}
