<?php
/**
 * Fix Tourism Image Paths - Clean Temporary Paths
 * This script fixes tourism images that have temporary file paths saved in the database
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel application
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Fix Tourism Image Paths</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 800px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .success { color: #28a745; }
        .error { color: #dc3545; }
        .warning { color: #ffc107; }
        .info { color: #17a2b8; }
        .step { margin: 15px 0; padding: 15px; border-left: 4px solid #007bff; background: #f8f9fa; }
        pre { background: #f1f1f1; padding: 10px; border-radius: 4px; overflow-x: auto; }
        .status-box { padding: 10px; margin: 10px 0; border-radius: 4px; }
        .status-success { background: #d4edda; border-color: #c3e6cb; color: #155724; }
        .status-error { background: #f8d7da; border-color: #f5c6cb; color: #721c24; }
        .status-warning { background: #fff3cd; border-color: #ffeaa7; color: #856404; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🖼️ Fix Tourism Image Paths</h1>
        <p>This tool fixes tourism images that have temporary file paths in the database.</p>

        <?php
        echo "<h2>🔍 Checking Image Data</h2>";
        
        try {
            $totalTourism = ObjekWisata::count();
            echo "<div class='status-box status-success'>✅ Total tourism objects: {$totalTourism}</div>";
            
            if ($totalTourism > 0) {
                $tourismWithImages = ObjekWisata::whereNotNull('foto_utama')->get();
                echo "<div class='status-box status-success'>✅ Tourism with foto_utama: " . $tourismWithImages->count() . "</div>";
                
                $tourismWithTempPaths = ObjekWisata::where('foto_utama', 'like', '%tmp%')->get();
                echo "<div class='status-box status-warning'>⚠️ Tourism with temporary paths: " . $tourismWithTempPaths->count() . "</div>";
                
                echo "<h3>📋 Current Image Paths:</h3>";
                
                foreach ($tourismWithImages as $tourism) {
                    echo "<div class='step'>";
                    echo "<h4>{$tourism->nama_wisata}</h4>";
                    echo "<p><strong>ID:</strong> {$tourism->id}</p>";
                    echo "<p><strong>Current foto_utama:</strong> {$tourism->foto_utama}</p>";
                    
                    // Check if path contains temporary file
                    if (strpos($tourism->foto_utama, 'tmp') !== false || strpos($tourism->foto_utama, 'php') !== false) {
                        echo "<p class='error'>❌ Contains temporary path!</p>";
                        
                        // Try to fix by setting to sample image or null
                        if (file_exists(public_path('storage/wisata/sample-tourism-1.png'))) {
                            $newPath = 'sample-tourism-1.png';
                            $updated = $tourism->update(['foto_utama' => $newPath]);
                            if ($updated) {
                                echo "<p class='success'>✅ Fixed: Set to sample-tourism-1.png</p>";
                            }
                        } else {
                            $updated = $tourism->update(['foto_utama' => null]);
                            if ($updated) {
                                echo "<p class='success'>✅ Fixed: Set to null (will use default)</p>";
                            }
                        }
                    } else {
                        // Check if file actually exists
                        $imagePath = public_path('storage/wisata/' . $tourism->foto_utama);
                        if (file_exists($imagePath)) {
                            echo "<p class='success'>✅ File exists in public/storage</p>";
                        } else {
                            echo "<p class='warning'>⚠️ File not found in public/storage</p>";
                            
                            // Try to find in storage/app/public
                            $storagePath = storage_path('app/public/wisata/' . $tourism->foto_utama);
                            if (file_exists($storagePath)) {
                                echo "<p class='info'>📁 File found in storage/app/public</p>";
                                
                                // Copy to public/storage
                                $publicDir = public_path('storage/wisata');
                                if (!is_dir($publicDir)) {
                                    mkdir($publicDir, 0755, true);
                                }
                                
                                $publicPath = $publicDir . '/' . $tourism->foto_utama;
                                if (copy($storagePath, $publicPath)) {
                                    echo "<p class='success'>✅ Copied to public/storage</p>";
                                } else {
                                    echo "<p class='error'>❌ Failed to copy to public/storage</p>";
                                }
                            } else {
                                echo "<p class='error'>❌ File not found anywhere</p>";
                                
                                // Set to sample image if available
                                if (file_exists(public_path('storage/wisata/sample-tourism-1.png'))) {
                                    $updated = $tourism->update(['foto_utama' => 'sample-tourism-1.png']);
                                    if ($updated) {
                                        echo "<p class='success'>✅ Set to sample image</p>";
                                    }
                                }
                            }
                        }
                        
                        // Test URL
                        $imageUrl = asset('storage/wisata/' . $tourism->foto_utama);
                        echo "<p><strong>Image URL:</strong> <a href='{$imageUrl}' target='_blank'>{$imageUrl}</a></p>";
                    }
                    echo "</div>";
                }
                
                // Check galeri_foto
                echo "<h3>🖼️ Checking Gallery Images:</h3>";
                
                $tourismWithGallery = ObjekWisata::whereNotNull('galeri_foto')->get();
                
                foreach ($tourismWithGallery as $tourism) {
                    $galeri = $tourism->galeri_foto;
                    
                    if (is_array($galeri) && count($galeri) > 0) {
                        echo "<div class='step'>";
                        echo "<h4>{$tourism->nama_wisata} - Gallery</h4>";
                        
                        $cleanGaleri = [];
                        $hasChanges = false;
                        
                        foreach ($galeri as $foto) {
                            echo "<p><strong>Gallery item:</strong> {$foto}</p>";
                            
                            if (strpos($foto, 'tmp') !== false || strpos($foto, 'php') !== false) {
                                echo "<p class='error'>❌ Contains temporary path - removing</p>";
                                $hasChanges = true;
                            } else if (is_string($foto) && !empty($foto)) {
                                $cleanGaleri[] = $foto;
                            }
                        }
                        
                        if ($hasChanges) {
                            $updated = $tourism->update(['galeri_foto' => $cleanGaleri]);
                            if ($updated) {
                                echo "<p class='success'>✅ Gallery cleaned: " . count($cleanGaleri) . " valid images remaining</p>";
                            }
                        }
                        
                        echo "</div>";
                    }
                }
                
            } else {
                echo "<div class='status-box status-warning'>⚠️ No tourism data found</div>";
            }
            
            echo "<h2>🔧 Symlink Status</h2>";
            
            $publicStorageExists = is_dir(public_path('storage'));
            $publicWisataExists = is_dir(public_path('storage/wisata'));
            $storageWisataExists = is_dir(storage_path('app/public/wisata'));
            
            echo "<div class='status-box " . ($publicStorageExists ? 'status-success' : 'status-error') . "'>";
            echo ($publicStorageExists ? '✅' : '❌') . " public/storage directory: " . ($publicStorageExists ? 'EXISTS' : 'NOT FOUND');
            echo "</div>";
            
            echo "<div class='status-box " . ($publicWisataExists ? 'status-success' : 'status-error') . "'>";
            echo ($publicWisataExists ? '✅' : '❌') . " public/storage/wisata directory: " . ($publicWisataExists ? 'EXISTS' : 'NOT FOUND');
            echo "</div>";
            
            echo "<div class='status-box " . ($storageWisataExists ? 'status-success' : 'status-error') . "'>";
            echo ($storageWisataExists ? '✅' : '❌') . " storage/app/public/wisata directory: " . ($storageWisataExists ? 'EXISTS' : 'NOT FOUND');
            echo "</div>";
            
            // Create missing directories
            if (!$publicWisataExists) {
                if (mkdir(public_path('storage/wisata'), 0755, true)) {
                    echo "<div class='status-box status-success'>✅ Created public/storage/wisata directory</div>";
                }
            }
            
            if (!$storageWisataExists) {
                if (mkdir(storage_path('app/public/wisata'), 0755, true)) {
                    echo "<div class='status-box status-success'>✅ Created storage/app/public/wisata directory</div>";
                }
            }
            
        } catch (Exception $e) {
            echo "<div class='status-box status-error'>❌ Error: " . $e->getMessage() . "</div>";
        }
        
        echo "<h2>🧪 Test URLs</h2>";
        
        $testUrls = [
            'Sample Tourism 1' => asset('storage/wisata/sample-tourism-1.png'),
            'Sample Tourism 2' => asset('storage/wisata/sample-tourism-2.png'),
            'Default Tourism' => asset('images/default-tourism.png'),
        ];
        
        foreach ($testUrls as $name => $url) {
            echo "<div class='step'>";
            echo "<strong>$name:</strong> <a href='$url' target='_blank'>$url</a>";
            echo "</div>";
        }
        
        echo "<h2>✅ Completion</h2>";
        echo "<p>All temporary image paths have been cleaned and directories have been created.</p>";
        echo "<p>Visit the <a href='/admin/objek-wisata' target='_blank'>Tourism Admin Page</a> to verify images are working.</p>";
        ?>
    </div>
</body>
</html>
