<?php
/**
 * Script untuk memperbaiki masalah akses gambar
 * Akses via: https://susukannet.id/fix-images.php
 */

// Include Laravel bootstrap
require_once __DIR__ . '/../vendor/autoload.php';
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

echo "<h1>Fix Images Script</h1>";

// Test file paths
$testFiles = [
    'umkm/logo/sample-logo.png',
    'umkm/galeri/sample-gallery1.png',
    'wisata/sample-tourism-1.png',
    'wisata/sample-tourism-2.png'
];

echo "<h2>File System Check</h2>";

foreach ($testFiles as $file) {
    echo "<h3>Testing: {$file}</h3>";
    
    // Check storage path
    $storagePath = storage_path('app/public/' . $file);
    $storageExists = file_exists($storagePath);
    echo "<div class='info'>Storage Path: {$storagePath}</div>";
    echo "<div class='" . ($storageExists ? 'success' : 'error') . "'>Storage Exists: " . ($storageExists ? 'YES' : 'NO') . "</div>";
    
    if ($storageExists) {
        echo "<div class='info'>Storage Size: " . filesize($storagePath) . " bytes</div>";
    }
    
    // Check public path
    $publicPath = public_path('storage/' . $file);
    $publicExists = file_exists($publicPath);
    echo "<div class='info'>Public Path: {$publicPath}</div>";
    echo "<div class='" . ($publicExists ? 'success' : 'error') . "'>Public Exists: " . ($publicExists ? 'YES' : 'NO') . "</div>";
    
    if ($publicExists) {
        echo "<div class='info'>Public Size: " . filesize($publicPath) . " bytes</div>";
    }
    
    // Try to copy if storage exists but public doesn't
    if ($storageExists && !$publicExists) {
        $publicDir = dirname($publicPath);
        if (!is_dir($publicDir)) {
            mkdir($publicDir, 0755, true);
            echo "<div class='info'>Created directory: {$publicDir}</div>";
        }
        
        if (copy($storagePath, $publicPath)) {
            echo "<div class='success'>✅ File copied to public successfully</div>";
        } else {
            echo "<div class='error'>❌ Failed to copy file to public</div>";
        }
    }
    
    // Test URL access
    $url = 'https://susukannet.id/storage/' . $file;
    echo "<div class='info'>Test URL: <a href='{$url}' target='_blank'>{$url}</a></div>";
    
    echo "<hr>";
}

// Test SharedHostingHelper
echo "<h2>SharedHostingHelper Test</h2>";

$helper = new \App\Helpers\SharedHostingHelper();

foreach ($testFiles as $file) {
    $url = $helper::getFileUrl($file);
    echo "<div class='info'>Helper URL for {$file}: <a href='{$url}' target='_blank'>{$url}</a></div>";
}

// Test symlink
echo "<h2>Symlink Test</h2>";

$symlinkPath = public_path('storage');
$symlinkExists = file_exists($symlinkPath);
$isLink = is_link($symlinkPath);

echo "<div class='info'>Symlink Path: {$symlinkPath}</div>";
echo "<div class='" . ($symlinkExists ? 'success' : 'error') . "'>Symlink Exists: " . ($symlinkExists ? 'YES' : 'NO') . "</div>";
echo "<div class='" . ($isLink ? 'success' : 'error') . "'>Is Link: " . ($isLink ? 'YES' : 'NO') . "</div>";

if ($isLink) {
    $target = readlink($symlinkPath);
    echo "<div class='info'>Symlink Target: {$target}</div>";
}

// Create missing sample files if they don't exist
echo "<h2>Creating Missing Sample Files</h2>";

$sampleFiles = [
    'umkm/logo/sample-logo.png' => 'UMKM Logo Sample',
    'umkm/galeri/sample-gallery1.png' => 'UMKM Gallery Sample 1',
    'wisata/sample-tourism-1.png' => 'Tourism Sample 1',
    'wisata/sample-tourism-2.png' => 'Tourism Sample 2'
];

foreach ($sampleFiles as $file => $text) {
    $storagePath = storage_path('app/public/' . $file);
    
    if (!file_exists($storagePath)) {
        $dir = dirname($storagePath);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        // Create a simple image
        $image = imagecreate(200, 150);
        $bg = imagecolorallocate($image, 100, 150, 200);
        $textColor = imagecolorallocate($image, 255, 255, 255);
        
        imagestring($image, 5, 20, 60, $text, $textColor);
        imagepng($image, $storagePath);
        imagedestroy($image);
        
        echo "<div class='success'>✅ Created sample file: {$file}</div>";
    } else {
        echo "<div class='info'>📋 Sample file already exists: {$file}</div>";
    }
}

// Copy all files from storage to public
echo "<h2>Syncing All Files to Public</h2>";

$storageDir = storage_path('app/public');
$publicDir = public_path('storage');

if (is_dir($storageDir)) {
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($storageDir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    $copied = 0;
    foreach ($iterator as $item) {
        if ($item->isFile()) {
            $relativePath = str_replace($storageDir . DIRECTORY_SEPARATOR, '', $item->getPathname());
            $relativePath = str_replace('\\', '/', $relativePath);
            
            $publicPath = $publicDir . DIRECTORY_SEPARATOR . $relativePath;
            $publicPath = str_replace('\\', '/', $publicPath);
            
            $publicFileDir = dirname($publicPath);
            if (!is_dir($publicFileDir)) {
                mkdir($publicFileDir, 0755, true);
            }
            
            if (!file_exists($publicPath)) {
                if (copy($item->getPathname(), $publicPath)) {
                    $copied++;
                    echo "<div class='success'>✅ Copied: {$relativePath}</div>";
                } else {
                    echo "<div class='error'>❌ Failed to copy: {$relativePath}</div>";
                }
            }
        }
    }
    
    echo "<div class='info'>📋 Total files copied: {$copied}</div>";
}

echo "<h2>Final Test</h2>";

foreach ($testFiles as $file) {
    $url = 'https://susukannet.id/storage/' . $file;
    echo "<div class='info'>Final URL: <a href='{$url}' target='_blank'>{$url}</a></div>";
}

echo "<h2>Script Completed</h2>";
echo "<p><a href='/admin/umkm'>Kembali ke Admin UMKM</a></p>";

?>
<style>
.success { color: green; font-weight: bold; }
.error { color: red; font-weight: bold; }
.info { color: blue; }
</style>
