import './bootstrap';

// Modern scroll utilities to replace deprecated DOMNodeInserted
class ModernScrollUtilities {
    constructor() {
        this.init();
    }

    init() {
        this.setupMutationObserver();
        this.replaceDeprecatedEvents();
    }

    // Use MutationObserver instead of DOMNodeInserted
    setupMutationObserver() {
        const observer = new MutationObserver((mutations) => {
            mutations.forEach((mutation) => {
                if (mutation.type === 'childList') {
                    // Handle added nodes
                    mutation.addedNodes.forEach((node) => {
                        if (node.nodeType === Node.ELEMENT_NODE) {
                            this.handleNewElement(node);
                        }
                    });
                }
            });
        });

        // Start observing
        observer.observe(document.body, {
            childList: true,
            subtree: true
        });
    }

    // Handle new elements (replacement for DOMNodeInserted)
    handleNewElement(element) {
        // Initialize any scroll-related functionality for new elements
        if (element.classList && element.classList.contains('scrollable')) {
            this.initializeScrollable(element);
        }
        
        // Check for any child scrollable elements
        const scrollableChildren = element.querySelectorAll('.scrollable');
        scrollableChildren.forEach(child => this.initializeScrollable(child));
    }

    // Initialize scrollable elements with modern APIs
    initializeScrollable(element) {
        // Modern scroll event handling
        element.addEventListener('scroll', (e) => {
            this.handleScrollEvent(e);
        }, { passive: true });
    }

    // Modern scroll event handler
    handleScrollEvent(event) {
        const element = event.target;
        // Custom scroll handling logic here
        // This replaces any deprecated scroll.js functionality
    }

    // Replace any deprecated event listeners
    replaceDeprecatedEvents() {
        // Check for any existing DOMNodeInserted listeners and replace them
        const scripts = document.querySelectorAll('script');
        scripts.forEach(script => {
            if (script.textContent && script.textContent.includes('DOMNodeInserted')) {
                console.warn('Deprecated DOMNodeInserted event found and should be replaced with MutationObserver');
            }
        });
    }
}

// Mobile App JavaScript Functionality
class MobileApp {
    constructor() {
        this.init();
    }

    init() {
        this.setupServiceWorker();
        this.setupGestures();
        this.setupAnimations();
        this.setupOfflineSupport();
    }

    // Service Worker for PWA functionality
    setupServiceWorker() {
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('/sw.js')
                .then(registration => console.log('SW registered:', registration))
                .catch(error => console.log('SW registration failed:', error));
        }
    }

    // Touch gestures and mobile interactions
    setupGestures() {
        // Pull to refresh
        let startY = 0;
        let isRefreshing = false;

        document.addEventListener('touchstart', (e) => {
            startY = e.touches[0].pageY;
        });

        document.addEventListener('touchmove', (e) => {
            const currentY = e.touches[0].pageY;
            const diff = currentY - startY;

            if (diff > 100 && window.scrollY === 0 && !isRefreshing) {
                this.handlePullToRefresh();
                isRefreshing = true;
            }
        });

        document.addEventListener('touchend', () => {
            setTimeout(() => {
                isRefreshing = false;
            }, 1000);
        });

        // Add haptic feedback for supported devices
        this.addHapticFeedback();
    }

    handlePullToRefresh() {
        // Show loading indicator
        window.showLoading && window.showLoading();
        
        // Simulate refresh
        setTimeout(() => {
            window.location.reload();
        }, 1000);
    }

    addHapticFeedback() {
        const buttons = document.querySelectorAll('.mobile-button, .mobile-nav-item');
        
        buttons.forEach(button => {
            button.addEventListener('click', () => {
                // Haptic feedback for supported devices
                if ('vibrate' in navigator) {
                    navigator.vibrate(10);
                }
            });
        });
    }

    // Enhanced animations
    setupAnimations() {
        // Intersection Observer for scroll animations (modern approach)
        const observerOptions = {
            threshold: 0.1,
            rootMargin: '0px 0px -50px 0px'
        };

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    entry.target.classList.add('animate-in');
                }
            });
        }, observerOptions);

        // Observe all animated elements
        document.querySelectorAll('.animate-slide-in-up, .animate-fade-in').forEach(el => {
            observer.observe(el);
        });

        // Add parallax effect for hero section
        this.setupParallax();
    }

    setupParallax() {
        const heroSection = document.querySelector('#home');
        if (!heroSection) return;

        // Use passive event listeners for better performance
        window.addEventListener('scroll', () => {
            const scrolled = window.pageYOffset;
            const parallax = heroSection.querySelector('.hero-content');
            
            if (parallax) {
                const speed = scrolled * 0.5;
                parallax.style.transform = `translateY(${speed}px)`;
            }
        }, { passive: true });
    }

    // Offline support and caching
    setupOfflineSupport() {
        // Check online status
        window.addEventListener('online', () => {
            window.showToast && window.showToast('Koneksi internet tersambung kembali', 'success');
        });

        window.addEventListener('offline', () => {
            window.showToast && window.showToast('Anda sedang offline. Beberapa fitur mungkin tidak tersedia.', 'warning');
        });

        // Cache critical resources
        this.cacheResources();
    }

    cacheResources() {
        if ('caches' in window) {
            const CACHE_NAME = 'portal-desa-v1';
            const urlsToCache = [
                '/',
                '/css/app.css',
                '/js/app.js',
                '/public/ajukan-surat',
                '/public/pengaduan',
                '/public/info-bantuan'
            ];

            caches.open(CACHE_NAME)
                .then(cache => cache.addAll(urlsToCache))
                .catch(error => console.log('Cache error:', error));
        }
    }
}

// Enhanced form handling
class FormHandler {
    constructor() {
        this.initFormValidation();
        this.initProgressiveForms();
    }

    initFormValidation() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(form => {
            form.addEventListener('submit', (e) => {
                if (!this.validateForm(form)) {
                    e.preventDefault();
                    window.showToast && window.showToast('Mohon lengkapi semua field yang wajib diisi', 'error');
                }
            });

            // Real-time validation
            const inputs = form.querySelectorAll('input, textarea, select');
            inputs.forEach(input => {
                input.addEventListener('blur', () => this.validateField(input));
            });
        });
    }

    validateForm(form) {
        const requiredFields = form.querySelectorAll('[required]');
        let isValid = true;

        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                this.showFieldError(field, 'Field ini wajib diisi');
                isValid = false;
            } else {
                this.clearFieldError(field);
            }
        });

        return isValid;
    }

    validateField(field) {
        const value = field.value.trim();
        
        if (field.required && !value) {
            this.showFieldError(field, 'Field ini wajib diisi');
            return false;
        }

        // Email validation
        if (field.type === 'email' && value) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            if (!emailRegex.test(value)) {
                this.showFieldError(field, 'Format email tidak valid');
                return false;
            }
        }

        // Phone validation
        if (field.type === 'tel' && value) {
            const phoneRegex = /^(\+62|62|0)[0-9]{9,13}$/;
            if (!phoneRegex.test(value)) {
                this.showFieldError(field, 'Format nomor telepon tidak valid');
                return false;
            }
        }

        this.clearFieldError(field);
        return true;
    }

    showFieldError(field, message) {
        field.classList.add('border-red-500', 'bg-red-50');
        field.classList.remove('border-neutral-200');
        
        let errorEl = field.parentNode.querySelector('.field-error');
        if (!errorEl) {
            errorEl = document.createElement('div');
            errorEl.className = 'field-error text-red-500 text-sm mt-1';
            field.parentNode.appendChild(errorEl);
        }
        errorEl.textContent = message;
    }

    clearFieldError(field) {
        field.classList.remove('border-red-500', 'bg-red-50');
        field.classList.add('border-neutral-200');
        
        const errorEl = field.parentNode.querySelector('.field-error');
        if (errorEl) {
            errorEl.remove();
        }
    }

    initProgressiveForms() {
        // Auto-save form data in localStorage
        const forms = document.querySelectorAll('form[data-autosave]');
        
        forms.forEach(form => {
            const formId = form.id || 'default-form';
            
            // Load saved data
            this.loadFormData(form, formId);
            
            // Save on input
            form.addEventListener('input', () => {
                this.saveFormData(form, formId);
            });
            
            // Clear on submit
            form.addEventListener('submit', () => {
                localStorage.removeItem(`form-${formId}`);
            });
        });
    }

    saveFormData(form, formId) {
        const formData = new FormData(form);
        const data = Object.fromEntries(formData);
        localStorage.setItem(`form-${formId}`, JSON.stringify(data));
    }

    loadFormData(form, formId) {
        const savedData = localStorage.getItem(`form-${formId}`);
        if (savedData) {
            const data = JSON.parse(savedData);
            Object.keys(data).forEach(key => {
                const field = form.querySelector(`[name="${key}"]`);
                if (field) {
                    field.value = data[key];
                }
            });
        }
    }
}

// Performance optimization
class PerformanceOptimizer {
    constructor() {
        this.setupImageLazyLoading();
        this.setupCodeSplitting();
        this.setupPrefetching();
    }

    setupImageLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        imageObserver.unobserve(img);
                    }
                });
            });

            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }

    setupCodeSplitting() {
        // Dynamic imports for heavy components (commented out until components exist)
        const loadHeavyComponents = () => {
            // TODO: Uncomment when components are created
            // import('./components/chart').then(module => {
            //     module.initCharts();
            // });
            
            // import('./components/map').then(module => {
            //     module.initMaps();
            // });
            
            console.log('Heavy components loading requested');
        };

        // Load only when needed
        document.addEventListener('click', (e) => {
            if (e.target.matches('[data-heavy-component]')) {
                loadHeavyComponents();
            }
        });
    }

    setupPrefetching() {
        // Prefetch next page resources on hover
        document.addEventListener('mouseover', (e) => {
            if (e.target.matches('a[href]')) {
                const link = e.target;
                if (!link.dataset.prefetched) {
                    const prefetchLink = document.createElement('link');
                    prefetchLink.rel = 'prefetch';
                    prefetchLink.href = link.href;
                    document.head.appendChild(prefetchLink);
                    link.dataset.prefetched = 'true';
                }
            }
        });
    }
}

// Initialize everything when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new ModernScrollUtilities();
    new MobileApp();
    new FormHandler();
    new PerformanceOptimizer();
    
    // Add mobile-specific styles
    document.body.classList.add('mobile-optimized');
    
    // Initialize any existing functions
    if (typeof window.initializeAnimations === 'function') {
        window.initializeAnimations();
    }
});
