@extends('layouts.public')

@section('title', 'Peta Desa - Portal Desa Digital')

@section('content')
<div class="container-fluid px-0">
    <!-- Map Header -->
    <div class="bg-primary text-white py-3">
        <div class="container">
            <div class="row align-items-center">
                <div class="col">
                    <h4 class="mb-0 fw-bold">
                        <i class="fas fa-map-marked-alt me-2"></i>Peta Desa Interaktif
                    </h4>
                    <small class="text-light">Lokasi fasilitas umum dan UMKM</small>
                </div>
                <div class="col-auto">
                    <button class="btn btn-outline-light btn-sm" data-bs-toggle="modal" data-bs-target="#mapLegendModal">
                        <i class="fas fa-info-circle me-1"></i>Legenda
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Map Container -->
    <div id="map" style="height: 70vh; min-height: 500px;"></div>

    <!-- Map Controls -->
    <div class="position-fixed" style="top: 120px; right: 20px; z-index: 1000;">
        <div class="bg-white rounded shadow p-2">
            <div class="btn-group-vertical" role="group">
                <button type="button" class="btn btn-outline-primary btn-sm" onclick="centerMap()" title="Pusatkan Peta">
                    <i class="fas fa-crosshairs"></i>
                </button>
                <button type="button" class="btn btn-outline-primary btn-sm" onclick="toggleFullscreen()" title="Layar Penuh">
                    <i class="fas fa-expand"></i>
                </button>
                <button type="button" class="btn btn-outline-primary btn-sm" onclick="getCurrentLocation()" title="Lokasi Saya">
                    <i class="fas fa-location-arrow"></i>
                </button>
            </div>
        </div>
    </div>

    <!-- Statistics Panel -->
    <div class="bg-light py-3">
        <div class="container">
            <div class="row g-3">
                <div class="col-6 col-md-3">
                    <div class="text-center">
                        <h5 class="fw-bold text-primary mb-1">{{ $stats['total_fasilitas'] ?? 0 }}</h5>
                        <small class="text-muted">Fasilitas Umum</small>
                    </div>
                </div>
                <div class="col-6 col-md-3">
                    <div class="text-center">
                        <h5 class="fw-bold text-success mb-1">{{ $stats['total_umkm'] ?? 0 }}</h5>
                        <small class="text-muted">UMKM</small>
                    </div>
                </div>
                <div class="col-6 col-md-3">
                    <div class="text-center">
                        <h5 class="fw-bold text-warning mb-1">{{ $stats['total_wisata'] ?? 0 }}</h5>
                        <small class="text-muted">Wisata</small>
                    </div>
                </div>
                <div class="col-6 col-md-3">
                    <div class="text-center">
                        <h5 class="fw-bold text-info mb-1">{{ count($umkm_locations ?? []) }}</h5>
                        <small class="text-muted">Lokasi Terdaftar</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Map Legend Modal -->
<div class="modal fade" id="mapLegendModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-map-signs me-2"></i>Legenda Peta
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="row g-3">
                    <div class="col-12">
                        <h6 class="fw-bold">Marker Lokasi:</h6>
                        <div class="d-flex align-items-center mb-2">
                            <div class="marker-sample bg-primary rounded-circle me-3" style="width: 20px; height: 20px;"></div>
                            <span>Fasilitas Umum</span>
                        </div>
                        <div class="d-flex align-items-center mb-2">
                            <div class="marker-sample bg-success rounded-circle me-3" style="width: 20px; height: 20px;"></div>
                            <span>UMKM</span>
                        </div>
                        <div class="d-flex align-items-center mb-3">
                            <div class="marker-sample bg-warning rounded-circle me-3" style="width: 20px; height: 20px;"></div>
                            <span>Wisata</span>
                        </div>
                    </div>
                    <div class="col-12">
                        <h6 class="fw-bold">Kontrol Peta:</h6>
                        <ul class="list-unstyled mb-0">
                            <li><i class="fas fa-crosshairs me-2 text-primary"></i>Pusatkan tampilan peta</li>
                            <li><i class="fas fa-expand me-2 text-primary"></i>Mode layar penuh</li>
                            <li><i class="fas fa-location-arrow me-2 text-primary"></i>Cari lokasi Anda</li>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Tutup</button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<style>
    .leaflet-popup-content {
        font-size: 14px;
        line-height: 1.4;
    }
    
    .leaflet-popup-content h6 {
        margin-bottom: 0.5rem;
        color: var(--primary-color);
    }
    
    .info-item {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.25rem;
        font-size: 13px;
    }
    
    .info-item .label {
        font-weight: 600;
        color: #666;
    }
    
    .info-item .value {
        color: #333;
    }
    
    .leaflet-control-zoom {
        display: none !important;
    }
    
    @media (max-width: 768px) {
        #map {
            height: 60vh !important;
        }
        
        .position-fixed {
            top: 100px !important;
            right: 10px !important;
        }
    }
</style>
@endpush

@push('scripts')
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script>
    let map;
    let markers = [];
    let userMarker = null;
    
    // Data UMKM dari controller
    const umkmData = @json($umkm_locations ?? []);
    
    // Inisialisasi peta
    function initMap() {
        // Koordinat pusat desa (default: Sukamakmur, Cianjur)
        const centerLat = umkmData.length > 0 ? 
            umkmData.reduce((sum, p) => sum + parseFloat(p.latitude), 0) / umkmData.length : 
            -6.7175; // Cianjur
        const centerLng = umkmData.length > 0 ? 
            umkmData.reduce((sum, p) => sum + parseFloat(p.longitude), 0) / umkmData.length : 
            107.1426; // Cianjur
        
        map = L.map('map', {
            zoomControl: false
        }).setView([centerLat, centerLng], 13);
        
        // Tambahkan tile layer
        L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
            attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
        }).addTo(map);
        
        // Tambahkan marker untuk UMKM dan fasilitas
        addLocationMarkers();
    }
    
    function addLocationMarkers() {
        // Tambahkan marker UMKM
        umkmData.forEach(umkm => {
            if (umkm.latitude && umkm.longitude) {
                const icon = L.divIcon({
                    className: 'custom-div-icon',
                    html: `<div style="background-color: #28a745; width: 20px; height: 20px; border-radius: 50%; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>`,
                    iconSize: [20, 20],
                    iconAnchor: [10, 10]
                });
                
                const marker = L.marker([parseFloat(umkm.latitude), parseFloat(umkm.longitude)], { icon })
                    .bindPopup(createUmkmPopupContent(umkm))
                    .addTo(map);
                    
                markers.push(marker);
            }
        });

        // Tambahkan marker fasilitas umum (data statis)
        addFasilitasMarkers();
    }

    function addFasilitasMarkers() {
        const fasilitasData = [
            {
                nama: 'Kantor Desa Sirnajaya',
                alamat: 'Jl. Raya Desa Sirnajaya',
                kategori: 'Pemerintahan',
                latitude: -6.7175,
                longitude: 107.1426
            },
            {
                nama: 'Puskesmas Sukamakmur',
                alamat: 'Kecamatan Sukamakmur',
                kategori: 'Kesehatan',
                latitude: -6.7185,
                longitude: 107.1436
            },
            {
                nama: 'SDN Sirnajaya 01',
                alamat: 'Desa Sirnajaya',
                kategori: 'Pendidikan',
                latitude: -6.7165,
                longitude: 107.1416
            }
        ];

        fasilitasData.forEach(fasilitas => {
            const icon = L.divIcon({
                className: 'custom-div-icon',
                html: `<div style="background-color: #2c5aa0; width: 20px; height: 20px; border-radius: 50%; border: 2px solid white; box-shadow: 0 2px 4px rgba(0,0,0,0.3);"></div>`,
                iconSize: [20, 20],
                iconAnchor: [10, 10]
            });
            
            const marker = L.marker([fasilitas.latitude, fasilitas.longitude], { icon })
                .bindPopup(createFasilitasPopupContent(fasilitas))
                .addTo(map);
                
            markers.push(marker);
        });
    }
    
    function createUmkmPopupContent(umkm) {
        return `
            <div style="min-width: 200px;">
                <h6 class="fw-bold mb-2">
                    <i class="fas fa-store me-1"></i>
                    ${umkm.nama_umkm}
                </h6>
                <div class="info-item">
                    <span class="label">Kategori:</span>
                    <span class="value">${umkm.kategori || 'Tidak Diketahui'}</span>
                </div>
                <div class="info-item">
                    <span class="label">Alamat:</span>
                    <span class="value">${umkm.alamat || '-'}</span>
                </div>
                <hr class="my-2">
                <small class="text-muted">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    ${parseFloat(umkm.latitude).toFixed(6)}, ${parseFloat(umkm.longitude).toFixed(6)}
                </small>
            </div>
        `;
    }

    function createFasilitasPopupContent(fasilitas) {
        return `
            <div style="min-width: 200px;">
                <h6 class="fw-bold mb-2">
                    <i class="fas fa-building me-1"></i>
                    ${fasilitas.nama}
                </h6>
                <div class="info-item">
                    <span class="label">Kategori:</span>
                    <span class="value">${fasilitas.kategori}</span>
                </div>
                <div class="info-item">
                    <span class="label">Alamat:</span>
                    <span class="value">${fasilitas.alamat}</span>
                </div>
                <hr class="my-2">
                <small class="text-muted">
                    <i class="fas fa-map-marker-alt me-1"></i>
                    ${fasilitas.latitude.toFixed(6)}, ${fasilitas.longitude.toFixed(6)}
                </small>
            </div>
        `;
    }
    
    function centerMap() {
        if (markers.length > 0) {
            const group = new L.featureGroup(markers);
            map.fitBounds(group.getBounds().pad(0.1));
        } else {
            // Default ke koordinat Desa Sirnajaya
            map.setView([-6.7175, 107.1426], 13);
        }
    }
    
    function toggleFullscreen() {
        if (!document.fullscreenElement) {
            document.getElementById('map').requestFullscreen().catch(err => {
                console.log(`Error attempting to enable fullscreen: ${err.message}`);
            });
        } else {
            document.exitFullscreen();
        }
    }
    
    function getCurrentLocation() {
        if (navigator.geolocation) {
            navigator.geolocation.getCurrentPosition(
                function(position) {
                    const lat = position.coords.latitude;
                    const lng = position.coords.longitude;
                    
                    // Hapus marker lokasi sebelumnya
                    if (userMarker) {
                        map.removeLayer(userMarker);
                    }
                    
                    // Tambahkan marker lokasi pengguna
                    userMarker = L.marker([lat, lng], {
                        icon: L.divIcon({
                            className: 'user-location-icon',
                            html: '<div style="background-color: #ff4757; width: 15px; height: 15px; border-radius: 50%; border: 3px solid white; box-shadow: 0 2px 6px rgba(0,0,0,0.4); animation: pulse 2s infinite;"></div>',
                            iconSize: [15, 15],
                            iconAnchor: [7.5, 7.5]
                        })
                    }).bindPopup('<b>Lokasi Anda</b>').addTo(map);
                    
                    // Pusatkan peta ke lokasi pengguna
                    map.setView([lat, lng], 16);
                },
                function(error) {
                    alert('Tidak dapat mengakses lokasi Anda: ' + error.message);
                },
                {
                    enableHighAccuracy: true,
                    timeout: 10000,
                    maximumAge: 60000
                }
            );
        } else {
            alert('Geolocation tidak didukung oleh browser ini.');
        }
    }
    
    // Inisialisasi peta saat halaman dimuat
    document.addEventListener('DOMContentLoaded', function() {
        initMap();
    });
</script>

<style>
    @keyframes pulse {
        0% {
            transform: scale(1);
            opacity: 1;
        }
        50% {
            transform: scale(1.2);
            opacity: 0.7;
        }
        100% {
            transform: scale(1);
            opacity: 1;
        }
    }
</style>
@endpush 
