# Implementasi CAPTCHA untuk Login Admin dan Portal Warga

## Overview
Implementasi sistem CAPTCHA telah berhasil ditambahkan ke kedua form login:
1. **Login Admin** - `/login`
2. **Login Portal Warga** - Layanan Mandiri

## Package yang Digunakan
- **gregwar/captcha** - Library PHP untuk generate CAPTCHA image

## File yang Dimodifikasi

### 1. Service Layer
- `app/Services/CaptchaService.php` - Service untuk generate dan validasi CAPTCHA

### 2. Controllers
- `app/Http/Controllers/AuthController.php` - Tambah validasi CAPTCHA untuk login admin
- `app/Http/Controllers/Api/PortalWargaController.php` - Tambah validasi CAPTCHA untuk login portal

### 3. Views
- `resources/views/auth/login.blade.php` - Tambah field CAPTCHA dan JavaScript
- `resources/views/frontend/layanan-mandiri.php` - Tambah field CAPTCHA dan JavaScript

### 4. Routes
- `routes/web.php` - Tambah route `/captcha` dan `/captcha/portal`

## Fitur CAPTCHA

### Untuk Login Admin
- **URL CAPTCHA**: `/captcha`
- **Field**: Input text untuk memasukkan kode
- **Validasi**: Server-side validation di `AuthController::login()`
- **Refresh**: Tombol refresh dan klik gambar CAPTCHA
- **JavaScript**: `refreshCaptcha()` function

### Untuk Login Portal Warga
- **URL CAPTCHA**: `/captcha/portal`
- **Field**: Input text untuk memasukkan kode
- **Validasi**: Server-side validation di `PortalWargaController::login()`
- **Refresh**: Tombol refresh dan klik gambar CAPTCHA
- **JavaScript**: `refreshCaptchaPortal()` function

## Cara Kerja

### 1. Generate CAPTCHA
```php
// CaptchaService::generate()
$builder = new CaptchaBuilder();
$builder->build();
Session::put('captcha_phrase', $builder->getPhrase());
return $builder->inline(); // atau $builder->get() untuk image
```

### 2. Validasi CAPTCHA
```php
// CaptchaService::validate($input)
$phrase = Session::get('captcha_phrase');
$isValid = strtolower($input) === strtolower($phrase);
Session::forget('captcha_phrase'); // Clear setelah validasi
```

### 3. Route Endpoints
```php
// routes/web.php
Route::get('/captcha', function() {
    return \App\Services\CaptchaService::getCaptchaImage();
})->name('captcha');

Route::get('/captcha/portal', function() {
    return \App\Services\CaptchaService::getCaptchaImage();
})->name('captcha.portal');
```

## Implementasi Frontend

### Login Admin (Bootstrap)
```html
<!-- CAPTCHA Field -->
<div class="row">
    <div class="col-7">
        <input type="text" name="captcha" required>
    </div>
    <div class="col-5">
        <img src="{{ route('captcha') }}" onclick="refreshCaptcha()">
        <button onclick="refreshCaptcha()">Refresh</button>
    </div>
</div>
```

### Login Portal (TailwindCSS)
```html
<!-- CAPTCHA Field -->
<div class="flex space-x-2">
    <div class="flex-1">
        <input type="text" id="captcha" required>
    </div>
    <div class="w-20 relative">
        <img src="/captcha/portal" onclick="refreshCaptchaPortal()">
        <button onclick="refreshCaptchaPortal()">Refresh</button>
    </div>
</div>
```

## Validasi Server-Side

### AuthController (Admin Login)
```php
public function login(Request $request)
{
    $credentials = $request->validate([
        'email' => 'required|email',
        'password' => 'required|string',
        'captcha' => 'required|string'
    ]);

    // Validate CAPTCHA first
    if (!CaptchaService::validate($request->captcha)) {
        return back()->withErrors([
            'captcha' => 'Kode keamanan (CAPTCHA) tidak valid.',
        ])->onlyInput('email');
    }
    
    // Continue with authentication...
}
```

### PortalWargaController (Portal Login)
```php
public function login(Request $request)
{
    $validator = Validator::make($request->all(), [
        'nik' => 'required|string|size:16',
        'password' => 'required|string',
        'captcha' => 'required|string'
    ]);

    // Validate CAPTCHA first
    if (!CaptchaService::validate($request->captcha)) {
        return response()->json([
            'success' => false,
            'message' => 'Kode keamanan (CAPTCHA) tidak valid'
        ], 422);
    }
    
    // Continue with authentication...
}
```

## JavaScript Functions

### Refresh CAPTCHA Admin
```javascript
function refreshCaptcha() {
    const captchaImg = document.getElementById('captcha-image');
    const captchaInput = document.getElementById('captcha');
    
    captchaImg.src = '{{ route("captcha") }}?' + Math.random();
    captchaInput.value = '';
    captchaInput.focus();
}
```

### Refresh CAPTCHA Portal
```javascript
function refreshCaptchaPortal() {
    const captchaImg = document.getElementById('captcha-image-portal');
    const captchaInput = document.getElementById('captcha');
    
    captchaImg.src = '/captcha/portal?' + Math.random();
    captchaInput.value = '';
    captchaInput.focus();
}
```

## Security Features
1. **Session-based**: CAPTCHA phrase disimpan di session
2. **Case-insensitive**: Validasi tidak case sensitive
3. **One-time use**: Session CAPTCHA dihapus setelah validasi
4. **Cache prevention**: Timestamp random untuk prevent caching
5. **Server-side validation**: Validasi dilakukan di server

## Testing
1. Akses `/login` untuk test CAPTCHA admin
2. Akses layanan mandiri untuk test CAPTCHA portal
3. Test refresh functionality
4. Test validation dengan kode salah/benar

## Notes
- CAPTCHA wajib diisi untuk kedua form login
- Gambar CAPTCHA dapat di-refresh dengan klik atau tombol
- Validasi dilakukan sebelum proses authentication
- Error message muncul jika CAPTCHA salah
- Compatible dengan Laravel 11 dan PHP 8.1+
