<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\File;

class FixStoragePermissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'storage:fix
                            {--check : Only check directories without creating them}
                            {--force : Force creation of directories}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Fix storage permissions and create missing directories';

    /**
     * Required storage directories
     */
    private $directories = [
        'penduduk',
        'perangkat/foto',
        'perangkat/sk',
        'perangkat/ttd',
        'berita',
        'pengaduan',
        'produk',
        'produk/galeri',
        'wisata',
        'surat',
        'berkas-surat',
        'agenda/banner',
        'agenda/dokumentasi',
        'apbdes/bukti',
        'arsip-surat-keluar',
        'organisasi',
        'templates',
        'bukti_bayar',
        'qrcodes',
        'review',
        'checkin',
        'settings'
    ];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔧 Fixing storage permissions and directories...');
        
        // Check if storage:link exists
        $this->checkStorageLink();
        
        // Check/Create directories
        if ($this->option('check')) {
            $this->checkDirectories();
        } else {
            $this->createDirectories();
        }
        
        $this->info('✅ Storage fix completed!');
        
        return 0;
    }

    /**
     * Check storage link
     */
    private function checkStorageLink()
    {
        $linkPath = public_path('storage');
        $targetPath = storage_path('app/public');
        
        if (!file_exists($linkPath)) {
            $this->warn('❌ Storage link not found');
            if ($this->confirm('Create storage link?', true)) {
                $this->call('storage:link');
            }
        } else {
            $this->info('✅ Storage link exists');
        }
    }

    /**
     * Check directories without creating them
     */
    private function checkDirectories()
    {
        $this->info('📁 Checking storage directories...');
        
        $missing = [];
        $existing = [];
        
        foreach ($this->directories as $directory) {
            $fullPath = Storage::disk('public')->path($directory);
            
            if (file_exists($fullPath)) {
                $existing[] = $directory;
                $this->line("✅ {$directory}");
            } else {
                $missing[] = $directory;
                $this->line("❌ {$directory}");
            }
        }
        
        $this->newLine();
        $this->info("📊 Summary:");
        $this->info("   Existing: " . count($existing));
        $this->info("   Missing: " . count($missing));
        
        if (!empty($missing)) {
            $this->newLine();
            $this->warn("Missing directories:");
            foreach ($missing as $dir) {
                $this->line("   - {$dir}");
            }
        }
    }

    /**
     * Create missing directories
     */
    private function createDirectories()
    {
        $this->info('📁 Creating storage directories...');
        
        $created = 0;
        $existing = 0;
        $failed = 0;
        
        foreach ($this->directories as $directory) {
            $fullPath = Storage::disk('public')->path($directory);
            
            if (file_exists($fullPath)) {
                $existing++;
                $this->line("✅ {$directory} (exists)");
            } else {
                try {
                    if (mkdir($fullPath, 0755, true)) {
                        $created++;
                        $this->line("🆕 {$directory} (created)");
                        
                        // Create .gitkeep file
                        $gitkeepPath = $fullPath . '/.gitkeep';
                        file_put_contents($gitkeepPath, '');
                        
                    } else {
                        $failed++;
                        $this->error("❌ {$directory} (failed to create)");
                    }
                } catch (\Exception $e) {
                    $failed++;
                    $this->error("❌ {$directory} (error: {$e->getMessage()})");
                }
            }
        }
        
        $this->newLine();
        $this->info("📊 Summary:");
        $this->info("   Created: {$created}");
        $this->info("   Existing: {$existing}");
        if ($failed > 0) {
            $this->warn("   Failed: {$failed}");
        }
    }

    /**
     * Fix file permissions (Linux/Mac only)
     */
    private function fixPermissions()
    {
        if (PHP_OS_FAMILY === 'Windows') {
            $this->info('🪟 Windows detected - skipping permission fix');
            return;
        }
        
        $this->info('🔐 Fixing file permissions...');
        
        $storagePath = storage_path('app/public');
        
        try {
            // Set directory permissions
            exec("find {$storagePath} -type d -exec chmod 755 {} \;");
            
            // Set file permissions
            exec("find {$storagePath} -type f -exec chmod 644 {} \;");
            
            $this->info('✅ Permissions fixed');
        } catch (\Exception $e) {
            $this->error('❌ Failed to fix permissions: ' . $e->getMessage());
        }
    }

    /**
     * Test file upload
     */
    private function testUpload()
    {
        $this->info('🧪 Testing file upload...');
        
        try {
            $testFile = storage_path('app/public/test.txt');
            file_put_contents($testFile, 'Test file for upload functionality');
            
            if (file_exists($testFile)) {
                $this->info('✅ File creation test passed');
                unlink($testFile);
            } else {
                $this->error('❌ File creation test failed');
            }
        } catch (\Exception $e) {
            $this->error('❌ Upload test failed: ' . $e->getMessage());
        }
    }
}
