<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\WilayahAdministratif;

class OptimizeWilayahPerformance extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'wilayah:optimize {--force : Force optimization without confirmation}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Optimize wilayah administratif performance by adding indexes and updating statistics';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Starting Wilayah Administratif Performance Optimization...');
        
        if (!$this->option('force') && !$this->confirm('This will optimize the wilayah_administratif table. Continue?')) {
            $this->info('Operation cancelled.');
            return 0;
        }

        try {
            // 1. Add database indexes if they don't exist
            $this->addDatabaseIndexes();
            
            // 2. Update all statistics
            $this->updateAllStatistics();
            
            // 3. Clear and warm up cache
            $this->optimizeCache();
            
            // 4. Display performance tips
            $this->displayOptimizationTips();
            
            $this->info('✅ Wilayah Administratif optimization completed successfully!');
            
        } catch (\Exception $e) {
            $this->error('❌ Optimization failed: ' . $e->getMessage());
            return 1;
        }

        return 0;
    }

    private function addDatabaseIndexes()
    {
        $this->info('📊 Adding database indexes...');
        
        $tableName = 'wilayah_administratif';
        
        // Check if indexes already exist
        $indexes = [
            'idx_wilayah_jenis' => 'jenis_wilayah',
            'idx_wilayah_parent' => 'parent_id',
            'idx_wilayah_status' => 'status',
            'idx_wilayah_urutan' => 'urutan_tampil',
        ];

        foreach ($indexes as $indexName => $column) {
            try {
                // Check if index exists
                $indexExists = DB::select("SHOW INDEX FROM {$tableName} WHERE Key_name = ?", [$indexName]);
                
                if (empty($indexExists)) {
                    DB::statement("ALTER TABLE {$tableName} ADD INDEX {$indexName} ({$column})");
                    $this->line("  ✓ Added index: {$indexName} on {$column}");
                } else {
                    $this->line("  → Index {$indexName} already exists");
                }
            } catch (\Exception $e) {
                $this->warn("  ⚠ Could not add index {$indexName}: " . $e->getMessage());
            }
        }

        // Composite indexes
        $compositeIndexes = [
            'idx_wilayah_jenis_status' => ['jenis_wilayah', 'status'],
            'idx_wilayah_parent_jenis' => ['parent_id', 'jenis_wilayah'],
            'idx_wilayah_jenis_kode' => ['jenis_wilayah', 'kode_wilayah'],
        ];

        foreach ($compositeIndexes as $indexName => $columns) {
            try {
                $indexExists = DB::select("SHOW INDEX FROM {$tableName} WHERE Key_name = ?", [$indexName]);
                
                if (empty($indexExists)) {
                    $columnList = implode(', ', $columns);
                    DB::statement("ALTER TABLE {$tableName} ADD INDEX {$indexName} ({$columnList})");
                    $this->line("  ✓ Added composite index: {$indexName} on ({$columnList})");
                } else {
                    $this->line("  → Composite index {$indexName} already exists");
                }
            } catch (\Exception $e) {
                $this->warn("  ⚠ Could not add composite index {$indexName}: " . $e->getMessage());
            }
        }
    }

    private function updateAllStatistics()
    {
        $this->info('📈 Updating all wilayah statistics from real penduduk data...');
        
        $bar = $this->output->createProgressBar(3);
        $bar->start();

        $totalUpdated = 0;

        // Update RT statistics first (most granular level)
        $rtList = WilayahAdministratif::where('jenis_wilayah', 'rt')->where('status', 'aktif')->get();
        foreach ($rtList as $rt) {
            $stats = $rt->updateStatistikPenduduk();
            $totalUpdated++;
        }
        $bar->advance();
        $this->line("  ✓ Updated {$rtList->count()} RT statistics from real data");

        // Update RW statistics
        $rwList = WilayahAdministratif::where('jenis_wilayah', 'rw')->where('status', 'aktif')->get();
        foreach ($rwList as $rw) {
            $stats = $rw->updateStatistikPenduduk();
            $totalUpdated++;
        }
        $bar->advance();
        $this->line("  ✓ Updated {$rwList->count()} RW statistics from real data");

        // Update Dusun statistics
        $dusunList = WilayahAdministratif::where('jenis_wilayah', 'dusun')->where('status', 'aktif')->get();
        foreach ($dusunList as $dusun) {
            $stats = $dusun->updateStatistikPenduduk();
            $totalUpdated++;
        }
        $bar->advance();
        $this->line("  ✓ Updated {$dusunList->count()} Dusun statistics from real data");

        $bar->finish();
        $this->newLine();
        
        // Display comparison
        $this->displayStatisticsComparison();
    }
    
    private function displayStatisticsComparison()
    {
        $this->info('📊 Comparison of Real vs Stored Statistics:');
        
        try {
            $stats = WilayahAdministratif::getStatistikRealTime();
            $realtime = $stats['realtime'];
            
            $this->table(
                ['Metric', 'Real Data', 'Stored Data', 'Difference'],
                [
                    ['Total KK', number_format($realtime['total_kk_real']), number_format($realtime['total_kk_wilayah']), ($realtime['selisih_kk'] > 0 ? '+' : '') . number_format($realtime['selisih_kk'])],
                    ['Total Penduduk', number_format($realtime['total_penduduk_real']), number_format($realtime['total_penduduk_wilayah']), ($realtime['selisih_penduduk'] > 0 ? '+' : '') . number_format($realtime['selisih_penduduk'])]
                ]
            );
            
            if ($realtime['sinkron']) {
                $this->info('✅ Data is now synchronized!');
            } else {
                $this->warn('⚠️  Some differences still exist. Consider running sync again.');
            }
            
        } catch (\Exception $e) {
            $this->warn('Could not display comparison: ' . $e->getMessage());
        }
    }

    private function optimizeCache()
    {
        $this->info('🔄 Optimizing cache...');
        
        // Clear old cache
        \Illuminate\Support\Facades\Cache::forget('wilayah_statistik');
        $this->line('  ✓ Cleared old cache');
        
        // Warm up cache with fresh data
        WilayahAdministratif::getStatistikWilayah();
        $this->line('  ✓ Warmed up statistics cache');
        
        // Warm up tree structure cache
        WilayahAdministratif::getTreeStructure();
        $this->line('  ✓ Warmed up tree structure cache');
    }

    private function displayOptimizationTips()
    {
        $this->newLine();
        $this->info('💡 Performance Optimization Tips:');
        $this->line('');
        $this->line('1. 📄 Pagination: Data is now paginated (20 items per page)');
        $this->line('2. 🎯 Selective Queries: Only necessary columns are loaded');
        $this->line('3. 🚀 Caching: Statistics are cached for 1 hour');
        $this->line('4. 📊 Indexes: Database indexes added for faster queries');
        $this->line('5. 🔄 Loading Indicators: User-friendly loading states');
        $this->line('');
        $this->info('🔧 To maintain performance:');
        $this->line('• Run "php artisan wilayah:optimize" monthly');
        $this->line('• Monitor cache hit rates');
        $this->line('• Consider archiving old data');
        $this->line('• Regular database maintenance');
    }
}
