<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Log;

class SyncSharedHostingFiles extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'shared-hosting:sync-files {--force : Force overwrite existing files}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync files from storage/app/public to public/storage for shared hosting';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting shared hosting file sync...');
        
        $force = $this->option('force');
        $syncCount = 0;
        $skipCount = 0;
        $errorCount = 0;
        
        // Define sync mappings
        $syncMappings = [
            'wisata' => [
                'source' => storage_path('app/public/wisata'),
                'target' => public_path('storage/wisata')
            ],
            'penduduk' => [
                'source' => storage_path('app/public/penduduk'),
                'target' => public_path('storage/penduduk')
            ],
            'berita' => [
                'source' => storage_path('app/public/berita'),
                'target' => public_path('storage/berita')
            ],
            'umkm' => [
                'source' => storage_path('app/public/umkm'),
                'target' => public_path('storage/umkm')
            ]
        ];
        
        foreach ($syncMappings as $type => $mapping) {
            $this->info("Syncing {$type} files...");
            
            $sourceDir = $mapping['source'];
            $targetDir = $mapping['target'];
            
            // Create target directory if it doesn't exist
            if (!File::exists($targetDir)) {
                File::makeDirectory($targetDir, 0755, true);
                $this->info("Created directory: {$targetDir}");
            }
            
            if (!File::exists($sourceDir)) {
                $this->warn("Source directory does not exist: {$sourceDir}");
                continue;
            }
            
            // Get all files recursively
            try {
                $files = File::allFiles($sourceDir);
                
                foreach ($files as $file) {
                    $relativePath = str_replace($sourceDir . DIRECTORY_SEPARATOR, '', $file->getPathname());
                    $targetPath = $targetDir . DIRECTORY_SEPARATOR . $relativePath;
                    $targetDirPath = dirname($targetPath);
                    
                    // Create subdirectory if needed
                    if (!File::exists($targetDirPath)) {
                        File::makeDirectory($targetDirPath, 0755, true);
                    }
                    
                    // Check if we should copy the file
                    $shouldCopy = false;
                    
                    if (!File::exists($targetPath)) {
                        $shouldCopy = true;
                        $reason = 'new file';
                    } elseif ($force) {
                        $shouldCopy = true;
                        $reason = 'forced';
                    } elseif (File::lastModified($file->getPathname()) > File::lastModified($targetPath)) {
                        $shouldCopy = true;
                        $reason = 'newer file';
                    }
                    
                    if ($shouldCopy) {
                        try {
                            File::copy($file->getPathname(), $targetPath);
                            $this->line("✅ Copied ({$reason}): {$type}/{$relativePath}");
                            $syncCount++;
                            
                            Log::info('File synced for shared hosting', [
                                'type' => $type,
                                'file' => $relativePath,
                                'reason' => $reason
                            ]);
                        } catch (\Exception $e) {
                            $this->error("❌ Failed to copy: {$type}/{$relativePath} - {$e->getMessage()}");
                            $errorCount++;
                        }
                    } else {
                        $this->line("⏭️ Skipped (up-to-date): {$type}/{$relativePath}");
                        $skipCount++;
                    }
                }
                
            } catch (\Exception $e) {
                $this->error("Error scanning {$type} directory: {$e->getMessage()}");
                $errorCount++;
            }
        }
        
        // Summary
        $this->info('');
        $this->info('Sync Summary:');
        $this->info("✅ Files synced: {$syncCount}");
        $this->info("⏭️ Files skipped: {$skipCount}");
        $this->info("❌ Errors: {$errorCount}");
        
        if ($errorCount > 0) {
            $this->warn('Some files failed to sync. Check the logs for details.');
            return 1;
        }
        
        $this->info('🎉 Shared hosting file sync completed successfully!');
        return 0;
    }
}
