<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\File;

class SyncStorageForSharedHosting extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'storage:sync-shared-hosting';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Sync storage files to public directory for shared hosting compatibility';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Syncing storage files for shared hosting...');

        $sourcePath = storage_path('app/public');
        $targetPath = public_path('storage');

        // Ensure target directory exists
        if (!File::exists($targetPath)) {
            File::makeDirectory($targetPath, 0755, true);
        }

        // Copy all files and directories
        $this->copyDirectory($sourcePath, $targetPath);

        $this->info('Storage files synced successfully!');
        $this->info('Files are now accessible via /storage/ URL path.');
    }

    /**
     * Copy directory recursively
     */
    private function copyDirectory($source, $destination)
    {
        if (!File::exists($source)) {
            $this->error("Source directory does not exist: {$source}");
            return;
        }

        $files = File::allFiles($source);
        
        foreach ($files as $file) {
            $relativePath = str_replace($source . DIRECTORY_SEPARATOR, '', $file->getPathname());
            $targetFile = $destination . DIRECTORY_SEPARATOR . $relativePath;
            $targetDir = dirname($targetFile);

            // Create target directory if it doesn't exist
            if (!File::exists($targetDir)) {
                File::makeDirectory($targetDir, 0755, true);
            }

            // Copy file
            File::copy($file->getPathname(), $targetFile);
            $this->line("Copied: {$relativePath}");
        }

        $this->info("Copied " . count($files) . " files to public/storage/");
    }
}
