<?php

namespace App\Helpers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class SharedHostingHelper
{
    /**
     * Get file URL with fallback for shared hosting
     */
    public static function getFileUrl($path, $defaultPath = null)
    {
        // Get base URL for absolute URLs - force susukannet.id for this project
        $baseUrl = 'https://susukannet.id';
        
        // Handle null or empty path
        if (empty($path)) {
            return $defaultPath ? $baseUrl . '/' . $defaultPath : $baseUrl . '/images/default-tourism.png';
        }
        
        // Check for temporary file paths
        if (strpos($path, 'tmp') !== false || strpos($path, 'php') !== false) {
            Log::warning('Temporary file path detected: ' . $path);
            return $defaultPath ? $baseUrl . '/' . $defaultPath : $baseUrl . '/images/default-tourism.png';
        }
        
        // Clean and encode the path properly
        $cleanPath = self::cleanFilePath($path);
        
        // Build storage URL
        $url = $baseUrl . '/storage/' . $cleanPath;
        
        // Check if file exists in public storage (symlink working or manual sync)
        $publicPath = public_path('storage/' . $cleanPath);
        if (file_exists($publicPath)) {
            return $url;
        }
        
        // Check if file exists in storage/app/public
        $storagePath = storage_path('app/public/' . $cleanPath);
        if (file_exists($storagePath)) {
            // File exists but not in public, copy to public for shared hosting
            $publicDir = dirname($publicPath);
            if (!is_dir($publicDir)) {
                mkdir($publicDir, 0755, true);
            }
            
            if (copy($storagePath, $publicPath)) {
                Log::info('File copied from storage to public for shared hosting: ' . $cleanPath);
                return $url;
            } else {
                Log::error('Failed to copy file to public: ' . $cleanPath);
            }
        }
        
        // Try different possible locations on shared hosting
        $possiblePaths = [
            public_path($cleanPath), // Direct in public
            public_path('uploads/' . $cleanPath), // Common uploads folder
            public_path('files/' . $cleanPath), // Alternative files folder
        ];
        
        foreach ($possiblePaths as $possiblePath) {
            if (file_exists($possiblePath)) {
                $relativePath = str_replace(public_path() . '/', '', $possiblePath);
                return $baseUrl . '/' . $relativePath;
            }
        }
        
        // Fallback to Storage::url() for specific shared hosting setups
        try {
            $storageUrl = Storage::url($cleanPath);
            if ($storageUrl && $storageUrl !== asset('storage/' . $cleanPath)) {
                // Make sure URL is absolute
                if (strpos($storageUrl, 'http') !== 0) {
                    $storageUrl = $baseUrl . $storageUrl;
                }
                return $storageUrl;
            }
        } catch (\Exception $e) {
            Log::warning('Storage URL failed: ' . $e->getMessage());
        }
        
        // Return default if provided with absolute URL
        if ($defaultPath) {
            return $baseUrl . '/' . $defaultPath;
        }
        
        return $baseUrl . '/images/default-tourism.png';
    }
    
    /**
     * Check if storage symlink is working
     */
    public static function isStorageSymlinkWorking()
    {
        try {
            return file_exists(public_path('storage'));
        } catch (\Exception $e) {
            return false;
        }
    }
    
    /**
     * Get UMKM logo URL with fallback
     */
    public static function getUmkmLogoUrl($logoFilename)
    {
        if (!$logoFilename) {
            return 'https://susukannet.id/images/default-umkm-logo.png';
        }
        
        // Use the main getFileUrl method for consistency
        return self::getFileUrl($logoFilename, 'images/default-umkm-logo.png');
    }
    
    /**
     * Get UMKM gallery photo URL with fallback
     */
    public static function getUmkmGalleryUrl($photoFilename)
    {
        if (!$photoFilename) {
            return 'https://susukannet.id/images/default-umkm-logo.png';
        }
        
        // Use the main getFileUrl method for consistency
        return self::getFileUrl($photoFilename, 'images/default-umkm-logo.png');
    }
    
    /**
     * Get product photo URL with fallback
     */
    public static function getProductPhotoUrl($photoFilename)
    {
        if (!$photoFilename) {
            return asset('images/default-umkm-logo.png');
        }
        
        return self::getFileUrl('produk/' . $photoFilename, 'images/default-umkm-logo.png');
    }
    
    /**
     * Clean file path for URL encoding issues
     */
    private static function cleanFilePath($path)
    {
        // Decode URL encoded characters
        $path = urldecode($path);
        
        // Remove any double slashes
        $path = str_replace('//', '/', $path);
        
        // Remove leading slash if exists
        $path = ltrim($path, '/');
        
        // Handle special characters in filename
        $pathParts = explode('/', $path);
        $filename = end($pathParts);
        $directory = implode('/', array_slice($pathParts, 0, -1));
        
        // Clean filename - remove or replace problematic characters
        $cleanFilename = self::cleanFilename($filename);
        
        if ($directory) {
            return $directory . '/' . $cleanFilename;
        }
        
        return $cleanFilename;
    }
    
    /**
     * Clean filename for web compatibility
     */
    private static function cleanFilename($filename)
    {
        // Replace problematic characters
        $filename = str_replace(['%20', '%25', '%2B'], [' ', '%', '+'], $filename);
        
        // Remove or replace other problematic characters
        $filename = preg_replace('/[^\w\-_\.]/', '_', $filename);
        
        // Ensure it's not empty
        if (empty($filename)) {
            $filename = 'default_' . time() . '.png';
        }
        
        return $filename;
    }
}
