<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\PpobTransaction;
use App\Models\PpobSetting;
use App\Services\PpobService;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class PpobController extends Controller
{
    protected $ppobService;

    public function __construct(PpobService $ppobService)
    {
        $this->ppobService = $ppobService;
    }

    /**
     * Display PPOB management dashboard
     */
    public function index()
    {
        $stats = [
            'total_transactions' => PpobTransaction::count(),
            'pending_transactions' => PpobTransaction::where('status', 'pending')->count(),
            'success_transactions' => PpobTransaction::where('status', 'success')->count(),
            'failed_transactions' => PpobTransaction::where('status', 'failed')->count(),
            'total_revenue' => PpobTransaction::where('status', 'success')->sum('total_amount'),
            'admin_fee_revenue' => PpobTransaction::where('status', 'success')->sum('admin_fee'),
        ];

        // Recent transactions
        $recentTransactions = PpobTransaction::with('user')
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        // Monthly revenue chart data
        $monthlyRevenue = PpobTransaction::selectRaw('
                MONTH(created_at) as month,
                YEAR(created_at) as year,
                SUM(total_amount) as total,
                SUM(admin_fee) as admin_fee,
                COUNT(*) as count
            ')
            ->where('status', 'success')
            ->where('created_at', '>=', Carbon::now()->subMonths(12))
            ->groupBy('year', 'month')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get();

        // Product type statistics
        $productStats = PpobTransaction::selectRaw('
                product_type,
                COUNT(*) as count,
                SUM(total_amount) as total,
                SUM(admin_fee) as admin_fee
            ')
            ->where('status', 'success')
            ->groupBy('product_type')
            ->orderBy('count', 'desc')
            ->get();

        return view('admin.ppob.index', compact(
            'stats',
            'recentTransactions',
            'monthlyRevenue',
            'productStats'
        ));
    }

    /**
     * Display PPOB settings page
     */
    public function settings()
    {
        $settings = PpobSetting::orderBy('category')->orderBy('key')->get()->groupBy('category');
        
        // Default settings if none exist
        if ($settings->isEmpty()) {
            $this->createDefaultSettings();
            $settings = PpobSetting::orderBy('category')->orderBy('key')->get()->groupBy('category');
        }

        return view('admin.ppob.settings', compact('settings'));
    }

    /**
     * Update PPOB settings
     */
    public function updateSettings(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'settings' => 'required|array',
            'settings.*.key' => 'required|string',
            'settings.*.value' => 'required',
            'settings.*.type' => 'required|in:string,number,boolean,array,object',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            foreach ($request->settings as $settingData) {
                $value = $settingData['value'];
                
                // Convert value based on type
                switch ($settingData['type']) {
                    case 'number':
                        $value = (float) $value;
                        break;
                    case 'boolean':
                        $value = filter_var($value, FILTER_VALIDATE_BOOLEAN);
                        break;
                    case 'array':
                    case 'object':
                        if (is_string($value)) {
                            $value = json_decode($value, true);
                        }
                        break;
                }

                PpobSetting::updateOrCreate(
                    ['key' => $settingData['key']],
                    [
                        'value' => $value,
                        'type' => $settingData['type'],
                        'is_active' => true
                    ]
                );
            }

            DB::commit();

            return redirect()->back()->with('success', 'Pengaturan PPOB berhasil diperbarui');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Gagal memperbarui pengaturan: ' . $e->getMessage());
        }
    }

    /**
     * Display transaction details
     */
    public function showTransaction($id)
    {
        $transaction = PpobTransaction::with('user')->findOrFail($id);
        
        return view('admin.ppob.transaction-detail', compact('transaction'));
    }

    /**
     * Retry failed transaction
     */
    public function retryTransaction($id)
    {
        $transaction = PpobTransaction::findOrFail($id);
        
        if ($transaction->status !== 'failed') {
            return redirect()->back()->with('error', 'Hanya transaksi gagal yang dapat diulang');
        }

        try {
            // Check transaction status from payment gateway
            $result = $this->ppobService->checkTransactionStatus($transaction->order_id);
            
            if ($result['success']) {
                $transaction->update([
                    'status' => $result['data']['payment_status']['status'] ?? 'pending',
                    'payment_response' => json_encode($result['data'])
                ]);
                
                return redirect()->back()->with('success', 'Status transaksi berhasil diperbarui');
            }
            
            return redirect()->back()->with('error', $result['message']);

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal mengecek status transaksi: ' . $e->getMessage());
        }
    }

    /**
     * Export transactions to CSV
     */
    public function exportTransactions(Request $request)
    {
        $query = PpobTransaction::with('user');
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }
        
        if ($request->filled('product_type')) {
            $query->where('product_type', $request->product_type);
        }

        $transactions = $query->orderBy('created_at', 'desc')->get();

        $filename = 'ppob_transactions_' . date('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($transactions) {
            $file = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($file, [
                'Order ID',
                'User',
                'Product Type',
                'Provider',
                'Target Number',
                'Amount',
                'Admin Fee',
                'Total Amount',
                'Status',
                'Created At',
                'Updated At'
            ]);

            // CSV Data
            foreach ($transactions as $transaction) {
                fputcsv($file, [
                    $transaction->order_id,
                    $transaction->user->name ?? 'Guest',
                    $transaction->product_type,
                    $transaction->provider,
                    $transaction->target_number,
                    $transaction->amount,
                    $transaction->admin_fee,
                    $transaction->total_amount,
                    $transaction->status,
                    $transaction->created_at->format('Y-m-d H:i:s'),
                    $transaction->updated_at->format('Y-m-d H:i:s')
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Create default PPOB settings
     */
    private function createDefaultSettings()
    {
        $defaultSettings = [
            // General Settings
            [
                'key' => 'ppob_enabled',
                'value' => true,
                'type' => 'boolean',
                'description' => 'Enable/disable PPOB service',
                'category' => 'general'
            ],
            [
                'key' => 'ppob_maintenance_mode',
                'value' => false,
                'type' => 'boolean',
                'description' => 'Put PPOB in maintenance mode',
                'category' => 'general'
            ],
            [
                'key' => 'ppob_service_name',
                'value' => 'PPOB Desa Digital',
                'type' => 'string',
                'description' => 'PPOB service display name',
                'category' => 'general'
            ],

            // Fee Settings
            [
                'key' => 'default_admin_fee_percentage',
                'value' => 2.0,
                'type' => 'number',
                'description' => 'Default admin fee percentage (2%)',
                'category' => 'fees'
            ],
            [
                'key' => 'minimum_admin_fee',
                'value' => 1000,
                'type' => 'number',
                'description' => 'Minimum admin fee amount (Rp)',
                'category' => 'fees'
            ],
            [
                'key' => 'maximum_admin_fee',
                'value' => 50000,
                'type' => 'number',
                'description' => 'Maximum admin fee amount (Rp)',
                'category' => 'fees'
            ],

            // Product-specific fees
            [
                'key' => 'pulsa_admin_fee',
                'value' => 500,
                'type' => 'number',
                'description' => 'Admin fee for pulsa transactions (Rp)',
                'category' => 'fees'
            ],
            [
                'key' => 'pln_admin_fee',
                'value' => 2500,
                'type' => 'number',
                'description' => 'Admin fee for PLN transactions (Rp)',
                'category' => 'fees'
            ],
            [
                'key' => 'pbb_admin_fee',
                'value' => 3000,
                'type' => 'number',
                'description' => 'Admin fee for PBB transactions (Rp)',
                'category' => 'fees'
            ],

            // Limits
            [
                'key' => 'daily_transaction_limit',
                'value' => 5000000,
                'type' => 'number',
                'description' => 'Daily transaction limit per user (Rp)',
                'category' => 'limits'
            ],
            [
                'key' => 'single_transaction_limit',
                'value' => 2000000,
                'type' => 'number',
                'description' => 'Maximum single transaction amount (Rp)',
                'category' => 'limits'
            ],
            [
                'key' => 'minimum_transaction_amount',
                'value' => 10000,
                'type' => 'number',
                'description' => 'Minimum transaction amount (Rp)',
                'category' => 'limits'
            ],
            [
                'key' => 'max_retry_attempts',
                'value' => 3,
                'type' => 'number',
                'description' => 'Maximum retry attempts for failed transactions',
                'category' => 'limits'
            ],

            // Product Settings
            [
                'key' => 'enabled_products',
                'value' => ['pulsa', 'paket_data', 'pln', 'pbb', 'bpjs'],
                'type' => 'array',
                'description' => 'List of enabled PPOB products',
                'category' => 'products'
            ],
            [
                'key' => 'featured_products',
                'value' => ['pulsa', 'pln', 'pbb'],
                'type' => 'array',
                'description' => 'Featured products to show first',
                'category' => 'products'
            ],

            // Notification Settings
            [
                'key' => 'admin_notification_email',
                'value' => 'admin@desa.id',
                'type' => 'string',
                'description' => 'Email for admin notifications',
                'category' => 'notifications'
            ],
            [
                'key' => 'notify_admin_on_transaction',
                'value' => true,
                'type' => 'boolean',
                'description' => 'Send notification to admin on new transactions',
                'category' => 'notifications'
            ],
            [
                'key' => 'notify_user_on_success',
                'value' => true,
                'type' => 'boolean',
                'description' => 'Send notification to user on successful transaction',
                'category' => 'notifications'
            ],

            // Security Settings
            [
                'key' => 'require_user_verification',
                'value' => false,
                'type' => 'boolean',
                'description' => 'Require user verification before PPOB transactions',
                'category' => 'security'
            ],
            [
                'key' => 'transaction_timeout_minutes',
                'value' => 15,
                'type' => 'number',
                'description' => 'Transaction timeout in minutes',
                'category' => 'security'
            ],
            [
                'key' => 'enable_fraud_detection',
                'value' => true,
                'type' => 'boolean',
                'description' => 'Enable basic fraud detection',
                'category' => 'security'
            ]
        ];

        foreach ($defaultSettings as $setting) {
            PpobSetting::create($setting);
        }
    }
} 